<?php
/**
 * Helper functions for Gozer
 *
 * Contains all the helper functions to check various
 * conditions and exceptions for the force login feature.
 *
 * @package Gozer
 * @since   1.0.0
 */

defined( 'ABSPATH' ) || exit;

/*
 * ==========================================================================
 * LOGIN PAGE DETECTION
 * ==========================================================================
 */

/**
 * Check if current request is to the login page.
 *
 * @since 1.0.0
 * @param string $request_uri Current request URI.
 * @return bool True if login page, false otherwise.
 */
function gozer_is_login_page( $request_uri ) {

	if ( false !== strpos( $request_uri, 'wp-login.php' ) ) {
		return true;
	}

	if ( false !== strpos( $request_uri, 'wp-admin' ) ) {
		return true;
	}

	if ( false !== strpos( $request_uri, 'wp-register.php' ) ) {
		return true;
	}

	return false;
}

/*
 * ==========================================================================
 * SEO RELATED CHECKS
 * ==========================================================================
 */

/**
 * Check if current request is to a sitemap.
 *
 * @since 1.0.0
 * @param string $request_uri Current request URI.
 * @return bool True if sitemap request, false otherwise.
 */
function gozer_is_sitemap( $request_uri ) {

	if ( false !== strpos( $request_uri, 'sitemap' ) ) {
		return true;
	}

	if ( preg_match( '/wp-sitemap[^\/]*\.xml/i', $request_uri ) ) {
		return true;
	}

	if ( preg_match( '/sitemap[^\/]*\.xml/i', $request_uri ) ) {
		return true;
	}

	if ( preg_match( '/sitemap[^\/]*\.xsl/i', $request_uri ) ) {
		return true;
	}

	return false;
}

/**
 * Check if current request is to robots.txt.
 *
 * @since 1.0.0
 * @param string $request_uri Current request URI.
 * @return bool True if robots.txt request, false otherwise.
 */
function gozer_is_robots( $request_uri ) {

	$path = wp_parse_url( $request_uri, PHP_URL_PATH );
	return ( 'robots.txt' === basename( $path ) );
}

/**
 * Check if visitor is a search engine bot.
 *
 * @since 1.0.0
 * @return bool True if search bot, false otherwise.
 */
function gozer_is_search_bot() {

	$user_agent = isset( $_SERVER['HTTP_USER_AGENT'] )
		? sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) )
		: '';

	if ( empty( $user_agent ) ) {
		return false;
	}

	$search_bots = array(
		'Googlebot', 'AdsBot-Google', 'Mediapartners-Google', 'APIs-Google',
		'FeedFetcher-Google', 'Google-Read-Aloud', 'Chrome-Lighthouse',
		'Google-InspectionTool', 'Storebot-Google', 'Bingbot', 'msnbot',
		'BingPreview', 'adidxbot', 'Slurp', 'DuckDuckBot', 'YandexBot',
		'YandexImages', 'YandexAccessibilityBot', 'Baiduspider', 'Sogou',
		'Exabot', 'ia_archiver', 'facebot', 'FacebookExternalHit',
		'LinkedInBot', 'Twitterbot', 'Pinterest', 'Slackbot', 'WhatsApp',
		'TelegramBot', 'Applebot',
	);

	/**
	 * Filter the list of search bot user agents.
	 *
	 * @since 1.0.0
	 * @param array $search_bots Array of bot user agent strings.
	 */
	$search_bots = apply_filters( 'gozer_search_bots', $search_bots );

	foreach ( $search_bots as $bot ) {
		if ( false !== stripos( $user_agent, $bot ) ) {
			return true;
		}
	}

	return false;
}

/*
 * ==========================================================================
 * TECHNICAL CHECKS
 * ==========================================================================
 */

/**
 * Check if current request is HTTP HEAD method.
 *
 * @since 1.0.0
 * @return bool True if HEAD request, false otherwise.
 */
function gozer_is_head_request() {

	$method = isset( $_SERVER['REQUEST_METHOD'] )
		? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_METHOD'] ) )
		: '';

	return ( 'HEAD' === strtoupper( $method ) );
}

/**
 * Check if current request is for a static file.
 *
 * @since 1.0.0
 * @param string $request_uri Current request URI.
 * @return bool True if static file request, false otherwise.
 */
function gozer_is_static_file( $request_uri ) {

	$static_extensions = array(
		'css', 'js', 'jpg', 'jpeg', 'png', 'gif', 'webp', 'svg', 'ico',
		'bmp', 'avif', 'tiff', 'woff', 'woff2', 'ttf', 'eot', 'otf',
		'map', 'mp3', 'mp4', 'webm', 'ogg', 'wav', 'pdf',
	);

	/**
	 * Filter the list of static file extensions.
	 *
	 * @since 1.0.0
	 * @param array $static_extensions Array of file extensions.
	 */
	$static_extensions = apply_filters( 'gozer_static_extensions', $static_extensions );

	$path      = wp_parse_url( $request_uri, PHP_URL_PATH );
	$extension = strtolower( pathinfo( $path, PATHINFO_EXTENSION ) );

	return in_array( $extension, $static_extensions, true );
}

/*
 * ==========================================================================
 * CUSTOM EXCEPTION CHECKS
 * ==========================================================================
 */

/**
 * Check if current path is in allowed paths list.
 *
 * @since 1.0.0
 * @param string $request_uri   Current request URI.
 * @param string $allowed_paths Newline-separated list of allowed paths.
 * @return bool True if path is allowed, false otherwise.
 */
function gozer_is_allowed_path( $request_uri, $allowed_paths ) {

	if ( empty( $allowed_paths ) ) {
		return false;
	}

	$paths = preg_split( '/\r\n|\r|\n/', $allowed_paths );
	$paths = array_map( 'trim', $paths );
	$paths = array_filter( $paths );

	if ( empty( $paths ) ) {
		return false;
	}

	$current_path = wp_parse_url( $request_uri, PHP_URL_PATH );
	$current_path = trailingslashit( $current_path );

	foreach ( $paths as $allowed ) {
		$allowed = trim( $allowed );

		if ( empty( $allowed ) ) {
			continue;
		}

		if ( '/' === $allowed && '/' === untrailingslashit( $current_path ) ) {
			return true;
		}

		$allowed = trailingslashit( $allowed );

		if ( $current_path === $allowed ) {
			return true;
		}

		if ( 0 === strpos( $current_path, $allowed ) ) {
			return true;
		}
	}

	return false;
}

/**
 * Check if visitor IP is in allowed IPs list.
 *
 * Supports individual IPs, CIDR notation (192.168.1.0/24),
 * and wildcard patterns (192.168.*).
 *
 * @since 1.0.0
 * @param string $allowed_ips Newline-separated list of allowed IPs/ranges.
 * @return bool True if IP is allowed, false otherwise.
 */
function gozer_is_allowed_ip( $allowed_ips ) {

	if ( empty( $allowed_ips ) ) {
		return false;
	}

	$user_ip = gozer_get_user_ip();

	if ( empty( $user_ip ) ) {
		return false;
	}

	$ip_rules = preg_split( '/\r\n|\r|\n/', $allowed_ips );
	$ip_rules = array_map( 'trim', $ip_rules );
	$ip_rules = array_filter( $ip_rules );

	foreach ( $ip_rules as $rule ) {
		if ( gozer_ip_matches_rule( $user_ip, $rule ) ) {
			return true;
		}
	}

	return false;
}

/**
 * Check if an IP matches a rule (individual, CIDR, or wildcard).
 *
 * @since 1.0.0
 * @param string $ip   IP address to check.
 * @param string $rule Rule to match against.
 * @return bool True if IP matches the rule.
 */
function gozer_ip_matches_rule( $ip, $rule ) {

	$rule = trim( $rule );

	if ( empty( $rule ) ) {
		return false;
	}

	// CIDR notation (192.168.1.0/24).
	if ( false !== strpos( $rule, '/' ) ) {
		return gozer_ip_in_cidr( $ip, $rule );
	}

	// Wildcard pattern (192.168.*).
	if ( false !== strpos( $rule, '*' ) ) {
		return gozer_ip_matches_wildcard( $ip, $rule );
	}

	// Exact match.
	return ( $ip === $rule );
}

/**
 * Check if an IP is within a CIDR range.
 *
 * @since 1.0.0
 * @param string $ip   IP address to check.
 * @param string $cidr CIDR notation.
 * @return bool True if IP is within the CIDR range.
 */
function gozer_ip_in_cidr( $ip, $cidr ) {

	$parts = explode( '/', $cidr );

	if ( 2 !== count( $parts ) ) {
		return false;
	}

	list( $subnet, $mask ) = $parts;
	$mask = (int) $mask;

	if ( ! filter_var( $subnet, FILTER_VALIDATE_IP ) ) {
		return false;
	}

	// IPv4.
	if ( filter_var( $subnet, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4 ) ) {
		if ( $mask < 0 || $mask > 32 ) {
			return false;
		}

		$ip_long     = ip2long( $ip );
		$subnet_long = ip2long( $subnet );

		if ( false === $ip_long || false === $subnet_long ) {
			return false;
		}

		$mask_long = -1 << ( 32 - $mask );

		return ( ( $ip_long & $mask_long ) === ( $subnet_long & $mask_long ) );
	}

	// IPv6.
	if ( filter_var( $subnet, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6 ) ) {
		if ( $mask < 0 || $mask > 128 ) {
			return false;
		}

		$ip_bin     = inet_pton( $ip );
		$subnet_bin = inet_pton( $subnet );

		if ( false === $ip_bin || false === $subnet_bin ) {
			return false;
		}

		$mask_bin = str_repeat( 'f', intval( $mask / 4 ) );

		switch ( $mask % 4 ) {
			case 1:
				$mask_bin .= '8';
				break;
			case 2:
				$mask_bin .= 'c';
				break;
			case 3:
				$mask_bin .= 'e';
				break;
		}

		$mask_bin = str_pad( $mask_bin, 32, '0' );
		$mask_bin = pack( 'H*', $mask_bin );

		return ( ( $ip_bin & $mask_bin ) === ( $subnet_bin & $mask_bin ) );
	}

	return false;
}

/**
 * Check if an IP matches a wildcard pattern.
 *
 * @since 1.0.0
 * @param string $ip      IP address to check.
 * @param string $pattern Wildcard pattern (e.g., 192.168.*).
 * @return bool True if IP matches the pattern.
 */
function gozer_ip_matches_wildcard( $ip, $pattern ) {

	$regex = str_replace( '.', '\.', $pattern );
	$regex = str_replace( '*', '\d{1,3}', $regex );
	$regex = '/^' . $regex . '$/';

	return (bool) preg_match( $regex, $ip );
}

/**
 * Get the real IP address of the visitor.
 *
 * @since 1.0.0
 * @return string Visitor IP address or empty string.
 */
function gozer_get_user_ip() {

	$ip_headers = array(
		'HTTP_CF_CONNECTING_IP',
		'HTTP_X_REAL_IP',
		'HTTP_X_FORWARDED_FOR',
		'REMOTE_ADDR',
	);

	/**
	 * Filter the IP detection headers.
	 *
	 * @since 1.0.0
	 * @param array $ip_headers Array of server variable names.
	 */
	$ip_headers = apply_filters( 'gozer_ip_headers', $ip_headers );

	foreach ( $ip_headers as $header ) {
		if ( ! empty( $_SERVER[ $header ] ) ) {
			$ip = sanitize_text_field( wp_unslash( $_SERVER[ $header ] ) );

			if ( 'HTTP_X_FORWARDED_FOR' === $header && false !== strpos( $ip, ',' ) ) {
				$ips = explode( ',', $ip );
				$ip  = trim( $ips[0] );
			}

			if ( filter_var( $ip, FILTER_VALIDATE_IP ) ) {
				return $ip;
			}
		}
	}

	return '';
}

/**
 * Check if visitor's user agent is in allowed list.
 *
 * @since 1.0.0
 * @param string $allowed_agents Newline-separated list of user agent patterns.
 * @return bool True if user agent matches, false otherwise.
 */
function gozer_is_allowed_user_agent( $allowed_agents ) {

	if ( empty( $allowed_agents ) ) {
		return false;
	}

	$user_agent = isset( $_SERVER['HTTP_USER_AGENT'] )
		? sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) )
		: '';

	if ( empty( $user_agent ) ) {
		return false;
	}

	$agents = preg_split( '/\r\n|\r|\n/', $allowed_agents );
	$agents = array_map( 'trim', $agents );
	$agents = array_filter( $agents );

	foreach ( $agents as $pattern ) {
		if ( false !== stripos( $user_agent, $pattern ) ) {
			return true;
		}
	}

	return false;
}

/*
 * ==========================================================================
 * BYPASS TOKEN FUNCTIONS
 * ==========================================================================
 */

/**
 * Generate a new bypass token.
 *
 * @since 1.0.0
 * @param int    $hours Number of hours until token expires.
 * @param string $note  Optional note for the token.
 * @return array Token data.
 */
function gozer_generate_bypass_token( $hours = 24, $note = '' ) {

	$token   = wp_generate_password( 32, false );
	$expires = time() + ( absint( $hours ) * HOUR_IN_SECONDS );

	return array(
		'token'   => $token,
		'expires' => $expires,
		'created' => time(),
		'note'    => sanitize_text_field( $note ),
	);
}

/**
 * Validate a bypass token against stored tokens.
 *
 * @since 1.0.0
 * @param string $token  Token to validate.
 * @param array  $tokens Optional. Array of stored tokens. If not provided, will be fetched from options.
 * @return bool True if token is valid, false otherwise.
 */
function gozer_validate_bypass_token( $token, $tokens = null ) {

	if ( empty( $token ) ) {
		return false;
	}

	if ( null === $tokens ) {
		$tokens = get_option( 'gozer_bypass_tokens', array() );
	}

	if ( empty( $tokens ) || ! is_array( $tokens ) ) {
		return false;
	}

	foreach ( $tokens as $stored_token ) {
		if ( isset( $stored_token['token'] ) && hash_equals( $stored_token['token'], $token ) ) {
			if ( isset( $stored_token['expires'] ) && $stored_token['expires'] > time() ) {
				return true;
			}
		}
	}

	return false;
}

/**
 * Clean expired bypass tokens.
 *
 * @since 1.0.0
 * @return int Number of tokens removed.
 */
function gozer_clean_expired_tokens() {

	$tokens = get_option( 'gozer_bypass_tokens', array() );

	if ( empty( $tokens ) || ! is_array( $tokens ) ) {
		return 0;
	}

	$now            = time();
	$original_count = count( $tokens );

	$tokens = array_filter(
		$tokens,
		function ( $token ) use ( $now ) {
			return isset( $token['expires'] ) && $token['expires'] > $now;
		}
	);

	update_option( 'gozer_bypass_tokens', array_values( $tokens ) );

	return $original_count - count( $tokens );
}

/**
 * Check if current request has a valid bypass token.
 *
 * Optimized to avoid duplicate database queries by reusing tokens array.
 *
 * @since 1.0.0
 * @return bool True if valid bypass token found.
 */
function gozer_has_valid_bypass_token() {

	$token = '';

	// Get tokens once for reuse.
	$tokens = get_option( 'gozer_bypass_tokens', array() );

	// Check URL parameter.
	// phpcs:ignore WordPress.Security.NonceVerification.Recommended
	if ( isset( $_GET['gozer_bypass'] ) ) {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$token = sanitize_text_field( wp_unslash( $_GET['gozer_bypass'] ) );

		if ( gozer_validate_bypass_token( $token, $tokens ) ) {
			// Set cookie for subsequent requests.
			foreach ( $tokens as $stored_token ) {
				if ( isset( $stored_token['token'] ) && hash_equals( $stored_token['token'], $token ) ) {
					$cookie_expires = isset( $stored_token['expires'] ) ? $stored_token['expires'] : time() + DAY_IN_SECONDS;
					setcookie( 'gozer_bypass', $token, $cookie_expires, COOKIEPATH, COOKIE_DOMAIN, is_ssl(), true );
					break;
				}
			}
			return true;
		}
	}

	// Check cookie.
	if ( isset( $_COOKIE['gozer_bypass'] ) ) {
		$token = sanitize_text_field( wp_unslash( $_COOKIE['gozer_bypass'] ) );
		return gozer_validate_bypass_token( $token, $tokens );
	}

	return false;
}
