<?php
/**
 * Force Login functionality for Gozer
 *
 * Contains the main logic to force visitors to log in
 * and all exception handling based on plugin settings.
 *
 * @package Gozer
 * @since   1.0.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Main function to enforce login requirement.
 *
 * @since 1.0.0
 * @return void
 */
function gozer_force_login() {

	$settings = gozer_get_settings();

	// Check if plugin is enabled.
	if ( empty( $settings['enabled'] ) ) {
		return;
	}

	// Always allow logged-in users.
	if ( is_user_logged_in() ) {
		return;
	}

	// Always allow admin area.
	if ( is_admin() ) {
		return;
	}

	$request_uri = isset( $_SERVER['REQUEST_URI'] )
		? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) )
		: '';

	// Always allow login pages.
	if ( gozer_is_login_page( $request_uri ) ) {
		return;
	}

	// Check for valid bypass token.
	if ( gozer_has_valid_bypass_token() ) {
		return;
	}

	// System exceptions.
	if ( ! empty( $settings['allow_rest_api'] ) && defined( 'REST_REQUEST' ) && REST_REQUEST ) {
		return;
	}

	if ( ! empty( $settings['allow_cron'] ) && defined( 'DOING_CRON' ) && DOING_CRON ) {
		return;
	}

	if ( ! empty( $settings['allow_cli'] ) && defined( 'WP_CLI' ) && WP_CLI ) {
		return;
	}

	if ( ! empty( $settings['allow_ajax'] ) && defined( 'DOING_AJAX' ) && DOING_AJAX ) {
		return;
	}

	if ( ! empty( $settings['allow_xmlrpc'] ) && defined( 'XMLRPC_REQUEST' ) && XMLRPC_REQUEST ) {
		return;
	}

	// SEO exceptions.
	if ( ! empty( $settings['allow_sitemap'] ) && gozer_is_sitemap( $request_uri ) ) {
		return;
	}

	if ( ! empty( $settings['allow_robots'] ) && gozer_is_robots( $request_uri ) ) {
		return;
	}

	if ( ! empty( $settings['allow_feeds'] ) && is_feed() ) {
		return;
	}

	if ( ! empty( $settings['allow_seo_bots'] ) && gozer_is_search_bot() ) {
		return;
	}

	// Technical exceptions.
	if ( ! empty( $settings['allow_head'] ) && gozer_is_head_request() ) {
		return;
	}

	if ( ! empty( $settings['allow_static'] ) && gozer_is_static_file( $request_uri ) ) {
		return;
	}

	// Custom exceptions.
	if ( ! empty( $settings['allowed_paths'] ) && gozer_is_allowed_path( $request_uri, $settings['allowed_paths'] ) ) {
		return;
	}

	if ( ! empty( $settings['allowed_ips'] ) && gozer_is_allowed_ip( $settings['allowed_ips'] ) ) {
		return;
	}

	if ( ! empty( $settings['allowed_user_agents'] ) && gozer_is_allowed_user_agent( $settings['allowed_user_agents'] ) ) {
		return;
	}

	/**
	 * Action before blocking access.
	 *
	 * @since 1.0.0
	 * @param string $request_uri Current request URI.
	 * @param array  $settings    Plugin settings.
	 */
	do_action( 'gozer_before_block', $request_uri, $settings );

	gozer_block_access( $request_uri, $settings );
}
add_action( 'template_redirect', 'gozer_force_login' );

/**
 * Block access to the current page.
 *
 * @since 1.0.0
 * @param string $request_uri Current request URI.
 * @param array  $settings    Plugin settings.
 * @return void
 */
function gozer_block_access( $request_uri, $settings ) {

	$redirect_type = isset( $settings['redirect_type'] ) ? $settings['redirect_type'] : 'login';

	switch ( $redirect_type ) {

		case '403':
			status_header( 403 );
			nocache_headers();

			$template = locate_template( '403.php' );

			if ( $template ) {
				include $template;
			} else {
				wp_die(
					esc_html__( 'There is no access, only Gozer "The Gozerian". You must be logged in to view this content.', 'gozer' ),
					esc_html__( 'Access denied', 'gozer' ),
					array(
						'response'  => 403,
						'back_link' => false,
					)
				);
			}
			exit;

		case 'url':
			$custom_url = isset( $settings['redirect_url'] ) ? $settings['redirect_url'] : '';

			if ( ! empty( $custom_url ) && wp_http_validate_url( $custom_url ) ) {
				wp_safe_redirect( esc_url_raw( $custom_url ) );
				exit;
			}
			// Fall through to login if URL is invalid.

		case 'login':
		default:
			$redirect_to = home_url( $request_uri );

			/**
			 * Filter the redirect URL after login.
			 *
			 * @since 1.0.0
			 * @param string $redirect_to URL to redirect after login.
			 * @param string $request_uri Original request URI.
			 */
			$redirect_to = apply_filters( 'gozer_login_redirect', $redirect_to, $request_uri );

			wp_safe_redirect( wp_login_url( $redirect_to ) );
			exit;
	}
}

/**
 * Add login message for redirected users.
 *
 * @since 1.0.0
 * @param string $message Existing login message.
 * @return string Modified login message.
 */
function gozer_login_message( $message ) {

	// phpcs:ignore WordPress.Security.NonceVerification.Recommended
	if ( empty( $_GET['redirect_to'] ) ) {
		return $message;
	}

	$settings = gozer_get_settings();

	if ( empty( $settings['enabled'] ) ) {
		return $message;
	}

	$custom_message = sprintf(
		'<p class="message">%s</p>',
		esc_html__( 'You must log in to access this site.', 'gozer' )
	);

	/**
	 * Filter the login page message.
	 *
	 * @since 1.0.0
	 * @param string $custom_message The message HTML.
	 */
	$custom_message = apply_filters( 'gozer_login_message', $custom_message );

	return $custom_message . $message;
}
add_filter( 'login_message', 'gozer_login_message' );
