<?php
/**
 * Gozer Promotional Banner
 *
 * Promotional banner for Gozer plugin.
 *
 * @package Gozer
 * @since   1.0.1
 */

defined( 'ABSPATH' ) || exit;

/**
 * Gozer Promo Banner class
 */
class Gozer_Promo_Banner {

	/**
	 * Current plugin slug to exclude from recommendations.
	 *
	 * @var string
	 */
	private $current_plugin_slug;

	/**
	 * Text domain for translations.
	 *
	 * @var string
	 */
	private $textdomain;

	/**
	 * CSS class prefix.
	 *
	 * @var string
	 */
	private $css_prefix;

	/**
	 * Constructor.
	 *
	 * @param string $current_plugin_slug Current plugin slug.
	 * @param string $textdomain          Text domain for translations.
	 * @param string $css_prefix          CSS class prefix.
	 */
	public function __construct( $current_plugin_slug, $textdomain, $css_prefix ) {
		$this->current_plugin_slug = $current_plugin_slug;
		$this->textdomain          = $textdomain;
		$this->css_prefix          = $css_prefix;
	}

	/**
	 * Get plugins catalog.
	 *
	 * @return array
	 */
	private function get_plugins_catalog() {
		return array(
			'vigilante'                            => array(
				'icon'        => 'dashicons-shield',
				'title'       => __( 'Complete WordPress security', 'gozer' ),
				'description' => __( 'All-in-one security plugin: firewall, login protection, security headers, 2FA, file integrity monitoring, and activity logging.', 'gozer' ),
				'button'      => __( 'Install Vigilante', 'gozer' ),
			),
			'gozer'                                => array(
				'icon'        => 'dashicons-admin-network',
				'title'       => __( 'Restrict site access', 'gozer' ),
				'description' => __( 'Force visitors to log in before accessing your site with extensive exception controls for pages, posts, and user roles.', 'gozer' ),
				'button'      => __( 'Install Gozer', 'gozer' ),
			),
			'vigia'                                => array(
				'icon'        => 'dashicons-visibility',
				'title'       => __( 'Monitor AI crawler activity', 'gozer' ),
				'description' => __( 'Track which AI bots visit your site, analyze their behavior, and take control with blocking rules and robots.txt management.', 'gozer' ),
				'button'      => __( 'Install VigIA', 'gozer' ),
			),
			'ai-share-summarize'                   => array(
				'icon'        => 'dashicons-share',
				'title'       => __( 'Boost your AI presence', 'gozer' ),
				'description' => __( 'Add social sharing and AI summarize buttons. Help visitors share your content and let AIs learn from your site while getting backlinks.', 'gozer' ),
				'button'      => __( 'Install AI Share & Summarize', 'gozer' ),
			),
			'ai-content-signals'                   => array(
				'icon'        => 'dashicons-shield',
				'title'       => __( 'Control AI content usage', 'gozer' ),
				'description' => __( 'Cloudflare-endorsed plugin to define how AI systems can use your content: for training, search results, or both.', 'gozer' ),
				'button'      => __( 'Install AI Content Signals', 'gozer' ),
			),
			'wpo-tweaks'                           => array(
				'icon'        => 'dashicons-performance',
				'title'       => __( 'Speed up your WordPress', 'gozer' ),
				'description' => __( 'Comprehensive performance optimizations: critical CSS, lazy loading, cache rules, and 30+ tweaks with zero configuration.', 'gozer' ),
				'button'      => __( 'Install WPO Tweaks', 'gozer' ),
			),
			'no-gutenberg'                         => array(
				'icon'        => 'dashicons-edit-page',
				'title'       => __( 'Back to Classic Editor', 'gozer' ),
				'description' => __( 'Completely remove Gutenberg, FSE styles, and block widgets. Restore the classic editing experience with better performance.', 'gozer' ),
				'button'      => __( 'Install No Gutenberg', 'gozer' ),
			),
			'anticache'                            => array(
				'icon'        => 'dashicons-hammer',
				'title'       => __( 'Development toolkit', 'gozer' ),
				'description' => __( 'Bypass all caching during development. Auto-detects cache plugins, enables debug mode, and includes maintenance screen.', 'gozer' ),
				'button'      => __( 'Install Anti-Cache Kit', 'gozer' ),
			),
			'auto-capitalize-names-ayudawp'        => array(
				'icon'        => 'dashicons-editor-textcolor',
				'title'       => __( 'Fix customer names', 'gozer' ),
				'description' => __( 'Auto-capitalize names and addresses in WordPress and WooCommerce. Keep invoices and reports professionally formatted.', 'gozer' ),
				'button'      => __( 'Install Auto Capitalize', 'gozer' ),
			),
			'easy-actions-scheduler-cleaner-ayudawp' => array(
				'icon'        => 'dashicons-database-remove',
				'title'       => __( 'Clean Action Scheduler', 'gozer' ),
				'description' => __( 'Remove millions of completed, failed, and old actions from WooCommerce Action Scheduler. Reduce database size instantly.', 'gozer' ),
				'button'      => __( 'Install Scheduler Cleaner', 'gozer' ),
			),
			'native-sitemap-customizer'            => array(
				'icon'        => 'dashicons-networking',
				'title'       => __( 'Customize your sitemap', 'gozer' ),
				'description' => __( 'Control WordPress native sitemap: exclude post types, taxonomies, specific posts, and authors. No bloat, just options.', 'gozer' ),
				'button'      => __( 'Install Sitemap Customizer', 'gozer' ),
			),
			'post-visibility-control'              => array(
				'icon'        => 'dashicons-hidden',
				'title'       => __( 'Control post visibility', 'gozer' ),
				'description' => __( 'Hide posts from homepage, archives, feeds, or REST API while keeping them accessible via direct URL.', 'gozer' ),
				'button'      => __( 'Install Post Visibility', 'gozer' ),
			),
			'widget-visibility-control'            => array(
				'icon'        => 'dashicons-welcome-widgets-menus',
				'title'       => __( 'Smart widget display', 'gozer' ),
				'description' => __( 'Show or hide widgets based on pages, post types, categories, user roles, and more. Works with any theme.', 'gozer' ),
				'button'      => __( 'Install Widget Visibility', 'gozer' ),
			),
			'search-replace-text-blocks'           => array(
				'icon'        => 'dashicons-search',
				'title'       => __( 'Search & replace in blocks', 'gozer' ),
				'description' => __( 'Find and replace text across all your Gutenberg blocks. Bulk edit content without touching the database directly.', 'gozer' ),
				'button'      => __( 'Install Search Replace Blocks', 'gozer' ),
			),
			'seo-read-more-buttons-ayudawp'        => array(
				'icon'        => 'dashicons-admin-links',
				'title'       => __( 'Better read more links', 'gozer' ),
				'description' => __( 'Customize excerpt "read more" links with buttons, custom text, and nofollow option. Improve CTR and SEO.', 'gozer' ),
				'button'      => __( 'Install SEO Read More', 'gozer' ),
			),
			'show-only-lowest-prices-in-woocommerce-variable-products' => array(
				'icon'        => 'dashicons-tag',
				'title'       => __( 'Cleaner variable prices', 'gozer' ),
				'description' => __( 'Display only the lowest price for WooCommerce variable products instead of confusing price ranges.', 'gozer' ),
				'button'      => __( 'Install Lowest Price', 'gozer' ),
			),
			'multiple-sale-prices-scheduler'       => array(
				'icon'        => 'dashicons-calendar-alt',
				'title'       => __( 'Schedule sale prices', 'gozer' ),
				'description' => __( 'Set multiple future sale prices for WooCommerce products. Plan promotions in advance with start and end dates.', 'gozer' ),
				'button'      => __( 'Install Sale Scheduler', 'gozer' ),
			),
			'easy-store-management-ayudawp'        => array(
				'icon'        => 'dashicons-store',
				'title'       => __( 'Simplify store management', 'gozer' ),
				'description' => __( 'Clean up WordPress admin for Store Managers. Hide unnecessary menus, keep only orders, products, and customers, plus quick access shortcuts.', 'gozer' ),
				'button'      => __( 'Install Easy Store', 'gozer' ),
			),
			'lightbox-images-for-divi'             => array(
				'icon'        => 'dashicons-format-gallery',
				'title'       => __( 'Lightbox for Divi', 'gozer' ),
				'description' => __( 'Add native lightbox functionality to Divi theme images. No jQuery, fast loading, fully customizable.', 'gozer' ),
				'button'      => __( 'Install Divi Lightbox', 'gozer' ),
			),
		);
	}

	/**
	 * Get services catalog.
	 *
	 * @return array
	 */
	private function get_services_catalog() {
		return array(
			'maintenance' => array(
				'icon'        => 'dashicons-admin-tools',
				'title'       => __( 'Need help with your website?', 'gozer' ),
				'description' => __( 'Professional WordPress maintenance: security monitoring, regular backups, performance optimization, and priority support.', 'gozer' ),
				'button'      => __( 'Learn more', 'gozer' ),
				'url'         => 'https://mantenimiento.ayudawp.com',
			),
			'consultancy' => array(
				'icon'        => 'dashicons-businessman',
				'title'       => __( 'WordPress consultancy', 'gozer' ),
				'description' => __( 'One-on-one online sessions to solve your WordPress doubts, get expert advice, and make better decisions for your project.', 'gozer' ),
				'button'      => __( 'Book a session', 'gozer' ),
				'url'         => 'https://servicios.ayudawp.com/producto/consultoria-online-wordpress/',
			),
			'hacked'      => array(
				'icon'        => 'dashicons-sos',
				'title'       => __( 'Hacked website?', 'gozer' ),
				'description' => __( 'Fast recovery service for compromised WordPress sites. We clean malware, fix vulnerabilities, and restore your site security.', 'gozer' ),
				'button'      => __( 'Get help now', 'gozer' ),
				'url'         => 'https://servicios.ayudawp.com/producto/wordpress-hackeado/',
			),
			'development' => array(
				'icon'        => 'dashicons-editor-code',
				'title'       => __( 'Custom development', 'gozer' ),
				'description' => __( 'Need a custom plugin, theme modifications, or specific functionality? We build tailored WordPress solutions for your needs.', 'gozer' ),
				'button'      => __( 'Request a quote', 'gozer' ),
				'url'         => 'https://servicios.ayudawp.com/producto/desarrollo-wordpress/',
			),
		);
	}

	/**
	 * Get random plugins excluding current.
	 *
	 * @param int $count Number of plugins to return.
	 * @return array
	 */
	private function get_random_plugins( $count = 2 ) {
		$plugins = $this->get_plugins_catalog();

		// Remove current plugin.
		unset( $plugins[ $this->current_plugin_slug ] );

		// Get random keys.
		$random_keys = array_rand( $plugins, min( $count, count( $plugins ) ) );

		if ( ! is_array( $random_keys ) ) {
			$random_keys = array( $random_keys );
		}

		$result = array();
		foreach ( $random_keys as $key ) {
			$result[ $key ] = $plugins[ $key ];
		}

		return $result;
	}

	/**
	 * Get random service.
	 *
	 * @return array
	 */
	private function get_random_service() {
		$services   = $this->get_services_catalog();
		$random_key = array_rand( $services );

		return $services[ $random_key ];
	}

	/**
	 * Render the promotional banner.
	 */
	public function render() {
		$plugins = $this->get_random_plugins( 2 );
		$service = $this->get_random_service();
		$prefix  = $this->css_prefix;
		?>
		<!-- Promotional notice -->
		<div class="<?php echo esc_attr( $prefix ); ?>-promo-notice">
			<h4><?php esc_html_e( 'Starter kit for your site', 'gozer' ); ?></h4>
			<div class="<?php echo esc_attr( $prefix ); ?>-promo-columns">

				<?php foreach ( $plugins as $slug => $plugin ) : ?>
				<div class="<?php echo esc_attr( $prefix ); ?>-promo-column">
					<span class="dashicons <?php echo esc_attr( $plugin['icon'] ); ?>"></span>
					<h5><?php echo esc_html( $plugin['title'] ); ?></h5>
					<p><?php echo esc_html( $plugin['description'] ); ?></p>
					<a href="<?php echo esc_url( admin_url( 'plugin-install.php?tab=plugin-information&plugin=' . $slug . '&TB_iframe=true&width=772&height=618' ) ); ?>" class="button thickbox">
						<?php echo esc_html( $plugin['button'] ); ?>
					</a>
				</div>
				<?php endforeach; ?>

				<div class="<?php echo esc_attr( $prefix ); ?>-promo-column">
					<span class="dashicons <?php echo esc_attr( $service['icon'] ); ?>"></span>
					<h5><?php echo esc_html( $service['title'] ); ?></h5>
					<p><?php echo esc_html( $service['description'] ); ?></p>
					<a href="<?php echo esc_url( $service['url'] ); ?>" target="_blank" rel="noopener noreferrer" class="button button-primary">
						<?php echo esc_html( $service['button'] ); ?>
					</a>
				</div>

			</div>
		</div>
		<?php
	}
}
