<?php
/**
 * Admin Settings page for Gozer
 *
 * @package Gozer
 * @since   1.0.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Enqueue admin styles and scripts.
 *
 * @since 1.0.0
 * @param string $hook Current admin page hook.
 * @return void
 */
function gozer_admin_enqueue( $hook ) {

	if ( 'settings_page_gozer' !== $hook ) {
		return;
	}

	// Load thickbox for plugin install modals.
	add_thickbox();

	wp_enqueue_style(
		'gozer-admin',
		GOZER_PLUGIN_URL . 'assets/css/admin.css',
		array(),
		GOZER_VERSION
	);

	wp_enqueue_script(
		'gozer-admin',
		GOZER_PLUGIN_URL . 'assets/js/admin.js',
		array( 'jquery' ),
		GOZER_VERSION,
		true
	);

	wp_localize_script(
		'gozer-admin',
		'gozer',
		array(
			'ajaxUrl'       => admin_url( 'admin-ajax.php' ),
			'nonce'         => wp_create_nonce( 'gozer_admin' ),
			'confirmDelete' => __( 'Are you sure you want to delete this token?', 'gozer' ),
			'copied'        => __( 'Copied to clipboard!', 'gozer' ),
		)
	);
}
add_action( 'admin_enqueue_scripts', 'gozer_admin_enqueue' );

/**
 * Add settings page under Settings menu.
 *
 * @since 1.0.0
 * @return void
 */
function gozer_add_settings_page() {

	add_options_page(
		__( 'Gozer "The Gatekeeper"', 'gozer' ),
		__( 'Gozer', 'gozer' ),
		'manage_options',
		'gozer',
		'gozer_render_settings_page'
	);
}
add_action( 'admin_menu', 'gozer_add_settings_page' );

/**
 * Register plugin settings.
 *
 * @since 1.0.0
 * @return void
 */
function gozer_register_settings() {

	register_setting(
		'gozer_settings_group',
		'gozer_settings',
		array(
			'type'              => 'array',
			'sanitize_callback' => 'gozer_sanitize_settings',
			'default'           => gozer_get_default_settings(),
		)
	);
}
add_action( 'admin_init', 'gozer_register_settings' );

/**
 * Sanitize settings before saving.
 *
 * @since 1.0.0
 * @param array $input Raw input from settings form.
 * @return array Sanitized settings.
 */
function gozer_sanitize_settings( $input ) {

	$sanitized = array();

	$boolean_fields = array(
		'enabled', 'allow_rest_api', 'allow_cron', 'allow_cli', 'allow_ajax',
		'allow_xmlrpc', 'allow_sitemap', 'allow_robots', 'allow_feeds',
		'allow_seo_bots', 'allow_head', 'allow_static',
	);

	foreach ( $boolean_fields as $field ) {
		$sanitized[ $field ] = ! empty( $input[ $field ] );
	}

	$sanitized['allowed_paths']       = isset( $input['allowed_paths'] ) ? sanitize_textarea_field( $input['allowed_paths'] ) : '';
	$sanitized['allowed_ips']         = isset( $input['allowed_ips'] ) ? sanitize_textarea_field( $input['allowed_ips'] ) : '';
	$sanitized['allowed_user_agents'] = isset( $input['allowed_user_agents'] ) ? sanitize_textarea_field( $input['allowed_user_agents'] ) : '';

	$valid_redirect_types       = array( 'login', '403', 'url' );
	$sanitized['redirect_type'] = isset( $input['redirect_type'] ) && in_array( $input['redirect_type'], $valid_redirect_types, true )
		? $input['redirect_type']
		: 'login';

	$sanitized['redirect_url'] = isset( $input['redirect_url'] ) ? sanitize_url( $input['redirect_url'] ) : '';

	return $sanitized;
}

/**
 * Render the settings page HTML.
 *
 * @since 1.0.0
 * @return void
 */
function gozer_render_settings_page() {

	if ( ! current_user_can( 'manage_options' ) ) {
		return;
	}

	$settings = gozer_get_settings();
	$tokens   = get_option( 'gozer_bypass_tokens', array() );

	?>
	<div class="wrap gozer-wrap">
		<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>

		<?php settings_errors( 'gozer_settings' ); ?>

		<form method="post" action="options.php">
			<?php settings_fields( 'gozer_settings_group' ); ?>

			<!-- General Settings -->
			<div class="gozer-section">
				<h2><?php esc_html_e( 'General settings', 'gozer' ); ?></h2>
				<p class="description"><?php esc_html_e( 'Enable or disable the force login functionality globally.', 'gozer' ); ?></p>

				<table class="form-table" role="presentation">
					<tr>
						<th scope="row"><?php esc_html_e( 'Enable force login', 'gozer' ); ?></th>
						<td>
							<label>
								<input type="checkbox" name="gozer_settings[enabled]" value="1" <?php checked( $settings['enabled'] ); ?>>
								<?php esc_html_e( 'Require login to access the site', 'gozer' ); ?>
							</label>
							<p class="description">
								<?php esc_html_e( 'When enabled, visitors must log in to view any page on your site (except configured exceptions).', 'gozer' ); ?>
							</p>
							<p class="gozer-default">
								<?php
								printf(
									/* translators: %s: Default value */
									esc_html__( 'Default: %s. Protection is active immediately after plugin activation.', 'gozer' ),
									'<strong>' . esc_html__( 'Enabled', 'gozer' ) . '</strong>'
								);
								?>
							</p>
						</td>
					</tr>
				</table>
			</div>

			<!-- Two column layout -->
			<div class="gozer-grid">

				<!-- Left Column -->
				<div class="gozer-column">

					<!-- System Exceptions -->
					<div class="gozer-section">
						<h2><?php esc_html_e( 'System exceptions', 'gozer' ); ?></h2>
						<p class="description"><?php esc_html_e( 'These exceptions ensure WordPress core functionality works correctly.', 'gozer' ); ?></p>

						<?php
						$system_options = array(
							'allow_rest_api' => array(
								'label'       => __( 'REST API', 'gozer' ),
								'desc'        => __( 'Required for the block editor (Gutenberg), many plugins, and headless WordPress.', 'gozer' ),
								'default'     => __( 'Enabled', 'gozer' ),
								'default_why' => __( 'Disabling breaks the block editor.', 'gozer' ),
							),
							'allow_cron'     => array(
								'label'       => __( 'WP-Cron', 'gozer' ),
								'desc'        => __( 'Required for scheduled tasks like publishing posts and automated processes.', 'gozer' ),
								'default'     => __( 'Enabled', 'gozer' ),
								'default_why' => __( 'Disabling stops scheduled tasks.', 'gozer' ),
							),
							'allow_cli'      => array(
								'label'       => __( 'WP-CLI', 'gozer' ),
								'desc'        => __( 'Required for command-line administration and maintenance scripts.', 'gozer' ),
								'default'     => __( 'Enabled', 'gozer' ),
								'default_why' => __( 'Disabling breaks CLI operations.', 'gozer' ),
							),
							'allow_ajax'     => array(
								'label'       => __( 'AJAX', 'gozer' ),
								'desc'        => __( 'Required for contact forms and plugins using admin-ajax.php publicly.', 'gozer' ),
								'default'     => __( 'Enabled', 'gozer' ),
								'default_why' => __( 'Disabling may break forms and dynamic content.', 'gozer' ),
							),
							'allow_xmlrpc'   => array(
								'label'       => __( 'XML-RPC', 'gozer' ),
								'desc'        => __( 'Only enable if you use the WordPress mobile app or Jetpack.', 'gozer' ),
								'default'     => __( 'Disabled', 'gozer' ),
								'default_why' => __( 'This is a common attack vector. Keep disabled unless needed.', 'gozer' ),
								'warning'     => true,
							),
						);

						foreach ( $system_options as $key => $option ) :
							?>
							<label class="gozer-option">
								<input type="checkbox" name="gozer_settings[<?php echo esc_attr( $key ); ?>]" value="1" <?php checked( $settings[ $key ] ); ?>>
								<strong><?php echo esc_html( $option['label'] ); ?></strong>
								<span class="description"><?php echo esc_html( $option['desc'] ); ?></span>
								<span class="gozer-default<?php echo ! empty( $option['warning'] ) ? ' gozer-warning' : ''; ?>">
									<?php
									printf(
										/* translators: 1: Default value, 2: Reason */
										esc_html__( 'Default: %1$s. %2$s', 'gozer' ),
										'<strong>' . esc_html( $option['default'] ) . '</strong>',
										esc_html( $option['default_why'] )
									);
									?>
								</span>
							</label>
						<?php endforeach; ?>
					</div>

					<!-- Technical Exceptions -->
					<div class="gozer-section">
						<h2><?php esc_html_e( 'Technical exceptions', 'gozer' ); ?></h2>
						<p class="description"><?php esc_html_e( 'Additional technical exceptions for specific use cases.', 'gozer' ); ?></p>

						<label class="gozer-option">
							<input type="checkbox" name="gozer_settings[allow_head]" value="1" <?php checked( $settings['allow_head'] ); ?>>
							<strong><?php esc_html_e( 'HEAD requests', 'gozer' ); ?></strong>
							<span class="description"><?php esc_html_e( 'Used by uptime monitoring services (Pingdom, UptimeRobot, etc.).', 'gozer' ); ?></span>
							<span class="gozer-default">
								<?php
								printf(
									/* translators: 1: Default value, 2: Reason */
									esc_html__( 'Default: %1$s. %2$s', 'gozer' ),
									'<strong>' . esc_html__( 'Enabled', 'gozer' ) . '</strong>',
									esc_html__( 'Required for uptime monitoring.', 'gozer' )
								);
								?>
							</span>
						</label>

						<label class="gozer-option">
							<input type="checkbox" name="gozer_settings[allow_static]" value="1" <?php checked( $settings['allow_static'] ); ?>>
							<strong><?php esc_html_e( 'Static files', 'gozer' ); ?></strong>
							<span class="description"><?php esc_html_e( 'Allows public access to CSS, JavaScript, images, and fonts.', 'gozer' ); ?></span>
							<span class="gozer-default">
								<?php
								printf(
									/* translators: 1: Default value, 2: Reason */
									esc_html__( 'Default: %1$s. %2$s', 'gozer' ),
									'<strong>' . esc_html__( 'Disabled', 'gozer' ) . '</strong>',
									esc_html__( 'Theme handles its own assets. Enable only if login page has styling issues.', 'gozer' )
								);
								?>
							</span>
						</label>
					</div>

				</div>

				<!-- Right Column -->
				<div class="gozer-column">

					<!-- SEO Exceptions -->
					<div class="gozer-section">
						<h2><?php esc_html_e( 'SEO exceptions', 'gozer' ); ?></h2>
						<p class="description"><?php esc_html_e( 'These exceptions help search engines discover and index your site.', 'gozer' ); ?></p>

						<?php
						$seo_options = array(
							'allow_sitemap'  => array(
								'label'       => __( 'Sitemaps', 'gozer' ),
								'desc'        => __( 'Allows sitemap.xml, wp-sitemap.xml, and SEO plugin sitemaps.', 'gozer' ),
								'default'     => __( 'Enabled', 'gozer' ),
								'default_why' => __( 'Search engines need sitemap access for indexing.', 'gozer' ),
							),
							'allow_robots'   => array(
								'label'       => __( 'robots.txt', 'gozer' ),
								'desc'        => __( 'Crawlers check robots.txt to know which pages they can access.', 'gozer' ),
								'default'     => __( 'Enabled', 'gozer' ),
								'default_why' => __( 'Required for proper crawler behavior.', 'gozer' ),
							),
							'allow_feeds'    => array(
								'label'       => __( 'RSS feeds', 'gozer' ),
								'desc'        => __( 'Enable if you want RSS readers to access your content publicly.', 'gozer' ),
								'default'     => __( 'Disabled', 'gozer' ),
								'default_why' => __( 'Private sites usually do not need public feeds.', 'gozer' ),
							),
							'allow_seo_bots' => array(
								'label'       => __( 'Search engine bots', 'gozer' ),
								'desc'        => __( 'Allows Google, Bing, Yahoo, DuckDuckGo, and other major bots.', 'gozer' ),
								'default'     => __( 'Enabled', 'gozer' ),
								'default_why' => __( 'Allows search engines to index your public content.', 'gozer' ),
							),
						);

						foreach ( $seo_options as $key => $option ) :
							?>
							<label class="gozer-option">
								<input type="checkbox" name="gozer_settings[<?php echo esc_attr( $key ); ?>]" value="1" <?php checked( $settings[ $key ] ); ?>>
								<strong><?php echo esc_html( $option['label'] ); ?></strong>
								<span class="description"><?php echo esc_html( $option['desc'] ); ?></span>
								<span class="gozer-default">
									<?php
									printf(
										/* translators: 1: Default value, 2: Reason */
										esc_html__( 'Default: %1$s. %2$s', 'gozer' ),
										'<strong>' . esc_html( $option['default'] ) . '</strong>',
										esc_html( $option['default_why'] )
									);
									?>
								</span>
							</label>
						<?php endforeach; ?>
					</div>

					<!-- Redirect Behavior -->
					<div class="gozer-section">
						<h2><?php esc_html_e( 'Redirect behavior', 'gozer' ); ?></h2>
						<p class="description"><?php esc_html_e( 'What happens when an unauthorized visitor tries to access your site.', 'gozer' ); ?></p>

						<fieldset class="gozer-redirect-options">
							<label>
								<input type="radio" name="gozer_settings[redirect_type]" value="login" <?php checked( $settings['redirect_type'], 'login' ); ?>>
								<?php esc_html_e( 'Redirect to login page', 'gozer' ); ?>
								<span class="description"><?php esc_html_e( 'Visitors go to wp-login.php and return to the original page after login.', 'gozer' ); ?></span>
							</label>

							<label>
								<input type="radio" name="gozer_settings[redirect_type]" value="403" <?php checked( $settings['redirect_type'], '403' ); ?>>
								<?php esc_html_e( 'Show 403 Forbidden error', 'gozer' ); ?>
								<span class="description"><?php esc_html_e( 'Visitors see a 403 error page.', 'gozer' ); ?></span>
							</label>

							<label>
								<input type="radio" name="gozer_settings[redirect_type]" value="url" <?php checked( $settings['redirect_type'], 'url' ); ?>>
								<?php esc_html_e( 'Redirect to custom URL', 'gozer' ); ?>
								<span class="description"><?php esc_html_e( 'Visitors are redirected to a custom URL of your choice.', 'gozer' ); ?></span>
							</label>
						</fieldset>

						<div id="gozer-custom-url-row" style="<?php echo 'url' === $settings['redirect_type'] ? '' : 'display:none;'; ?>margin-top:15px;">
							<label for="gozer_redirect_url"><strong><?php esc_html_e( 'Custom redirect URL', 'gozer' ); ?></strong></label>
							<input type="url" id="gozer_redirect_url" name="gozer_settings[redirect_url]" value="<?php echo esc_url( $settings['redirect_url'] ); ?>" class="large-text" placeholder="https://example.com/access-restricted/">
						</div>

						<p class="gozer-default" style="margin-top:15px;">
							<?php
							printf(
								/* translators: %s: Default value */
								esc_html__( 'Default: %s. Most user-friendly option.', 'gozer' ),
								'<strong>' . esc_html__( 'Login page', 'gozer' ) . '</strong>'
							);
							?>
						</p>
					</div>

				</div>

			</div>

			<!-- Custom Exceptions -->
			<div class="gozer-section">
				<h2><?php esc_html_e( 'Custom exceptions', 'gozer' ); ?></h2>
				<p class="description"><?php esc_html_e( 'Define specific pages, IPs, or user agents that should bypass the login requirement.', 'gozer' ); ?></p>

				<div class="gozer-custom-fields">
					<div class="gozer-field">
						<label for="gozer_allowed_paths"><strong><?php esc_html_e( 'Allowed paths', 'gozer' ); ?></strong></label>
						<textarea id="gozer_allowed_paths" name="gozer_settings[allowed_paths]" rows="4" class="large-text code" placeholder="/contact/&#10;/landing-page/&#10;/shop/"><?php echo esc_textarea( $settings['allowed_paths'] ); ?></textarea>
						<p class="description"><?php esc_html_e( 'One path per line. Paths are relative to site root. Use "/" for homepage. Subpages included.', 'gozer' ); ?></p>
					</div>

					<div class="gozer-field">
						<label for="gozer_allowed_ips"><strong><?php esc_html_e( 'Allowed IP addresses', 'gozer' ); ?></strong></label>
						<textarea id="gozer_allowed_ips" name="gozer_settings[allowed_ips]" rows="4" class="large-text code" placeholder="192.168.1.1&#10;10.0.0.0/8&#10;172.16.*"><?php echo esc_textarea( $settings['allowed_ips'] ); ?></textarea>
						<p class="description"><?php esc_html_e( 'One entry per line. Supports individual IPs, CIDR notation (192.168.1.0/24), and wildcards (192.168.*).', 'gozer' ); ?></p>
					</div>

					<div class="gozer-field">
						<label for="gozer_allowed_user_agents"><strong><?php esc_html_e( 'Allowed user agents', 'gozer' ); ?></strong></label>
						<textarea id="gozer_allowed_user_agents" name="gozer_settings[allowed_user_agents]" rows="4" class="large-text code" placeholder="UptimeRobot&#10;Pingdom&#10;StatusCake"><?php echo esc_textarea( $settings['allowed_user_agents'] ); ?></textarea>
						<p class="description"><?php esc_html_e( 'One user agent pattern per line. Partial matching (e.g., "UptimeRobot" matches any UA containing it).', 'gozer' ); ?></p>
					</div>
				</div>
			</div>

			<?php submit_button( __( 'Save settings', 'gozer' ) ); ?>

		</form>

		<!-- Bypass Tokens Section -->
		<div class="gozer-section">
			<h2><?php esc_html_e( 'Temporary bypass tokens', 'gozer' ); ?></h2>
			<p class="description"><?php esc_html_e( 'Generate temporary access links for visitors who need to view the site without logging in (e.g., clients reviewing a staging site).', 'gozer' ); ?></p>

			<div class="gozer-token-generator">
				<div class="gozer-token-form">
					<label for="gozer-token-hours"><?php esc_html_e( 'Valid for:', 'gozer' ); ?></label>
					<select id="gozer-token-hours">
						<option value="1">1 <?php esc_html_e( 'hour', 'gozer' ); ?></option>
						<option value="6">6 <?php esc_html_e( 'hours', 'gozer' ); ?></option>
						<option value="24" selected>24 <?php esc_html_e( 'hours', 'gozer' ); ?></option>
						<option value="48">48 <?php esc_html_e( 'hours', 'gozer' ); ?></option>
						<option value="168">7 <?php esc_html_e( 'days', 'gozer' ); ?></option>
					</select>

					<label for="gozer-token-note"><?php esc_html_e( 'Note (optional):', 'gozer' ); ?></label>
					<input type="text" id="gozer-token-note" placeholder="<?php esc_attr_e( 'e.g., For client review', 'gozer' ); ?>">

					<button type="button" id="gozer-generate-token" class="button button-secondary"><?php esc_html_e( 'Generate token', 'gozer' ); ?></button>
				</div>
			</div>

			<?php if ( ! empty( $tokens ) ) : ?>
				<table class="wp-list-table widefat fixed striped gozer-tokens-table">
					<thead>
						<tr>
							<th class="gozer-col-link"><?php esc_html_e( 'Access link', 'gozer' ); ?></th>
							<th class="gozer-col-note"><?php esc_html_e( 'Note', 'gozer' ); ?></th>
							<th class="gozer-col-expires"><?php esc_html_e( 'Expires', 'gozer' ); ?></th>
							<th class="gozer-col-actions"><?php esc_html_e( 'Actions', 'gozer' ); ?></th>
						</tr>
					</thead>
					<tbody>
						<?php foreach ( $tokens as $index => $token_data ) : ?>
							<?php
							if ( $token_data['expires'] < time() ) {
								continue;
							}
							$bypass_url = add_query_arg( 'gozer_bypass', $token_data['token'], home_url( '/' ) );
							?>
							<tr>
								<td class="gozer-col-link">
									<code class="gozer-token-url"><?php echo esc_url( $bypass_url ); ?></code>
								</td>
								<td class="gozer-col-note"><?php echo esc_html( $token_data['note'] ); ?></td>
								<td class="gozer-col-expires">
									<?php
									/* translators: %s: Human-readable time difference */
									printf( esc_html__( 'in %s', 'gozer' ), esc_html( human_time_diff( time(), $token_data['expires'] ) ) );
									?>
								</td>
								<td class="gozer-col-actions">
									<button type="button" class="button button-small gozer-copy-token" data-url="<?php echo esc_url( $bypass_url ); ?>"><?php esc_html_e( 'Copy', 'gozer' ); ?></button>
									<button type="button" class="button button-small button-link-delete gozer-delete-token" data-index="<?php echo esc_attr( $index ); ?>"><?php esc_html_e( 'Delete', 'gozer' ); ?></button>
								</td>
							</tr>
						<?php endforeach; ?>
					</tbody>
				</table>
			<?php else : ?>
				<p class="gozer-no-tokens"><?php esc_html_e( 'No active tokens. Generate one above to share temporary access.', 'gozer' ); ?></p>
			<?php endif; ?>
		</div>

		<?php
		// Promotional banner with random plugins and services.
		$promo_banner = new Gozer_Promo_Banner( 'gozer', 'gozer', 'gozer' );
		$promo_banner->render();
		?>

	</div>
	<?php
}

/**
 * Handle AJAX token generation.
 *
 * @since 1.0.0
 * @return void
 */
function gozer_ajax_generate_token() {

	check_ajax_referer( 'gozer_admin', 'nonce' );

	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( array( 'message' => __( 'Permission denied.', 'gozer' ) ) );
	}

	$hours = isset( $_POST['hours'] ) ? absint( $_POST['hours'] ) : 24;

	// Validate hours against allowed values.
	$allowed_hours = array( 1, 6, 24, 48, 168 );
	if ( ! in_array( $hours, $allowed_hours, true ) ) {
		$hours = 24;
	}

	$note = isset( $_POST['note'] ) ? sanitize_text_field( wp_unslash( $_POST['note'] ) ) : '';

	$token_data = gozer_generate_bypass_token( $hours, $note );

	$tokens   = get_option( 'gozer_bypass_tokens', array() );
	$tokens[] = $token_data;
	update_option( 'gozer_bypass_tokens', $tokens );

	$bypass_url = add_query_arg( 'gozer_bypass', $token_data['token'], home_url( '/' ) );

	wp_send_json_success(
		array(
			'url'     => $bypass_url,
			'expires' => human_time_diff( time(), $token_data['expires'] ),
			'note'    => $note,
			'index'   => count( $tokens ) - 1,
		)
	);
}
add_action( 'wp_ajax_gozer_generate_token', 'gozer_ajax_generate_token' );

/**
 * Handle AJAX token deletion.
 *
 * @since 1.0.0
 * @return void
 */
function gozer_ajax_delete_token() {

	check_ajax_referer( 'gozer_admin', 'nonce' );

	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( array( 'message' => __( 'Permission denied.', 'gozer' ) ) );
	}

	$index = isset( $_POST['index'] ) ? absint( $_POST['index'] ) : -1;

	$tokens = get_option( 'gozer_bypass_tokens', array() );

	if ( isset( $tokens[ $index ] ) ) {
		unset( $tokens[ $index ] );
		$tokens = array_values( $tokens );
		update_option( 'gozer_bypass_tokens', $tokens );
		wp_send_json_success();
	}

	wp_send_json_error( array( 'message' => __( 'Token not found.', 'gozer' ) ) );
}
add_action( 'wp_ajax_gozer_delete_token', 'gozer_ajax_delete_token' );
