<?php
/**
 * Admin Bar functionality for Gozer
 *
 * Adds a status indicator and quick toggle to the admin bar.
 *
 * @package Gozer
 * @since   1.0.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Add Force Login indicator to admin bar.
 *
 * @since 1.0.0
 * @param WP_Admin_Bar $wp_admin_bar Admin bar object.
 * @return void
 */
function gozer_admin_bar_menu( $wp_admin_bar ) {

	// Only show to users who can manage options.
	if ( ! current_user_can( 'manage_options' ) ) {
		return;
	}

	$settings   = gozer_get_settings();
	$is_enabled = ! empty( $settings['enabled'] );

	// Main menu item.
	$wp_admin_bar->add_node(
		array(
			'id'    => 'gozer',
			'title' => gozer_get_admin_bar_title( $is_enabled ),
			'href'  => admin_url( 'options-general.php?page=gozer' ),
			'meta'  => array(
				'class' => $is_enabled ? 'gozer-enabled' : 'gozer-disabled',
				'title' => $is_enabled
					? __( 'Gozer is active - click to configure', 'gozer' )
					: __( 'Gozer is inactive - click to configure', 'gozer' ),
			),
		)
	);

	// Toggle submenu item.
	$wp_admin_bar->add_node(
		array(
			'parent' => 'gozer',
			'id'     => 'gozer-toggle',
			'title'  => gozer_get_toggle_html( $is_enabled ),
			'href'   => '#',
			'meta'   => array(
				'class' => 'gozer-toggle-item',
			),
		)
	);

	// Settings link.
	$wp_admin_bar->add_node(
		array(
			'parent' => 'gozer',
			'id'     => 'gozer-settings',
			'title'  => __( 'Settings', 'gozer' ),
			'href'   => admin_url( 'options-general.php?page=gozer' ),
		)
	);
}
add_action( 'admin_bar_menu', 'gozer_admin_bar_menu', 100 );

/**
 * Get admin bar title with status indicator.
 *
 * @since 1.0.0
 * @param bool $is_enabled Whether force login is enabled.
 * @return string Admin bar title HTML.
 */
function gozer_get_admin_bar_title( $is_enabled ) {

	$status_dot = $is_enabled
		? '<span class="gozer-dot gozer-dot-on"></span>'
		: '<span class="gozer-dot gozer-dot-off"></span>';

	return $status_dot . '<span class="gozer-label">' . esc_html__( 'Gozer', 'gozer' ) . '</span>';
}

/**
 * Get toggle switch HTML.
 *
 * @since 1.0.0
 * @param bool $is_enabled Whether force login is enabled.
 * @return string Toggle HTML.
 */
function gozer_get_toggle_html( $is_enabled ) {

	$checked = $is_enabled ? 'checked' : '';
	$label   = $is_enabled
		? __( 'Protection: ON', 'gozer' )
		: __( 'Protection: OFF', 'gozer' );

	return sprintf(
		'<label class="gozer-switch">
			<input type="checkbox" id="gozer-toggle-checkbox" %s>
			<span class="gozer-slider"></span>
		</label>
		<span class="gozer-toggle-label">%s</span>',
		$checked,
		esc_html( $label )
	);
}

/**
 * Enqueue admin bar styles and scripts.
 *
 * Loads assets only when admin bar is showing and user has permissions.
 *
 * @since 1.0.0
 * @return void
 */
function gozer_admin_bar_assets() {

	if ( ! is_admin_bar_showing() || ! current_user_can( 'manage_options' ) ) {
		return;
	}

	// Enqueue admin bar styles.
	wp_enqueue_style(
		'gozer-admin-bar',
		GOZER_PLUGIN_URL . 'assets/css/admin-bar.css',
		array(),
		GOZER_VERSION
	);

	// Enqueue admin bar scripts.
	wp_enqueue_script(
		'gozer-admin-bar',
		GOZER_PLUGIN_URL . 'assets/js/admin-bar.js',
		array(),
		GOZER_VERSION,
		true
	);

	// Pass data to script.
	wp_localize_script(
		'gozer-admin-bar',
		'gozerBar',
		array(
			'ajaxUrl' => admin_url( 'admin-ajax.php' ),
			'nonce'   => wp_create_nonce( 'gozer_toggle' ),
		)
	);
}
add_action( 'wp_enqueue_scripts', 'gozer_admin_bar_assets' );
add_action( 'admin_enqueue_scripts', 'gozer_admin_bar_assets' );

/**
 * Handle AJAX toggle request.
 *
 * @since 1.0.0
 * @return void
 */
function gozer_ajax_toggle() {

	// Verify nonce.
	if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'gozer_toggle' ) ) {
		wp_send_json_error( array( 'message' => __( 'Security check failed.', 'gozer' ) ) );
	}

	// Check permissions.
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( array( 'message' => __( 'Permission denied.', 'gozer' ) ) );
	}

	// Get new state.
	$enabled = isset( $_POST['enabled'] ) ? absint( $_POST['enabled'] ) : 0;

	// Update settings.
	$settings            = gozer_get_settings();
	$settings['enabled'] = (bool) $enabled;

	update_option( 'gozer_settings', $settings );

	wp_send_json_success(
		array(
			'enabled' => $enabled,
			'message' => $enabled
				? __( 'Gozer enabled.', 'gozer' )
				: __( 'Gozer disabled.', 'gozer' ),
		)
	);
}
add_action( 'wp_ajax_gozer_toggle', 'gozer_ajax_toggle' );
