<?php
/**
 * Plugin Name:       Gozer
 * Plugin URI:        https://servicios.ayudawp.com
 * Description:       Force visitors to log in before accessing your site. Configure exceptions for REST API, feeds, sitemaps, specific pages, IPs, and more through an intuitive settings page.
 * Version:           1.0.1
 * Requires at least: 5.0
 * Requires PHP:      7.4
 * Author:            Fernando Tellado
 * Author URI:        https://ayudawp.com/
 * License:           GPLv2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       gozer
 *
 * @package Gozer
 */

defined( 'ABSPATH' ) || exit;

/*
 * ==========================================================================
 * PLUGIN CONSTANTS
 * ==========================================================================
 */

define( 'GOZER_VERSION', '1.0.1' );
define( 'GOZER_PLUGIN_FILE', __FILE__ );
define( 'GOZER_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'GOZER_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'GOZER_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );

/*
 * ==========================================================================
 * INCLUDE REQUIRED FILES
 * ==========================================================================
 */

// Helper functions for checking exceptions.
require_once GOZER_PLUGIN_DIR . 'includes/helpers.php';

// Main force login functionality.
require_once GOZER_PLUGIN_DIR . 'includes/force-login.php';

// Admin settings page (only in admin area).
if ( is_admin() ) {
	require_once GOZER_PLUGIN_DIR . 'includes/class-gozer-promo-banner.php';
	require_once GOZER_PLUGIN_DIR . 'includes/admin-settings.php';
}

// Admin bar functionality.
require_once GOZER_PLUGIN_DIR . 'includes/admin-bar.php';

/*
 * ==========================================================================
 * PLUGIN ACTIVATION / DEACTIVATION
 * ==========================================================================
 */

/**
 * Plugin activation hook.
 *
 * Sets default options on first activation and triggers activation notice.
 *
 * @since 1.0.0
 * @return void
 */
function gozer_activate() {

	// Only set defaults if options don't exist (fresh install).
	if ( false === get_option( 'gozer_settings' ) ) {
		$defaults = gozer_get_default_settings();
		add_option( 'gozer_settings', $defaults );
	}

	// Initialize bypass tokens array.
	if ( false === get_option( 'gozer_bypass_tokens' ) ) {
		add_option( 'gozer_bypass_tokens', array() );
	}

	// Set transient to show activation notice.
	set_transient( 'gozer_activation_notice', true, 60 );

	// Clear any cached data.
	wp_cache_flush();
}
register_activation_hook( __FILE__, 'gozer_activate' );

/**
 * Plugin deactivation hook.
 *
 * Cleans up transients, scheduled events, and caches.
 *
 * @since 1.0.0
 * @return void
 */
function gozer_deactivate() {
	delete_transient( 'gozer_activation_notice' );
	wp_clear_scheduled_hook( 'gozer_cleanup_tokens' );
	wp_cache_flush();
}
register_deactivation_hook( __FILE__, 'gozer_deactivate' );

/*
 * ==========================================================================
 * ACTIVATION NOTICE
 * ==========================================================================
 */

/**
 * Display activation notice with settings summary.
 *
 * Shows a dismissible notice after plugin activation with quick
 * access to settings and default exceptions status.
 *
 * @since 1.0.0
 * @return void
 */
function gozer_activation_notice() {

	// Only show if transient exists and user can manage options.
	if ( ! get_transient( 'gozer_activation_notice' ) || ! current_user_can( 'manage_options' ) ) {
		return;
	}

	$settings = gozer_get_settings();

	// Build status indicators for enabled/disabled exceptions.
	$enabled_exceptions  = array();
	$disabled_exceptions = array();

	// System exceptions.
	$exception_labels = array(
		'allow_rest_api' => __( 'REST API', 'gozer' ),
		'allow_cron'     => __( 'WP-Cron', 'gozer' ),
		'allow_cli'      => __( 'WP-CLI', 'gozer' ),
		'allow_ajax'     => __( 'AJAX', 'gozer' ),
		'allow_xmlrpc'   => __( 'XML-RPC', 'gozer' ),
		'allow_sitemap'  => __( 'Sitemaps', 'gozer' ),
		'allow_robots'   => __( 'robots.txt', 'gozer' ),
		'allow_feeds'    => __( 'RSS Feeds', 'gozer' ),
		'allow_seo_bots' => __( 'SEO Bots', 'gozer' ),
		'allow_head'     => __( 'HEAD requests', 'gozer' ),
		'allow_static'   => __( 'Static files', 'gozer' ),
	);

	foreach ( $exception_labels as $key => $label ) {
		if ( ! empty( $settings[ $key ] ) ) {
			$enabled_exceptions[] = $label;
		} else {
			$disabled_exceptions[] = $label;
		}
	}

	$settings_url = admin_url( 'options-general.php?page=gozer' );

	?>
	<div class="notice notice-success is-dismissible gozer-activation-notice">
		<p>
			<strong><?php esc_html_e( 'Gozer "The Gatekeeper" activated.', 'gozer' ); ?></strong>
			<?php esc_html_e( 'Force login is now enabled. Visitors must log in to access your site.', 'gozer' ); ?>
		</p>

		<p>
			<strong><?php esc_html_e( 'Allowed by default:', 'gozer' ); ?></strong>
			<?php echo esc_html( implode( ', ', $enabled_exceptions ) ); ?>
			<br>
			<strong><?php esc_html_e( 'Blocked by default:', 'gozer' ); ?></strong>
			<?php echo esc_html( implode( ', ', $disabled_exceptions ) ); ?>
		</p>

		<p>
			<a href="<?php echo esc_url( $settings_url ); ?>" class="button button-primary">
				<?php esc_html_e( 'Configure exceptions', 'gozer' ); ?>
			</a>
		</p>
	</div>
	<?php

	// Delete transient so notice only shows once.
	delete_transient( 'gozer_activation_notice' );
}
add_action( 'admin_notices', 'gozer_activation_notice' );

/*
 * ==========================================================================
 * DEFAULT SETTINGS
 * ==========================================================================
 */

/**
 * Get default plugin settings.
 *
 * Centralizes all default values for the plugin options.
 * Defaults are set to ensure WordPress core functionality
 * works correctly while maximizing security.
 *
 * @since 1.0.0
 * @return array Default settings array.
 */
function gozer_get_default_settings() {

	return array(
		// General - enabled by default to activate protection immediately.
		'enabled'             => true,

		// System exceptions - enabled by default to prevent breaking core functionality.
		'allow_rest_api'      => true,  // Required for Gutenberg editor.
		'allow_cron'          => true,  // Required for scheduled tasks.
		'allow_cli'           => true,  // Required for command-line operations.
		'allow_ajax'          => true,  // Required for dynamic functionality.

		// SEO exceptions - enabled by default to maintain search visibility.
		'allow_sitemap'       => true,  // Search engines need sitemap access.
		'allow_robots'        => true,  // Required for crawler instructions.
		'allow_seo_bots'      => true,  // Allow search engine indexing.

		// Security-sensitive - disabled by default for security.
		'allow_xmlrpc'        => false, // Common attack vector.
		'allow_feeds'         => false, // Usually not needed for private sites.
		'allow_static'        => false, // Theme should handle its own assets.

		// Technical - HEAD enabled for monitoring services.
		'allow_head'          => true,  // Uptime monitoring needs this.

		// Custom exceptions - empty by default.
		'allowed_paths'       => '',
		'allowed_ips'         => '',
		'allowed_user_agents' => '',

		// Redirect behavior - login page is most user-friendly.
		'redirect_type'       => 'login',
		'redirect_url'        => '',
	);
}

/**
 * Get plugin settings with defaults.
 *
 * Retrieves saved settings and merges with defaults
 * to ensure all keys exist.
 *
 * @since 1.0.0
 * @return array Plugin settings.
 */
function gozer_get_settings() {

	$defaults = gozer_get_default_settings();
	$settings = get_option( 'gozer_settings', array() );

	return wp_parse_args( $settings, $defaults );
}

/*
 * ==========================================================================
 * PLUGIN LINKS
 * ==========================================================================
 */

/**
 * Add settings link to plugins page.
 *
 * @since 1.0.0
 * @param array $links Existing plugin action links.
 * @return array Modified links array.
 */
function gozer_plugin_action_links( $links ) {

	$settings_link = sprintf(
		'<a href="%s">%s</a>',
		esc_url( admin_url( 'options-general.php?page=gozer' ) ),
		esc_html__( 'Settings', 'gozer' )
	);

	array_unshift( $links, $settings_link );

	return $links;
}
add_filter( 'plugin_action_links_' . GOZER_PLUGIN_BASENAME, 'gozer_plugin_action_links' );

/**
 * Add meta links to plugins page.
 *
 * @since 1.0.0
 * @param array  $links Existing meta links.
 * @param string $file  Plugin file path.
 * @return array Modified links array.
 */
function gozer_plugin_row_meta( $links, $file ) {

	if ( GOZER_PLUGIN_BASENAME !== $file ) {
		return $links;
	}

	$meta_links = array(
		'docs'    => sprintf(
			'<a href="%s" target="_blank" rel="noopener">%s</a>',
			esc_url( 'https://ayudawp.com/' ),
			esc_html__( 'Documentation', 'gozer' )
		),
		'support' => sprintf(
			'<a href="%s" target="_blank" rel="noopener">%s</a>',
			esc_url( 'https://servicios.ayudawp.com/' ),
			esc_html__( 'Support', 'gozer' )
		),
	);

	return array_merge( $links, $meta_links );
}
add_filter( 'plugin_row_meta', 'gozer_plugin_row_meta', 10, 2 );

/*
 * ==========================================================================
 * SCHEDULED CLEANUP
 * ==========================================================================
 */

/**
 * Schedule token cleanup on plugin load.
 *
 * @since 1.0.0
 * @return void
 */
function gozer_schedule_cleanup() {
	if ( ! wp_next_scheduled( 'gozer_cleanup_tokens' ) ) {
		wp_schedule_event( time(), 'daily', 'gozer_cleanup_tokens' );
	}
}
add_action( 'wp', 'gozer_schedule_cleanup' );

/**
 * Clean expired tokens daily.
 *
 * @since 1.0.0
 * @return void
 */
function gozer_do_cleanup_tokens() {
	gozer_clean_expired_tokens();
}
add_action( 'gozer_cleanup_tokens', 'gozer_do_cleanup_tokens' );
