<?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */
namespace SimpleCalendar\plugin_deps\Symfony\Component\Translation\Test;

use SimpleCalendar\plugin_deps\PHPUnit\Framework\MockObject\MockObject;
use SimpleCalendar\plugin_deps\PHPUnit\Framework\TestCase;
use SimpleCalendar\plugin_deps\Psr\Log\LoggerInterface;
use SimpleCalendar\plugin_deps\Symfony\Component\HttpClient\MockHttpClient;
use SimpleCalendar\plugin_deps\Symfony\Component\Translation\Dumper\XliffFileDumper;
use SimpleCalendar\plugin_deps\Symfony\Component\Translation\Exception\IncompleteDsnException;
use SimpleCalendar\plugin_deps\Symfony\Component\Translation\Exception\UnsupportedSchemeException;
use SimpleCalendar\plugin_deps\Symfony\Component\Translation\Loader\LoaderInterface;
use SimpleCalendar\plugin_deps\Symfony\Component\Translation\Provider\Dsn;
use SimpleCalendar\plugin_deps\Symfony\Component\Translation\Provider\ProviderFactoryInterface;
use SimpleCalendar\plugin_deps\Symfony\Component\Translation\TranslatorBagInterface;
use SimpleCalendar\plugin_deps\Symfony\Contracts\HttpClient\HttpClientInterface;
/**
 * A test case to ease testing a translation provider factory.
 *
 * @author Mathieu Santostefano <msantostefano@protonmail.com>
 */
abstract class ProviderFactoryTestCase extends TestCase
{
    protected HttpClientInterface $client;
    protected LoggerInterface|MockObject $logger;
    protected string $defaultLocale;
    protected LoaderInterface|MockObject $loader;
    protected XliffFileDumper|MockObject $xliffFileDumper;
    protected TranslatorBagInterface|MockObject $translatorBag;
    abstract public function createFactory(): ProviderFactoryInterface;
    /**
     * @return iterable<array{0: bool, 1: string}>
     */
    abstract public static function supportsProvider(): iterable;
    /**
     * @return iterable<array{0: string, 1: string}>
     */
    abstract public static function createProvider(): iterable;
    /**
     * @return iterable<array{0: string, 1: string|null}>
     */
    public static function unsupportedSchemeProvider(): iterable
    {
        return [];
    }
    /**
     * @return iterable<array{0: string, 1: string|null}>
     */
    public static function incompleteDsnProvider(): iterable
    {
        return [];
    }
    /**
     * @dataProvider supportsProvider
     */
    public function testSupports(bool $expected, string $dsn)
    {
        $factory = $this->createFactory();
        $this->assertSame($expected, $factory->supports(new Dsn($dsn)));
    }
    /**
     * @dataProvider createProvider
     */
    public function testCreate(string $expected, string $dsn)
    {
        $factory = $this->createFactory();
        $provider = $factory->create(new Dsn($dsn));
        $this->assertSame($expected, (string) $provider);
    }
    /**
     * @dataProvider unsupportedSchemeProvider
     */
    public function testUnsupportedSchemeException(string $dsn, ?string $message = null)
    {
        $factory = $this->createFactory();
        $dsn = new Dsn($dsn);
        $this->expectException(UnsupportedSchemeException::class);
        if (null !== $message) {
            $this->expectExceptionMessage($message);
        }
        $factory->create($dsn);
    }
    /**
     * @dataProvider incompleteDsnProvider
     */
    public function testIncompleteDsnException(string $dsn, ?string $message = null)
    {
        $factory = $this->createFactory();
        $dsn = new Dsn($dsn);
        $this->expectException(IncompleteDsnException::class);
        if (null !== $message) {
            $this->expectExceptionMessage($message);
        }
        $factory->create($dsn);
    }
    protected function getClient(): HttpClientInterface
    {
        return $this->client ??= new MockHttpClient();
    }
    protected function getLogger(): LoggerInterface
    {
        return $this->logger ??= $this->createMock(LoggerInterface::class);
    }
    protected function getDefaultLocale(): string
    {
        return $this->defaultLocale ??= 'en';
    }
    protected function getLoader(): LoaderInterface
    {
        return $this->loader ??= $this->createMock(LoaderInterface::class);
    }
    protected function getXliffFileDumper(): XliffFileDumper
    {
        return $this->xliffFileDumper ??= $this->createMock(XliffFileDumper::class);
    }
    protected function getTranslatorBag(): TranslatorBagInterface
    {
        return $this->translatorBag ??= $this->createMock(TranslatorBagInterface::class);
    }
}
