<?php
/**
 * Capabilities class.
 *
 * @access public
 * @since 6.0.0
 *
 * @package ExactMetrics
 * @subpackage Capabilities
 * @author  Chris Christoff
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Map ExactMetrics Capabilities.
 *
 * Using meta caps, we're creating virtual capabilities that are
 * for backwards compatibility reasons given to users with manage_options, and to
 * users who have at least of the roles selected in the options on the permissions
 * tab of the ExactMetrics settings.
 *
 * @access public
 *
 * @param array $caps Array of capabilities the user has.
 * @param string $cap The current cap being filtered.
 * @param int $user_id User to check permissions for.
 * @param array $args Extra parameters. Unused.
 *
 * @return array Array of caps needed to have this meta cap. If returned array is empty, user has the capability.
 * @since 6.0.0
 *
 */
function exactmetrics_add_capabilities( $caps, $cap, $user_id, $args ) {

	switch ( $cap ) {
		case 'exactmetrics_view_dashboard' :
			$roles = exactmetrics_get_option( 'view_reports', array() );

			$user_can_via_settings = false;
			if ( ! empty( $roles ) && is_array( $roles ) ) {
				foreach ( $roles as $role ) {
					if ( is_string( $role ) ) {
						if ( user_can( $user_id, $role ) ) {
							$user_can_via_settings = true;
							break;
						}
					}
				}
			} else if ( ! empty( $roles ) && is_string( $roles ) ) {
				if ( user_can( $user_id, $roles ) ) {
					$user_can_via_settings = true;
				}
			}

			if ( user_can( $user_id, 'manage_options' ) || $user_can_via_settings ) {
				$caps = array();
			}

			break;
		case 'exactmetrics_save_settings' :
			$roles = exactmetrics_get_option( 'save_settings', array() );

			$user_can_via_settings = false;
			if ( ! empty( $roles ) && is_array( $roles ) ) {
				foreach ( $roles as $role ) {
					if ( is_string( $role ) ) {
						if ( user_can( $user_id, $role ) ) {
							$user_can_via_settings = true;
							break;
						}
					}
				}
			} else if ( ! empty( $roles ) && is_string( $roles ) ) {
				if ( user_can( $user_id, $roles ) ) {
					$user_can_via_settings = true;
				}
			}

			if ( user_can( $user_id, 'manage_options' ) || $user_can_via_settings ) {
				$caps = array();
			}

			break;
	}

	return $caps;
}

add_filter( 'map_meta_cap', 'exactmetrics_add_capabilities', 10, 4 );

/**
 * Get the list of settings that only users with manage_options can modify.
 *
 * These are access-control settings that, if modified by a delegated user,
 * could lead to privilege escalation.
 *
 * @since 9.5.2
 *
 * @return array Array of admin-only setting keys.
 */
function exactmetrics_get_admin_only_settings() {
	$settings = array(
		'save_settings',
		'view_reports',
		'ignore_users',
	);

	return apply_filters( 'exactmetrics_admin_only_settings', $settings );
}

/**
 * Check if a given setting key is an admin-only setting.
 *
 * @since 9.5.2
 *
 * @param string $setting The setting key to check.
 *
 * @return bool True if the setting is admin-only, false otherwise.
 */
function exactmetrics_is_admin_only_setting( $setting ) {
	return in_array( $setting, exactmetrics_get_admin_only_settings(), true );
}
