<?php
/**
 * Plugin Name: Golala Email Sender
 * Plugin URI: https://wordpress.org/plugins/golala-email-sender/
 * Description: Allow user to change WordPress default mail sender name
 * Version: 1.0.0
 * Author: Golala Technology
 * Author URI: https://golalatechnology.com/
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: golala-email-sender
 * Requires PHP: 7.0
 * Requires at least: 5.0
 * Tested up to: 6.9
 *
 * @package GolalaEmailSender
 * @author Golala Technology
 * @license GPL v2 or later
 *
 * This program is free software; you can redistribute it and/or modify it under the terms of the GNU
 * General Public License version 2, as published by the Free Software Foundation. You may NOT assume
 * that you can use any other version of the GPL.
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Golala_Email_Sender {
    
    private $options;
    
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'settings_init'));
        add_filter('wp_mail_from', array($this, 'change_email_from'));
        add_filter('wp_mail_from_name', array($this, 'change_email_from_name'));
        add_action('admin_enqueue_scripts', array($this, 'admin_scripts'));
        
        // Plugin action links
        add_filter('plugin_action_links_' . plugin_basename(__FILE__), array($this, 'add_plugin_action_links'));
    }
    
    /**
     * Get sanitized domain from server or home URL
     */
    private function get_sanitized_domain() {
        // Get domain from home URL (most reliable)
        $domain = parse_url(home_url(), PHP_URL_HOST);
        
        // Sanitize the domain
        $domain = sanitize_text_field($domain);
        
        // Remove port if present
        $domain = strtok($domain, ':');
        
        // Basic validation - only allow alphanumeric, dots, and hyphens
        $domain = preg_replace('/[^a-zA-Z0-9.-]/', '', $domain);
        
        return $domain;
    }
    
    /**
     * Add plugin action links
     */
    public function add_plugin_action_links($links) {
        $settings_link = '<a href="' . esc_url(admin_url('options-general.php?page=golala-email-sender')) . '">' . esc_html__('Settings', 'golala-email-sender') . '</a>';
        array_unshift($links, $settings_link);
        return $links;
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_options_page(
            'Golala Email Sender',
            'Email Sender',
            'manage_options',
            'golala-email-sender',
            array($this, 'settings_page')
        );
    }
    
    /**
     * Initialize settings
     */
    public function settings_init() {
        register_setting(
            'golala_email_sender_settings',
            'golala_email_sender_options',
            array($this, 'sanitize_options')
        );
        
        add_settings_section(
            'golala_email_sender_section',
            'Email Sender Settings',
            array($this, 'section_callback'),
            'golala-email-sender'
        );
        
        add_settings_field(
            'sender_name',
            'Sender Name',
            array($this, 'sender_name_render'),
            'golala-email-sender',
            'golala_email_sender_section'
        );
        
        add_settings_field(
            'sender_email',
            'Sender Email',
            array($this, 'sender_email_render'),
            'golala-email-sender',
            'golala_email_sender_section'
        );
        
        add_settings_field(
            'test_email',
            'Test Email',
            array($this, 'test_email_render'),
            'golala-email-sender',
            'golala_email_sender_section'
        );
    }
    
    /**
     * Validate email domain for better deliverability
     */
    private function validate_email_domain($email) {
        if (!is_email($email)) {
            return false;
        }
        
        $parts = explode('@', $email);
        if (count($parts) !== 2) {
            return false;
        }
        
        $domain = $parts[1];
        $site_domain = $this->get_sanitized_domain();
        
        // Allow emails from the same domain or subdomains
        if (strpos($domain, $site_domain) !== false) {
            return true;
        }
        
        // Allow common email providers (optional - can be removed)
        $allowed_domains = array(
            'gmail.com',
            'outlook.com',
            'yahoo.com',
            'hotmail.com',
            'icloud.com',
            'protonmail.com',
            'zoho.com'
        );
        
        if (in_array($domain, $allowed_domains)) {
            return true;
        }
        
        // Allow via filter for developers
        return apply_filters('golala_email_sender_allow_email_domain', false, $domain, $site_domain);
    }
    
    /**
     * Enhanced sanitize options with validation
     */
    public function sanitize_options($input) {
        $sanitized = array();
        $errors = array();
        
        // Sanitize sender name
        if (isset($input['sender_name'])) {
            $sanitized['sender_name'] = sanitize_text_field($input['sender_name']);
            if (empty($sanitized['sender_name'])) {
                $errors[] = __('Sender name cannot be empty.', 'golala-email-sender');
            }
        }
        
        // Sanitize and validate sender email
        if (isset($input['sender_email'])) {
            $email = sanitize_email($input['sender_email']);
            
            if (!is_email($email)) {
                $errors[] = __('Please enter a valid email address.', 'golala-email-sender');
            } elseif (!$this->validate_email_domain($email)) {
                // Show warning but allow it (less restrictive)
                add_settings_error(
                    'golala_email_sender_options',
                    'email_domain_warning',
                    sprintf(
                        __('Warning: Using email "%s" from a different domain may affect email deliverability.', 'golala-email-sender'),
                        $email
                    ),
                    'warning'
                );
                $sanitized['sender_email'] = $email;
            } else {
                $sanitized['sender_email'] = $email;
            }
        }
        
        // Show errors if any
        if (!empty($errors)) {
            foreach ($errors as $error) {
                add_settings_error('golala_email_sender_options', 'validation_error', $error, 'error');
            }
            // Return current options on error
            return get_option('golala_email_sender_options', array());
        }
        
        return $sanitized;
    }
    
    /**
     * Section callback
     */
    public function section_callback() {
        echo '<p>' . esc_html__('Change the default WordPress email sender name and address.', 'golala-email-sender') . '</p>';
    }
    
    /**
     * Sender name field
     */
    public function sender_name_render() {
        $options = get_option('golala_email_sender_options');
        $value = isset($options['sender_name']) ? $options['sender_name'] : get_bloginfo('name');
        ?>
        <input type="text" name="golala_email_sender_options[sender_name]" 
               value="<?php echo esc_attr($value); ?>" 
               class="regular-text"
               placeholder="<?php echo esc_attr(sprintf(/* translators: %s: Site name */ __('e.g., %s', 'golala-email-sender'), get_bloginfo('name'))); ?>">
        <p class="description"><?php esc_html_e('This will replace "WordPress" as the sender name.', 'golala-email-sender'); ?></p>
        <?php
    }
    
    /**
     * Sender email field - FIXED: Line 229
     */
    public function sender_email_render() {
        $options = get_option('golala_email_sender_options');
        $value = isset($options['sender_email']) ? $options['sender_email'] : get_option('admin_email');
        $site_domain = $this->get_sanitized_domain(); // FIXED: Using sanitized domain
        ?>
        <input type="email" name="golala_email_sender_options[sender_email]" 
               value="<?php echo esc_attr($value); ?>" 
               class="regular-text"
               placeholder="<?php echo esc_attr(sprintf(/* translators: %s: Domain name */ __('e.g., noreply@%s', 'golala-email-sender'), $site_domain)); ?>">
        <p class="description"><?php esc_html_e('Use an email from your domain (not wordpress@...)', 'golala-email-sender'); ?></p>
        <?php
    }
    
    /**
     * Test email field
     */
    public function test_email_render() {
        ?>
        <input type="email" id="test_email_address" 
               value="<?php echo esc_attr(get_option('admin_email')); ?>" 
               class="regular-text"
               placeholder="<?php esc_attr_e('Enter email to test', 'golala-email-sender'); ?>">
        <button type="button" id="send_test_email" class="button button-secondary">
            <?php esc_html_e('Send Test Email', 'golala-email-sender'); ?>
        </button>
        <span id="test_email_status"></span>
        <p class="description"><?php esc_html_e('Send a test email to verify your settings.', 'golala-email-sender'); ?></p>
        <?php
    }
    
    /**
     * Settings page - FIXED: Line 265
     */
    public function settings_page() {
        ?>
        <div class="wrap golala-email-sender-wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <div class="notice notice-info">
                <p><strong><?php esc_html_e('Current default:', 'golala-email-sender'); ?></strong> WordPress &lt;wordpress@<?php echo esc_html($this->get_sanitized_domain()); ?>&gt;</p> <!-- FIXED: Using sanitized domain -->
                <p><strong><?php esc_html_e('New sender:', 'golala-email-sender'); ?></strong> 
                    <?php 
                    $options = get_option('golala_email_sender_options');
                    $name = isset($options['sender_name']) && !empty($options['sender_name']) ? $options['sender_name'] : get_bloginfo('name');
                    $email = isset($options['sender_email']) && !empty($options['sender_email']) ? $options['sender_email'] : get_option('admin_email');
                    echo esc_html($name) . ' &lt;' . esc_html($email) . '&gt;';
                    ?>
                </p>
            </div>
            
            <form action="options.php" method="post">
                <?php
                settings_fields('golala_email_sender_settings');
                do_settings_sections('golala-email-sender');
                submit_button(esc_html__('Save Changes', 'golala-email-sender'));
                ?>
            </form>
            
            <div class="golala-email-sender-info">
                <h3><?php esc_html_e('How It Works', 'golala-email-sender'); ?></h3>
                <ul>
                    <li><strong><?php esc_html_e('Sender Name:', 'golala-email-sender'); ?></strong> <?php esc_html_e('Replaces "WordPress" in all outgoing emails', 'golala-email-sender'); ?></li>
                    <li><strong><?php esc_html_e('Sender Email:', 'golala-email-sender'); ?></strong> <?php esc_html_e('Changes from wordpress@yourdomain.com to your custom email', 'golala-email-sender'); ?></li>
                    <li><strong><?php esc_html_e('Affected Emails:', 'golala-email-sender'); ?></strong> <?php esc_html_e('Password resets, notifications, comments, registration emails, etc.', 'golala-email-sender'); ?></li>
                </ul>
                
                <h3><?php esc_html_e('Tips for Better Email Delivery', 'golala-email-sender'); ?></h3>
                <ol>
                    <li><?php esc_html_e('Use an email address from your domain (not Gmail/Yahoo)', 'golala-email-sender'); ?></li>
                    <li><?php esc_html_e('For best results, use: noreply@yourdomain.com', 'golala-email-sender'); ?></li>
                    <li><?php esc_html_e('Consider using an SMTP plugin for better deliverability', 'golala-email-sender'); ?></li>
                </ol>
                
                <h3><?php esc_html_e('Support', 'golala-email-sender'); ?></h3>
                <p><?php esc_html_e('Need help?', 'golala-email-sender'); ?> <a href="https://wordpress.org/support/plugin/golala-email-sender/" target="_blank"><?php esc_html_e('Visit the support forums', 'golala-email-sender'); ?></a>.</p>
            </div>
        </div>
        <?php
    }
    
    /**
     * Change email from address
     */
    public function change_email_from($original_email) {
        $options = get_option('golala_email_sender_options');
        
        if (isset($options['sender_email']) && !empty($options['sender_email']) && is_email($options['sender_email'])) {
            return sanitize_email($options['sender_email']);
        }
        
        return $original_email;
    }
    
    /**
     * Change email from name
     */
    public function change_email_from_name($original_name) {
        $options = get_option('golala_email_sender_options');
        
        if (isset($options['sender_name']) && !empty($options['sender_name'])) {
            return sanitize_text_field($options['sender_name']);
        }
        
        // Fallback to site name if custom name not set
        return get_bloginfo('name');
    }
    
    /**
     * Admin scripts
     */
    public function admin_scripts($hook) {
        if ($hook != 'settings_page_golala-email-sender') {
            return;
        }
        
        // Enqueue script
        wp_enqueue_script(
            'golala-email-sender-admin',
            plugin_dir_url(__FILE__) . 'admin.js',
            array('jquery'),
            '1.0.0',
            true
        );
        
        // Localize script for AJAX
        wp_localize_script(
            'golala-email-sender-admin',
            'golala_email_sender',
            array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('golala_email_sender_test'),
                'text_enter_email' => __('Please enter an email address', 'golala-email-sender'),
                'text_valid_email' => __('Please enter a valid email address', 'golala-email-sender'),
                'text_sending' => __('Sending...', 'golala-email-sender'),
                'text_sent' => __('Test email sent to', 'golala-email-sender'),
                'text_failed' => __('Failed to send test email', 'golala-email-sender'),
                'text_error' => __('Error sending test email', 'golala-email-sender')
            )
        );
        
        // Add inline CSS properly
        wp_add_inline_style('wp-admin', '
            .golala-email-sender-wrap {
                max-width: 800px;
            }
            .golala-email-sender-info {
                background: #fff;
                border: 1px solid #ccd0d4;
                padding: 20px;
                margin-top: 30px;
                border-radius: 4px;
            }
            .golala-email-sender-info h3 {
                margin-top: 0;
                color: #1d2327;
            }
            .golala-email-sender-info ul,
            .golala-email-sender-info ol {
                padding-left: 20px;
            }
            .golala-email-sender-info li {
                margin-bottom: 8px;
            }
            #test_email_status {
                margin-left: 10px;
                display: inline-block;
                vertical-align: middle;
            }
            #test_email_status .spinner {
                float: none;
                margin: 0 5px;
            }
            .golala-email-sender-wrap .notice {
                max-width: 600px;
            }
        ');
    }
    
    /**
     * Enhanced send test email with validation
     */
    public function send_test_email() {
        check_ajax_referer('golala_email_sender_test', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized', 403);
        }
        
        $to = isset($_POST['email']) ? sanitize_email($_POST['email']) : '';
        
        if (!is_email($to)) {
            wp_send_json_error(__('Invalid recipient email address.', 'golala-email-sender'));
            return;
        }
        
        $options = get_option('golala_email_sender_options');
        $sender_name = isset($options['sender_name']) ? $options['sender_name'] : get_bloginfo('name');
        $sender_email = isset($options['sender_email']) ? $options['sender_email'] : get_option('admin_email');
        
        // Validate sender email before using
        if (!is_email($sender_email)) {
            wp_send_json_error(__('Invalid sender email in settings.', 'golala-email-sender'));
            return;
        }
        
        $subject = sprintf(__('Test Email from %s', 'golala-email-sender'), $sender_name);
        $message = __('This is a test email from your WordPress site.', 'golala-email-sender') . "\n\n";
        $message .= sprintf(__('Sender Name: %s', 'golala-email-sender'), $sender_name) . "\n";
        $message .= sprintf(__('Sender Email: %s', 'golala-email-sender'), $sender_email) . "\n";
        $message .= sprintf(__('Sent: %s', 'golala-email-sender'), date('Y-m-d H:i:s')) . "\n";
        $message .= sprintf(__('Website: %s', 'golala-email-sender'), get_bloginfo('url')) . "\n\n";
        $message .= __('This confirms your email settings are working correctly.', 'golala-email-sender');
        
        $headers = array(
            'From: ' . $sender_name . ' <' . $sender_email . '>',
            'Content-Type: text/plain; charset=UTF-8',
            'X-Golala-Email-Sender: 1.0.0'
        );
        
        // Add filters temporarily to ensure our settings are used
        add_filter('wp_mail_from', array($this, 'change_email_from'), 999);
        add_filter('wp_mail_from_name', array($this, 'change_email_from_name'), 999);
        
        $sent = wp_mail($to, $subject, $message, $headers);
        
        // Remove filters
        remove_filter('wp_mail_from', array($this, 'change_email_from'), 999);
        remove_filter('wp_mail_from_name', array($this, 'change_email_from_name'), 999);
        
        if ($sent) {
            wp_send_json_success(__('Test email sent successfully.', 'golala-email-sender'));
        } else {
            wp_send_json_error(__('Failed to send test email. Check your WordPress mail configuration.', 'golala-email-sender'));
        }
    }
    
    /**
     * Plugin deactivation cleanup
     */
    public static function deactivate() {
        // Clear any scheduled events
        wp_clear_scheduled_hook('golala_email_sender_daily_check');
    }
}

// Initialize the plugin
$golala_email_sender = new Golala_Email_Sender();

// AJAX handler for test email
add_action('wp_ajax_golala_email_sender_test_email', array($golala_email_sender, 'send_test_email'));

/**
 * Activation hook
 */
register_activation_hook(__FILE__, function() {
    // Set default options on activation
    $default_options = array(
        'sender_name' => get_bloginfo('name'),
        'sender_email' => get_option('admin_email')
    );
    
    if (!get_option('golala_email_sender_options')) {
        add_option('golala_email_sender_options', $default_options);
    }
    
    // Add activation notice
    set_transient('golala_email_sender_activated', true, 30);
});

/**
 * Deactivation hook
 */
register_deactivation_hook(__FILE__, array('Golala_Email_Sender', 'deactivate'));

/**
 * Display activation notice
 */
add_action('admin_notices', function() {
    if (get_transient('golala_email_sender_activated')) {
        ?>
        <div class="notice notice-success is-dismissible">
            <p><?php 
            printf(
                esc_html__('Thank you for installing Golala Email Sender! Configure it at %s.', 'golala-email-sender'),
                '<a href="' . esc_url(admin_url('options-general.php?page=golala-email-sender')) . '">' . esc_html__('Settings → Email Sender', 'golala-email-sender') . '</a>'
            );
            ?></p>
        </div>
        <?php
        delete_transient('golala_email_sender_activated');
    }
});