const { registerBlockType } = wp.blocks;
const { PanelBody, TextControl, CheckboxControl, Button, Spinner } =
    wp.components;
const { InspectorControls } = wp.blockEditor;
const { useState, useEffect } = wp.element;

registerBlockType("glycerine/table", {
    title: "Glycerine Table",
    icon: "table-col-after",
    category: "common",
    description: "Embed a paginated table of annotations from Glycerine with search filters.",
    example: {
        attributes: {
            isExample: true,
        },
    },

    attributes: {
        selectedCollections: {
            type: "array",
            default: [],
        },
        isRendered: {
            type: "boolean",
            default: false,
        },
        selectedColumns: {
            type: "array",
            default: [
                "Collection",
                "Image Set",
                "Annotation Set",
                "Image",
                "Title",
                "Description",
                "Notes",
                "Tags",
            ],
        },
        paging: {
            type: "number",
            default: 50,
        },
        imageQuality: {
            type: "number",
            default: 600,
        },
        columnLabelMap: {
            type: "object",
            default: {},
        },
        collectionMap: { type: "object", default: {} },
        isExample: {
            type: "boolean",
            default: false,
        }
    },

    edit: (props) => {
        const { attributes, setAttributes } = props;
        const [collections, setCollections] = useState([]);
        const [collectionMap, setCollectionMap] = useState({});
        const [isLoading, setIsLoading] = useState(true);

        const allColumns = [
            "Collection",
            "Image Set",
            "Annotation Set",
            "Image",
            "Title",
            "Description",
            "Notes",
            "Tags",
        ];

        if(attributes.isExample) {
            return createElement('img' , {
                src: "/wp-content/plugins/glycerine/assets/image/glycerine-table-example.png",
                alt: "Glycerine Table Example",
                style: { width: "100%", height: "auto", border: "2px solid black" }
            })
        }

        useEffect(() => {
            wp.apiRequest({
                path: "/glycerine/v1/collections",
                method: "GET",
            }).then((response) => {
                setCollections(response);
                setIsLoading(false);

                // Convert response to ID-to-Name map
                const idToNameMap = response.reduce((acc, collection) => {
                    acc[collection.id] = collection.name;
                    return acc;
                }, {});

                setCollectionMap(idToNameMap);
                setAttributes({ collectionMap: idToNameMap });
            });
        }, []);

        // Handle column selection
        const handleColumnChange = (column) => {
            let newSelectedColumns = attributes.selectedColumns.includes(column)
                ? attributes.selectedColumns.filter((col) => col !== column)
                : [...attributes.selectedColumns, column];

            // Prevent selectedColumns from being empty
            if (newSelectedColumns.length === 0) {
                newSelectedColumns = allColumns;
            }

            setAttributes({ selectedColumns: newSelectedColumns });
        };

        // Handle collection checkbox change
        const handleCheckboxChange = (collectionId) => {
            const newSelectedCollections =
                attributes.selectedCollections.includes(collectionId)
                    ? attributes.selectedCollections.filter(
                          (id) => id !== collectionId
                      )
                    : [...attributes.selectedCollections, collectionId];

            setAttributes({ selectedCollections: newSelectedCollections });
        };

        const handleColumnOrderChange = (column, value) => {
            let newOrder = [...attributes.selectedColumns];
            const index = newOrder.indexOf(column);

            if (index !== -1) {
                newOrder.splice(index, 1); // Remove column
                newOrder.splice(value - 1, 0, column); // Insert at new position
            }
            setAttributes({ selectedColumns: newOrder });
        };

        return (
            <div>
                <InspectorControls>
                    <PanelBody title="Collections">
                        {isLoading ? (
                            <Spinner />
                        ) : (
                            collections.map((collection) => (
                                <CheckboxControl
                                    key={collection.id}
                                    label={collection.name}
                                    checked={attributes.selectedCollections.includes(
                                        collection.id
                                    )}
                                    onChange={() =>
                                        handleCheckboxChange(collection.id)
                                    }
                                />
                            ))
                        )}
                        {!isLoading && (
                            <Button
                                isPrimary
                                onClick={() =>
                                    setAttributes({ isRendered: true })
                                }
                            >
                                Render
                            </Button>
                        )}
                    </PanelBody>

                    <div
                        style={{
                            backgroundColor: "#d3f8d3",
                            padding: "10px",
                            borderRadius: "5px",
                        }}
                    >
                        <PanelBody title="Table Settings">
                            <TextControl
                                label="Paging (Rows per Page)"
                                type="number"
                                value={attributes.paging}
                                onChange={(value) =>
                                    setAttributes({
                                        paging: value
                                            ? parseInt(value, 10)
                                            : 50,
                                    })
                                }
                                min={1}
                            />

                            <TextControl
                                label="Image quality (300-1200)"
                                type="number"
                                value={attributes.imageQuality}
                                onChange={(value) =>
                                    setAttributes({
                                        imageQuality: value
                                            ? parseInt(value, 10)
                                            : 600,
                                    })
                                }
                                min={1}
                            />
                        </PanelBody>

                        <PanelBody title="Columns to Display">
                            {allColumns.map((column) => {
                                const isChecked =
                                    attributes.selectedColumns.includes(column);
                                const index =
                                    attributes.selectedColumns.indexOf(column);
                                const overrideLabel =
                                    attributes.columnLabelMap[column] || "";

                                return (
                                    <div
                                        key={column}
                                        style={{
                                            marginBottom: "10px",
                                            borderBottom: "2px solid black",
                                            paddingBottom: "5px",
                                        }}
                                    >
                                        <div
                                            style={{
                                                display: "flex",
                                                justifyContent: "space-between",
                                            }}
                                        >
                                            <CheckboxControl
                                                label={column}
                                                checked={isChecked}
                                                onChange={() =>
                                                    handleColumnChange(column)
                                                }
                                            />
                                            {isChecked && (
                                                <input
                                                    type="number"
                                                    value={index + 1}
                                                    min="1"
                                                    max={
                                                        attributes
                                                            .selectedColumns
                                                            .length
                                                    }
                                                    style={{
                                                        width: "40px",
                                                        height: "20px",
                                                    }}
                                                    onChange={(e) =>
                                                        handleColumnOrderChange(
                                                            column,
                                                            parseInt(
                                                                e.target.value,
                                                                10
                                                            )
                                                        )
                                                    }
                                                />
                                            )}
                                        </div>
                                        {isChecked && (
                                            <TextControl
                                                label="Override Label"
                                                value={overrideLabel}
                                                onChange={(newLabel) => {
                                                    setAttributes({
                                                        columnLabelMap: {
                                                            ...attributes.columnLabelMap,
                                                            [column]: newLabel,
                                                        },
                                                    });
                                                }}
                                            />
                                        )}
                                    </div>
                                );
                            })}
                        </PanelBody>
                    </div>
                </InspectorControls>

                <div className="glycerine-table">
                    <h4 style={{ textAlign: "center", marginBottom: "10px" }}>
                        Glycerine Table
                    </h4>

                    {attributes.isRendered && (
                        <div className="rendered-content">
                            <h5
                                style={{
                                    textAlign: "center",
                                    marginBottom: "0",
                                    marginTop: "0",
                                }}
                            >
                                Selected Collections:{" "}
                                {attributes.selectedCollections
                                    .map(
                                        (id) => collectionMap[id] || `ID ${id}`
                                    )
                                    .join(", ")}
                            </h5>
                            <table
                                style={{
                                    width: "100%",
                                    borderCollapse: "collapse",
                                    backgroundColor: "#f8f9fa",
                                    height: "150px",
                                }}
                            >
                                <thead>
                                    <tr>
                                        {attributes.selectedColumns.map(
                                            (column) => (
                                                <th
                                                    key={column}
                                                    style={{
                                                        padding: "10px",
                                                        border: "2px solid black",
                                                        backgroundColor:
                                                            "#d3f8d3",
                                                    }}
                                                >
                                                    {attributes.columnLabelMap[
                                                        column
                                                    ] || column}
                                                </th>
                                            )
                                        )}
                                    </tr>
                                </thead>
                                <tbody>
                                    <tr>
                                        {attributes.selectedColumns.map(
                                            (column) => (
                                                <td
                                                    key={column}
                                                    style={{
                                                        padding: "10px",
                                                        border: "1px solid #ddd",
                                                        height: "100px",
                                                        textAlign: "center",
                                                    }}
                                                >
                                                    (Data will appear here)
                                                </td>
                                            )
                                        )}
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                    )}
                </div>
            </div>
        );
    },

    save: (props) => {
        const { attributes } = props;

        return (
            <div className="glycerine-table">
                {attributes.isRendered ? (
                    <div className="rendered-content">
                        {attributes.selectedCollections.length > 0
                            ? attributes.selectedCollections.join(", ")
                            : "No collections selected."}
                    </div>
                ) : (
                    <p>Click "Render" to see selected collections.</p>
                )}
            </div>
        );
    },
});
