const { registerBlockType } = wp.blocks;
const { PanelBody, TextControl, CheckboxControl, Button, Spinner } =
    wp.components;
const { InspectorControls } = wp.blockEditor;
const { useState, useEffect } = wp.element;

registerBlockType("glycerine/image-set", {
    title: "Glycerine Image Set",
    icon: "format-gallery",
    category: "common",
    description: "Embed a Glycerine Image Set using the Glycerine Viewer.",
    example: {
        attributes: {
            isExample: true,
        },
    },

    attributes: {
        selectedImageSet: {
            type: "object",
            default: null,
        },
        urlParameters: {
            type: "string",
            default:
                "fullscreenbutton=1&indexbutton=1&annotationviewbutton=1&aboutpanebutton=1&settingpanebutton=1&infopanel=1",
        },
        isExample: {
            type: "boolean",
            default: false,
        }
    },

    edit: (props) => {
        const { attributes, setAttributes } = props;
        const [collections, setCollections] = useState([]);
        const [imageSets, setImageSets] = useState([]);
        const [loadingImageSets, setLoadingImageSets] = useState(false);
        const [isLoading, setIsLoading] = useState(true);
        const [selectedCollection, setSelectedCollection] = useState(null);

        if(attributes.isExample) {
            return createElement('img' , {
                src: "/wp-content/plugins/glycerine/assets/image/glycerine-image-set-example.png",
                alt: "Glycerine Image Set Example",
                style: { width: "100%", height: "auto", border: "2px solid black" }
            })
        }

        useEffect(() => {
            wp.apiRequest({
                path: "/glycerine/v1/collections",
                method: "GET",
            }).then((response) => {
                setCollections(response);
                setIsLoading(false);
            });
        }, []);

        const handleCheckboxChange = (param) => {
            const params = new URLSearchParams(attributes.urlParameters);
            const currentValue = params.get(param) === "1" ? "0" : "1";
            params.set(param, currentValue);
            setAttributes({ urlParameters: params.toString() });
        };

        const fetchImageSets = (collectionId) => {
            setLoadingImageSets(true);
            wp.apiRequest({
                path: `/glycerine/v1/collection/${collectionId}/image_sets`,
                method: "GET",
            })
                .then((response) => {
                    setLoadingImageSets(false);
                    if (response.length === 0) {
                        alert("No public image sets found for this collection");
                        return;
                    }
                    setImageSets(response);
                    setSelectedCollection(collectionId);
                })
                .finally(() => {
                    setLoadingImageSets(false);
                });
        };

        return (
            <div>
                <InspectorControls>
                    <PanelBody title="Choose an Image Set">
                        <Spinner
                            style={{
                                display: loadingImageSets ? "block" : "none",
                                marginBottom: "10px",
                            }}
                        />

                        {collections.map((collection) => (
                            <div key={collection.id}>
                                <div
                                    onClick={() =>
                                        selectedCollection === collection.id
                                            ? setSelectedCollection(null)
                                            : fetchImageSets(collection.id)
                                    }
                                    style={{
                                        display: "flex",
                                        justifyContent: "space-between",
                                        padding: "10px",
                                        border: "1px solid #ddd",
                                        cursor: "pointer",
                                        marginBottom: "5px",
                                        backgroundColor:
                                            selectedCollection === collection.id
                                                ? "#f0f0f0"
                                                : "#fff",
                                    }}
                                >
                                    <span
                                        style={{
                                            display: "flex",
                                            alignItems: "center",
                                        }}
                                    >
                                        <span style={{ marginRight: "10px" }}>
                                            📋
                                        </span>
                                        {collection.name}
                                    </span>
                                    <span
                                        style={{
                                            color: "#000",
                                            fontWeight: "bold",
                                        }}
                                    >
                                        {selectedCollection === collection.id
                                            ? "▼"
                                            : "→"}
                                    </span>
                                </div>

                                {selectedCollection === collection.id &&
                                    !loadingImageSets && (
                                        <div
                                            style={{
                                                overflow: "auto",
                                                maxHeight: "500px",
                                                transition:
                                                    "max-height 1s ease-in-out",
                                                border: "1px solid #ddd",
                                            }}
                                        >
                                            {imageSets.map((imageSet) => (
                                                <div
                                                    key={imageSet.id}
                                                    onClick={() => {
                                                        setAttributes({
                                                            selectedImageSet:
                                                                imageSet,
                                                        });
                                                    }}
                                                    style={{
                                                        display: "flex",
                                                        alignItems: "center",
                                                        padding: "10px",
                                                        cursor: "pointer",
                                                    }}
                                                >
                                                    <img
                                                        src={imageSet.thumbnail}
                                                        alt={imageSet.name}
                                                        style={{
                                                            minWidth: "50px",
                                                            minHeight: "50px",
                                                            maxWidth: "50px",
                                                            maxHeight: "50px",
                                                            marginRight: "10px",
                                                        }}
                                                    />
                                                    <p
                                                        style={{
                                                            margin: 0,
                                                            fontWeight: "bold",
                                                        }}
                                                    >
                                                        {imageSet.name}
                                                    </p>
                                                </div>
                                            ))}
                                        </div>
                                    )}
                            </div>
                        ))}
                    </PanelBody>

                    <PanelBody title="Display">
                        {[
                            { key: "fullscreenbutton", label: "Full Screen" },
                            { key: "indexbutton", label: "Index Page" },
                            {
                                key: "annotationviewbutton",
                                label: "Annotation Views",
                            },
                            { key: "aboutpanebutton", label: "About Pane" },
                            {
                                key: "settingpanebutton",
                                label: "Settings Pane",
                            },
                            { key: "infopanel", label: "Information Panel" },
                        ].map(({ key, label }) => (
                            <CheckboxControl
                                key={key}
                                label={label}
                                checked={
                                    new URLSearchParams(
                                        attributes.urlParameters
                                    ).get(key) === "1"
                                }
                                onChange={() => handleCheckboxChange(key)}
                            />
                        ))}
                    </PanelBody>
                </InspectorControls>

                {attributes.selectedImageSet ? (
                    <iframe
                        src={`${attributes.selectedImageSet.public_url}?${attributes.urlParameters}`}
                        style={{
                            width: "100%",
                            height: "500px",
                            border: "2px solid black",
                            marginTop: "10px",
                        }}
                    ></iframe>
                ) : (
                    <div className="glycerine-image-set">
                        <h4
                            style={{
                                textAlign: "center",
                                border: "2px solid black",
                                padding: "20px",
                            }}
                        >
                            Select an image set in the block settings
                        </h4>
                    </div>
                )}
            </div>
        );
    },
});
