const { registerBlockType } = wp.blocks;
const { PanelBody, TextControl, CheckboxControl, Button, Spinner } =
    wp.components;
const { InspectorControls } = wp.blockEditor;
const { useState, useEffect } = wp.element;

registerBlockType("glycerine/collection", {
    title: "Glycerine Collection",
    icon: "images-alt",
    category: "common",
    description: "Embed a Glycerine Collection using the Glycerine Viewer.",
    example: {
        attributes: {
            isExample: true,
        },
    },

    attributes: {
        selectedCollection: {
            type: "object",
            default: null,
        },
        urlParameters: {
            type: "string",
            default:
                "fullscreenbutton=1&indexbutton=1&annotationviewbutton=1&aboutpanebutton=1&settingpanebutton=1&infopanel=1",
        },
        isExample: {
            type: "boolean",
            default: false,
        }
    },

    edit: (props) => {
        const { attributes, setAttributes } = props;
        const [collections, setCollections] = useState([]);

        if(attributes.isExample) {
            return createElement('img' , {
                src: "/wp-content/plugins/glycerine/assets/image/glycerine-collection-example.png",
                alt: "Glycerine Collection Example",
                style: { width: "100%", height: "auto", border: "2px solid black" }
            })
        }

        useEffect(() => {
            wp.apiRequest({
                path: "/glycerine/v1/collections",
                method: "GET",
            }).then((response) => {
                const published = response.filter(
                    (collection) => collection.is_published === true
                );
                setCollections(published);
            });
        }, []);

        const handleCheckboxChange = (param) => {
            const params = new URLSearchParams(attributes.urlParameters);
            const currentValue = params.get(param) === "1" ? "0" : "1";
            params.set(param, currentValue);
            setAttributes({ urlParameters: params.toString() });
        };

        const setSelectedCollection = (collection) => {
            setAttributes({ selectedCollection: collection });
        };

        return (
            <div>
                <InspectorControls>
                    <PanelBody title="Choose a Collection">
                        {collections.map((collection) => (
                            <div key={collection.id}>
                                <div
                                    onClick={() =>
                                        setSelectedCollection(collection)
                                    }
                                    style={{
                                        display: "flex",
                                        justifyContent: "space-between",
                                        padding: "10px",
                                        border: "1px solid #ddd",
                                        cursor: "pointer",
                                        marginBottom: "5px",
                                        backgroundColor: "#d0ebff",
                                    }}
                                >
                                    <span
                                        style={{
                                            display: "flex",
                                            alignItems: "center",
                                        }}
                                    >
                                        <span style={{ marginRight: "10px" }}>
                                            📋
                                        </span>
                                        {collection.name}
                                    </span>
                                </div>
                            </div>
                        ))}
                    </PanelBody>

                    <PanelBody title="Display">
                        {[
                            { key: "fullscreenbutton", label: "Full Screen" },
                            { key: "indexbutton", label: "Index Page" },
                            {
                                key: "annotationviewbutton",
                                label: "Annotation Views",
                            },
                            { key: "aboutpanebutton", label: "About Pane" },
                            {
                                key: "settingpanebutton",
                                label: "Settings Pane",
                            },
                            { key: "infopanel", label: "Information Panel" },
                        ].map(({ key, label }) => (
                            <CheckboxControl
                                key={key}
                                label={label}
                                checked={
                                    new URLSearchParams(
                                        attributes.urlParameters
                                    ).get(key) === "1"
                                }
                                onChange={() => handleCheckboxChange(key)}
                            />
                        ))}
                    </PanelBody>
                </InspectorControls>

                {attributes.selectedCollection ? (
                    <iframe
                        src={`https://iaw.ardc-hdcl-sia-iaw.cloud.edu.au/publications/collections/${attributes.selectedCollection.publication_id}?${attributes.urlParameters}`}
                        style={{
                            width: "100%",
                            height: "500px",
                            border: "2px solid black",
                            marginTop: "10px",
                        }}
                    ></iframe>
                ) : (
                    <div className="glycerine-collection">
                        <h4
                            style={{
                                textAlign: "center",
                                border: "2px solid black",
                                padding: "20px",
                            }}
                        >
                            Select an Collection in the block settings
                        </h4>
                    </div>
                )}
            </div>
        );
    },
});
