import { getAnnotationUrl } from "../assets/js/annotation-cutout-displayer.js";

const { registerBlockType } = wp.blocks;
const { PanelBody, TextControl, CheckboxControl, Button, Spinner } =
    wp.components;
const { InspectorControls } = wp.blockEditor;
const { useState, useEffect, useRef } = wp.element;

registerBlockType("glycerine/annotation-cutouts", {
    title: "Glycerine Annotation Cutouts",
    icon: "admin-tools",
    category: "common",
    description: "Embed an image annotation cutout from Glycerine.",
    example: {
        attributes: {
            isExample: true,
        },
    },

    attributes: {
        selectedImageSet: {
            type: "object",
            default: null,
        },
        annotationData: {
            type: "object",
            default: null,
        },
        showAnnotation: {
            type: "boolean",
            default: false,
        },
        selectedFields: {
            type: "array",
            default: ["title"],
        },
        annotationTitle: {
            type: "string",
            default: null,
        },
        annotationDescription: {
            type: "string",
            default: null,
        },
        isExample: {
            type: "boolean",
            default: false,
        },
    },

    edit: (props) => {
        const { attributes, setAttributes } = props;
        const [collections, setCollections] = useState([]);
        const [imageSets, setImageSets] = useState([]);
        const [loadingImageSets, setLoadingImageSets] = useState(false);
        const [setIsLoading] = useState(true);
        const [selectedCollection, setSelectedCollection] = useState(null);

        const viewerId = `glycerine-standalone-viewer-${props.clientId}`;
        const annotationImageId = `annotation-image-${props.clientId}`;

        if (attributes.isExample) {
            return createElement("img", {
                src: "/wp-content/plugins/glycerine/assets/image/glycerine-annotation-example.png",
                alt: "Glycerine Annotation cutout Example",
                style: {
                    width: "100%",
                    height: "auto",
                    border: "2px solid black",
                },
            });
        }

        useEffect(() => {
            wp.apiRequest({
                path: "/glycerine/v1/collections",
                method: "GET",
            }).then((response) => {
                setCollections(response);
                setIsLoading(false);
            });
        }, []);

        // User click image set -> show glycerine viewer
        useEffect(() => {
            if (!attributes.selectedImageSet) return;

            const ele = document.getElementById(viewerId);
            if (!ele) return;

            ele.innerHTML = "";

            const iframe = document.createElement("iframe");
            iframe.setAttribute("src", attributes.selectedImageSet.public_url);
            iframe.setAttribute("style", "width:100%;height:700px;border:0;");
            iframe.setAttribute("title", "Glycerine Viewer");
            ele.appendChild(iframe);

            const onMessage = (event) => {
                const data = event?.data;
                if (!data || data.event !== "annotation-popup-opened") return;

                const match = data.details.match(
                    /annotation-sets\/([^/]+)\/annotations\/(\d+)/
                );
                if (match) {
                    let annotationSetPublishedId = match[1];
                    let annotationId = match[2];

                    wp.apiRequest({
                        path: `/glycerine/v1/annotations?annotation_set_publish_id=${annotationSetPublishedId}`,
                        method: "GET",
                    })
                        .then((annotationSetData) => {
                            if (
                                !annotationSetData ||
                                !annotationSetData.annotations ||
                                annotationSetData.annotations.length === 0
                            ) {
                                alert("Annotation not found");
                                return;
                            }

                            let found = false;
                            for (let annotation of annotationSetData.annotations) {
                                if (annotation.id == annotationId) {
                                    setAttributes({
                                        annotationData: annotation,
                                    });
                                    found = true;
                                    break;
                                }
                            }

                            if (!found) {
                                alert("Annotation not found");
                                return;
                            }

                            setTimeout(() => {
                                document
                                    .getElementById("embed-annotation-button")
                                    ?.scrollIntoView({
                                        behavior: "smooth",
                                        block: "center",
                                    });
                            }, 100);
                        })
                        .catch((error) => {
                            console.error(
                                "Failed to fetch annotation data:",
                                error
                            );
                        });
                }
            };
            window.addEventListener("message", onMessage);

            // Cleanup on unmount or dependency change
            return () => {
                window.removeEventListener("message", onMessage);
            };

        }, [attributes.selectedImageSet]);


        // After get annotation data, fetch the image URL
        useEffect(() => {
            if (attributes.annotationData && attributes.showAnnotation) {
                (async () => {
                    // Destroy current viewer if exists
                    const ele = document.getElementById(viewerId);
                    if (ele) {
                        ele.innerHTML = "";
                    }

                    const src = await getAnnotationUrl(
                        attributes.annotationData.target
                    );
                    document.getElementById(annotationImageId).src = src;
                })();
            }
        }, [attributes.annotationData]); // Runs when annotationData updates

        // After click collection, fetch image sets
        const fetchImageSets = (collectionId) => {
            setLoadingImageSets(true);
            wp.apiRequest({
                path: `/glycerine/v1/collection/${collectionId}/image_sets`,
                method: "GET",
            }).then((response) => {
                setLoadingImageSets(false);
                if (response.length === 0) {
                    alert("No public image sets found for this collection");
                    return;
                }
                setImageSets(response);
                setSelectedCollection(collectionId);
            });
        };

        // User click "Embed This Annotation" button
        const embedAnnotation = async () => {
            setAttributes({ showAnnotation: true });

            // Destroy current viewer if exists
            const ele = document.getElementById(viewerId);
            if (ele) {
                ele.innerHTML = "";
            }

            const src = await getAnnotationUrl(
                attributes.annotationData.target
            );

            document.getElementById(annotationImageId).src = src;
        };

        const handleCheckboxChange = (field) => {
            const newFields = attributes.selectedFields.includes(field)
                ? attributes.selectedFields.filter((f) => f !== field) // Remove if exists
                : [...attributes.selectedFields, field]; // Add if not exists

            setAttributes({ selectedFields: newFields });
        };

        return (
            <div>
                <InspectorControls>
                    {attributes.annotationData && (
                        <div
                            style={{
                                padding: "10px",
                                marginBottom: "20px",
                                display: "flex",
                                justifyContent: "center",
                            }}
                        >
                            <Button
                                id="embed-annotation-button"
                                isPrimary
                                onClick={() => embedAnnotation()}
                            >
                                Embed This Annotation
                            </Button>
                        </div>
                    )}

                    <PanelBody title="Choose an Image Set, then select an annotation in the viewer. Select 'Embed This Annotation' in the block settings.">
                        <Spinner
                            style={{
                                display: loadingImageSets ? "block" : "none",
                                marginBottom: "10px",
                            }}
                        />

                        {collections.map((collection) => (
                            <div key={collection.id}>
                                <div
                                    onClick={() => {
                                        setAttributes({
                                            annotationData: null,
                                            showAnnotation: false,
                                            selectedImageSet: null,
                                        });
                                        if (
                                            selectedCollection === collection.id
                                        ) {
                                            setSelectedCollection(null);
                                        } else {
                                            fetchImageSets(collection.id);
                                        }
                                    }}
                                    style={{
                                        display: "flex",
                                        justifyContent: "space-between",
                                        padding: "10px",
                                        border: "1px solid #ddd",
                                        cursor: "pointer",
                                        marginBottom: "5px",
                                        backgroundColor:
                                            selectedCollection === collection.id
                                                ? "#f0f0f0"
                                                : "#fff",
                                    }}
                                >
                                    <span
                                        style={{
                                            display: "flex",
                                            alignItems: "center",
                                        }}
                                    >
                                        <span style={{ marginRight: "10px" }}>
                                            📋
                                        </span>
                                        {collection.name}
                                    </span>
                                    <span
                                        style={{
                                            color: "#000",
                                            fontWeight: "bold",
                                        }}
                                    >
                                        {selectedCollection === collection.id
                                            ? "▼"
                                            : "→"}
                                    </span>
                                </div>

                                {selectedCollection === collection.id &&
                                    !loadingImageSets && (
                                        <div
                                            style={{
                                                overflow: "auto",
                                                maxHeight: "500px",
                                                transition:
                                                    "max-height 1s ease-in-out",
                                                border: "1px solid #ddd",
                                            }}
                                        >
                                            {imageSets.map((imageSet) => (
                                                <div
                                                    key={imageSet.id}
                                                    onClick={() => {
                                                        setAttributes({
                                                            annotationData:
                                                                null,
                                                            showAnnotation: false,
                                                            selectedImageSet:
                                                                imageSet,
                                                        });
                                                    }}
                                                    style={{
                                                        display: "flex",
                                                        alignItems: "center",
                                                        padding: "10px",
                                                        cursor: "pointer",
                                                    }}
                                                >
                                                    <img
                                                        src={imageSet.thumbnail}
                                                        alt={imageSet.name}
                                                        style={{
                                                            minWidth: "50px",
                                                            minHeight: "50px",
                                                            maxWidth: "50px",
                                                            maxHeight: "50px",
                                                            marginRight: "10px",
                                                        }}
                                                    />
                                                    <p
                                                        style={{
                                                            margin: 0,
                                                            fontWeight: "bold",
                                                        }}
                                                    >
                                                        {imageSet.name}
                                                    </p>
                                                </div>
                                            ))}
                                        </div>
                                    )}
                            </div>
                        ))}
                    </PanelBody>

                    <PanelBody title="Caption">
                        <CheckboxControl
                            key="title"
                            label="Title"
                            checked={attributes.selectedFields?.includes(
                                "title"
                            )}
                            onChange={() => handleCheckboxChange("title")}
                        />

                        <TextControl
                            label="Annotation Title"
                            type="string"
                            value={attributes.annotationTitle}
                            onChange={(value) =>
                                setAttributes({
                                    annotationTitle: value,
                                })
                            }
                        />

                        <CheckboxControl
                            key="description"
                            label="Description"
                            checked={attributes.selectedFields?.includes(
                                "description"
                            )}
                            onChange={() => handleCheckboxChange("description")}
                        />

                        <TextControl
                            label="Annotation Description"
                            type="string"
                            value={attributes.annotationDescription}
                            onChange={(value) =>
                                setAttributes({
                                    annotationDescription: value,
                                })
                            }
                        />
                    </PanelBody>
                </InspectorControls>

                {attributes.showAnnotation ? (
                    attributes.annotationData && (
                        <div
                            style={{
                                display: "flex",
                                justifyContent: "center",
                                height: "100%",
                            }}
                        >
                            <img
                                src={glycerineSettings.loadingImageUrl}
                                id={annotationImageId}
                                alt="Loading..."
                                style={{
                                    maxWidth: "250px",
                                    maxHeight: "220px",
                                }}
                            />
                        </div>
                    )
                ) : attributes.selectedImageSet ? (
                    <div
                        id={viewerId}
                        style={{
                            border: "2px solid black",
                            marginTop: "10px",
                        }}
                    ></div>
                ) : (
                    <div>
                        <h4
                            style={{
                                textAlign: "center",
                                border: "2px solid black",
                                padding: "20px",
                            }}
                        >
                            Select an image set in the block settings
                        </h4>
                    </div>
                )}
            </div>
        );
    },
});
