<?php
if (! defined('ABSPATH')) exit; // Exit if accessed directly
require_once('GlycerineIAWClient.php');

/**
 * Plugin Name: Glycerine
 * Description: Enhance your WordPress content with annotation visualisations pulled from <a href="https://glycerine.io/">Glycerine</a>, a IIIF-compatible platform for scholarly annotation publishing.
 * Version: 1.0.4
 * Author: Systemik Solutions
 * Author URI: https://www.systemiksolutions.com/
 * Contributors: systemiksolutions, mufengniu
 * Tags: iiif, annotation, glycerine
 * License: GPLv2
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: glycerine
 */

//Register the admin page
function glycerine_embeds_add_admin_page()
{
    add_menu_page(
        'Glycerine',                         // Page title
        'Glycerine',                         // Menu title
        'manage_options',                    // Capability
        'glycerine_embeds',                  // Menu slug
        'glycerine_embeds_page',             // Function to render the page
        'dashicons-admin-plugins',           // Icon (optional)
        100                                  // Position
    );
}
add_action('admin_menu', 'glycerine_embeds_add_admin_page');
include_once(plugin_dir_path(__FILE__) . 'includes/admin-settings.php');


function glycerine_enqueue_assets()
{

    $force_enqueue = !empty($GLOBALS['glycerine_force_enqueue']);

    wp_enqueue_style(
        'glycerine-main-css',
        plugin_dir_url(__FILE__) . 'assets/css/main.css',
        array(),
        filemtime(plugin_dir_path(__FILE__) . 'assets/css/main.css')
    );


    // annotation table
    if ((is_singular() && (has_block('glycerine/table') || has_block('glycerine/annotation-cutouts'))
    ) || $force_enqueue) {
        wp_enqueue_script(
            'glycerine-annotation-cropper',
            plugin_dir_url(__FILE__) . 'assets/js/annotation-cropper.js',
            array('jquery'),
            filemtime(plugin_dir_path(__FILE__) . 'assets/js/annotation-cropper.js'),
            true
        );

        wp_enqueue_script(
            'glycerine-image-cropper',
            plugin_dir_url(__FILE__) . 'assets/js/image-cropper.js',
            array('jquery'),
            filemtime(plugin_dir_path(__FILE__) . 'assets/js/image-cropper.js'),
            true
        );

        wp_enqueue_script(
            'glycerine-image-loader',
            plugin_dir_url(__FILE__) . 'assets/js/image-loader.js',
            array('jquery'),
            filemtime(plugin_dir_path(__FILE__) . 'assets/js/image-loader.js'),
            true
        );
    }


    if ((is_singular() && has_block('glycerine/table')) || $force_enqueue) {
        //Choices.js
        wp_enqueue_style('choices', plugin_dir_url(__FILE__) . 'assets/css/choices.min.css', array(), null);
        wp_enqueue_script('choices', plugin_dir_url(__FILE__) . 'assets/js/choices.min.js', array(), null, true);

        wp_enqueue_script(
            'glycerine-annotation-table',
            plugin_dir_url(__FILE__) . 'assets/js/annotation-table.js',
            array('jquery'),
            filemtime(plugin_dir_path(__FILE__) . 'assets/js/annotation-table.js'),
            true
        );
    }

    if ((is_singular() && has_block('glycerine/annotation-cutouts')) || $force_enqueue) {

        wp_enqueue_script(
            'glycerine-annotation-cutouts',
            plugin_dir_url(__FILE__) . 'assets/js/annotation-cutouts.js',
            array('jquery'),
            filemtime(plugin_dir_path(__FILE__) . 'assets/js/annotation-cutouts.js'),
            true
        );
    }
}
add_action('wp_enqueue_scripts', 'glycerine_enqueue_assets');


function glycerine_pass_image_path()
{
    wp_localize_script(
        'wp-block-editor',
        'glycerineSettings',
        array(
            'loadingImageUrl' => plugin_dir_url(__FILE__) . 'assets/image/loading.gif',
        )
    );
}
add_action('enqueue_block_editor_assets', 'glycerine_pass_image_path');


// Register the blocks
function glycerine_register_table_block()
{
    wp_register_script(
        'glycerine-table-block-js',
        plugin_dir_url(__FILE__) . 'build/glycerine-table.js',
        array('wp-blocks', 'wp-editor', 'wp-components', 'wp-element', 'wp-i18n'),
        filemtime(plugin_dir_path(__FILE__) . 'build/glycerine-table.js')
    );

    // Register the block with a render callback
    register_block_type('glycerine/table', array(
        'editor_script' => 'glycerine-table-block-js',
        'render_callback' => 'glycerine_render_table_block',
    ));
}
add_action('init', 'glycerine_register_table_block');

function glycerine_register_image_set_block()
{
    wp_register_script(
        'glycerine-image-set-block-js',
        plugin_dir_url(__FILE__) . 'build/glycerine-image-set.js',
        array('wp-blocks', 'wp-editor', 'wp-components', 'wp-element', 'wp-i18n'),
        filemtime(plugin_dir_path(__FILE__) . 'build/glycerine-image-set.js')
    );

    register_block_type('glycerine/image-set', array(
        'editor_script' => 'glycerine-image-set-block-js',
        'render_callback' => 'glycerine_render_image_set_block',
    ));
}
add_action('init', 'glycerine_register_image_set_block');


function glycerine_register_annotation_cutouts_block()
{
    wp_register_script(
        'glycerine-annotation-cutouts-js',
        plugin_dir_url(__FILE__) . 'build/glycerine-annotation-cutouts.js',
        array('wp-blocks', 'wp-editor', 'wp-components', 'wp-element', 'wp-i18n'),
        filemtime(plugin_dir_path(__FILE__) . 'build/glycerine-annotation-cutouts.js')
    );

    register_block_type('glycerine/annotation-cutouts', array(
        'editor_script' => 'glycerine-annotation-cutouts-js',
        'render_callback' => 'glycerine_render_annotation_cutout_block',
    ));
}
add_action('init', 'glycerine_register_annotation_cutouts_block');


function glycerine_register_collection_block()
{
    wp_register_script(
        'glycerine-collection',
        plugin_dir_url(__FILE__) . 'build/glycerine-collection.js',
        array('wp-blocks', 'wp-editor', 'wp-components', 'wp-element', 'wp-i18n'),
        filemtime(plugin_dir_path(__FILE__) . 'build/glycerine-collection.js')
    );

    register_block_type('glycerine/collection', array(
        'editor_script' => 'glycerine-collection',
        'render_callback' => 'glycerine_render_collection',
    ));
}
add_action('init', 'glycerine_register_collection_block');


// Register the API endpoints
function glycerine_register_api_endpoints()
{
    register_rest_route('glycerine/v1', '/collections', array(
        'methods' => 'GET',
        'callback' => 'glycerine_get_collections',
        'permission_callback' => '__return_true', // Allow public access
    ));

    register_rest_route('glycerine/v1', '/collection/(?P<id>\d+)/image_sets', array(
        'methods' => 'GET',
        'callback' => 'glycerine_get_image_sets_from_collection',
        'permission_callback' => '__return_true',
    ));

    register_rest_route('glycerine/v1', '/annotations', array(
        'methods' => 'GET',
        'callback' => 'glycerine_get_annotations',
        'permission_callback' => '__return_true',
        'args' => array(
            'annotation_set_publish_id' => array(
                'required' => true,
            ),
        ),
    ));

    register_rest_route('glycerine/v1', '/collection/(?P<id>\d+)/publish', array(
        'methods' => 'GET',
        'callback' => 'glycerine_publish_collection',
        'permission_callback' => '__return_true',
    ));

    register_rest_route('glycerine/v1', '/collection/(?P<id>\d+)/unpublish', array(
        'methods' => 'GET',
        'callback' => 'glycerine_unpublish_collection',
        'permission_callback' => '__return_true',
    ));

    register_rest_route('glycerine/v1', '/proxy', [
        'methods' => 'GET',
        'callback' => 'glycerine_proxy_request',
        'permission_callback' => '__return_true',
    ]);
}
add_action('rest_api_init', 'glycerine_register_api_endpoints');


// Render callback functions
function glycerine_render_table_block($attributes)
{
    // Retrieve the selected collection IDs from block attributes
    $selected_collections = isset($attributes['selectedCollections']) ? $attributes['selectedCollections'] : [];

    $selected_columns = isset($attributes['selectedColumns']) ? $attributes['selectedColumns'] : [];
    if (empty($selected_columns)) {
        $selected_columns = ['Collection', 'Image Set', 'Annotation Set', 'Image', 'Title', 'Description', 'Tags'];
    }
    $paging = isset($attributes['paging']) && is_numeric($attributes['paging']) ? (int)$attributes['paging'] : 50;
    $imageQuality = isset($attributes['imageQuality']) && is_numeric($attributes['imageQuality']) ? (int)$attributes['imageQuality'] : 600;
    $collection_map = isset($attributes['collectionMap']) ? $attributes['collectionMap'] : [];

    $annotations = isset($attributes['annotations']) ? $attributes['annotations'] : [];
    // If no collections are selected, return a default message
    if (empty($selected_collections) && empty($annotations)) {
        return '<div class="glycerine-table"><p>No collections selected.</p></div>';
    }

    $columnLabelMap = isset($attributes['columnLabelMap']) ? $attributes['columnLabelMap'] : [];

    $filterColumns = isset($attributes['filterColumns']) ? $attributes['filterColumns'] : [
        'Collection',
        'Image Set',
        'Tags'
    ];

    $filterColumnLabelMap = isset($attributes['filterColumnLabelMap']) ? $attributes['filterColumnLabelMap'] : [
        'Collection' => 'Collection',
        'Image Set' => 'Image Set',
        'Tags' => 'Tags'
    ];


    $block_cache_id = json_encode([
        'collections' => $selected_collections,
        'columns' => $selected_columns,
        'paging' => $paging,
        'quality' => $imageQuality,
        'columnLabelMap' => $columnLabelMap,
    ]);
    $block_cache_id = 'annotation_table_' . substr(md5($block_cache_id), 0, 16);


    // Start output buffering
    ob_start();

    // Prevent rendering inside the block editor
    if (!is_singular() && !is_archive() && !is_home() && !is_front_page()) {
        return;
    }

    $template_path = plugin_dir_path(__FILE__) . 'includes/annotation-table.php';

    if (file_exists($template_path)) {
        include $template_path;
    } else {
        return '<div class="glycerine-table"><p>Template file not found.</p></div>';
    }

    return ob_get_clean();
}

function glycerine_render_image_set_block($attributes)
{
    if (!isset($attributes['selectedImageSet']) || empty($attributes['selectedImageSet']['public_url'])) {
        return ''; // Return nothing if no image set is selected
    }

    $public_url = esc_url($attributes['selectedImageSet']['public_url']);
    $urlParameters = isset($attributes['urlParameters']) ? $attributes['urlParameters'] : 'fullscreenbutton=1&indexbutton=1&annotationviewbutton=1&aboutpanebutton=1&settingpanebutton=1&infopanel=1';

    return '<div class="glycerine-image-set"><iframe src="' . $public_url . '?' . $urlParameters  . '" style="width:100%; height:600px; border: 2px solid black;"></iframe></div>';
}

function glycerine_render_collection($attributes)
{
    if (!isset($attributes['selectedCollection']) || empty($attributes['selectedCollection']['publication_id'])) {
        return '';
    }

    $public_url = 'https://iaw.ardc-hdcl-sia-iaw.cloud.edu.au/publications/collections/' .  $attributes['selectedCollection']['publication_id'];
    $urlParameters = isset($attributes['urlParameters']) ? $attributes['urlParameters'] : 'fullscreenbutton=1&indexbutton=1&annotationviewbutton=1&aboutpanebutton=1&settingpanebutton=1&infopanel=1';

    return '<div class="glycerine-collection"><iframe src="' . $public_url . '?' . $urlParameters  . '" style="width:100%; height:600px; border: 2px solid black;"></iframe></div>';
}

function glycerine_render_annotation_cutout_block($attributes)
{

    $annotationData = $attributes['annotationData'];
    $selectedImageSet = $attributes['selectedImageSet'];
    $selectedFields =  isset($attributes['selectedFields']) ? $attributes['selectedFields'] : ['title'];
    if (!isset($annotationData)) {
        return '';
    }

    $annotationTitle = null;
    $annotationDescription = null;
    $annotationDataFields = isset($annotationData['fields']) ? $annotationData['fields'] : [];
    foreach ($annotationDataFields as $field) {
        if ($field['field']['name'] == 'Title' && isset($field['values']['en'])) {
            $annotationTitle = $field['values']['en'][0];
        }

        if ($field['field']['name'] == 'Description' && isset($field['values']['en'])) {
            $annotationDescription = $field['values']['en'][0];
        }
    }

    // Prevent rendering inside the block editor
    if (!is_singular() && !is_archive() && !is_home() && !is_front_page()) {
        return;
    }

    // Start output buffering
    ob_start();

    $template_path = plugin_dir_path(__FILE__) . 'includes/annotation-cutout.php';

    if (file_exists($template_path)) {
        include $template_path;
    }
    return ob_get_clean();
}


function glycerine_get_collections()
{
    $api_key = get_option('glycerine_api_key');
    $base_url = get_option('glycerine_iaw_base_url');

    $GlycerineIawClient = new GlycerineIAWClient($base_url, $api_key);
    $allCollections = $GlycerineIawClient->getAllCollections();

    return rest_ensure_response($allCollections);
}

function glycerine_get_annotations($request)
{

    $annotation_set_publish_id = $request->get_param('annotation_set_publish_id');

    if (!$annotation_set_publish_id) {
        return null;
    }

    $api_key = get_option('glycerine_api_key');
    $base_url = get_option('glycerine_iaw_base_url');

    $GlycerineIawClient = new GlycerineIAWClient($base_url, $api_key);
    $annotation = $GlycerineIawClient->getAnnotationsFromAnnotationSet($annotation_set_publish_id, false);

    return rest_ensure_response($annotation);
}

function glycerine_get_image_sets_from_collection($request)
{

    $collection_id = $request->get_param('id'); // Extract ID from request
    if (!$collection_id) {
        return [];
    }

    $api_key = get_option('glycerine_api_key');
    $base_url = get_option('glycerine_iaw_base_url');

    $GlycerineIawClient = new GlycerineIAWClient($base_url, $api_key);
    $image_sets = $GlycerineIawClient->searchWithCollection($collection_id, 'image_set', []);

    $res = [];

    foreach ($image_sets as $image_set) {
        if (isset($image_set['publication']) && $image_set['publication']['active'] &&  isset($image_set['publication']['persistent_url'])) {

            $res[] = [
                'id' => $image_set['id'],
                'name' => $image_set['name'],
                'thumbnail' => $image_set['thumbnail'],
                'persistent_url' => $image_set['publication']['persistent_url'],
                'public_url' => str_replace('https://w3id.org/iaw/data/', 'https://w3id.org/iaw/', $image_set['publication']['persistent_url'])
            ];
        }
    }

    return $res;
}

function glycerine_publish_collection($request)
{

    $collection_id = $request->get_param('id');
    if (!$collection_id) {
        wp_send_json_error([
            'error' => true,
            'message' => __('Collection ID is required.', 'glycerine')
        ], 400);
    }

    $api_key = get_option('glycerine_api_key');
    $base_url = get_option('glycerine_iaw_base_url');

    $GlycerineIawClient = new GlycerineIAWClient($base_url, $api_key);
    $response = json_decode($GlycerineIawClient->publishCollection($collection_id));

    if (!isset($response) || !isset($response->id)) {
        wp_send_json_error([
            'error' => true,
            'message' => __('Error publish collection', 'glycerine')
        ], 400);
    }

    return [
        'id' => $response->id,
        'name' => $response->name,
        'description' => $response->description,
        'is_published' => isset($response->publication) && $response->publication->active,
        'publication_id' => isset($response->publication) ? $response->publication->id : null
    ];
}

function glycerine_unpublish_collection($request)
{

    $collection_id = $request->get_param('id');
    if (!$collection_id) {
        wp_send_json_error([
            'error' => true,
            'message' => __('Collection ID is required.', 'glycerine')
        ], 400);
    }

    $api_key = get_option('glycerine_api_key');
    $base_url = get_option('glycerine_iaw_base_url');

    $GlycerineIawClient = new GlycerineIAWClient($base_url, $api_key);
    $response = json_decode($GlycerineIawClient->unpublishCollection($collection_id));

    if (!isset($response) || !isset($response->id)) {
        wp_send_json_error([
            'error' => true,
            'message' => __('Error unpublish collection', 'glycerine')
        ], 400);
    }

    return [
        'id' => $response->id,
        'name' => $response->name,
        'description' => $response->description,
        'is_published' => isset($response->publication) && $response->publication->active,
        'publication_id' => isset($response->publication) ? $response->publication->id : null
    ];
}


// Site Ajax 
function glycerine_get_image_set()
{
    if (!isset($_POST['image_set_id']) || !isset($_POST['collection_id'])) {
        wp_send_json_error([
            'error' => true,
            'message' => __('Image set ID is required.', 'glycerine')
        ], 400);
    }

    $api_key = get_option('glycerine_api_key');
    $base_url = get_option('glycerine_iaw_base_url');

    $GlycerineIawClient = new GlycerineIAWClient($base_url, $api_key);
    $collection_id = isset($_POST['collection_id']) ? absint(wp_unslash($_POST['collection_id'])) : 0;
    $image_set_id  = isset($_POST['image_set_id'])  ? absint(wp_unslash($_POST['image_set_id']))  : 0;

    $image_set = $GlycerineIawClient->getImageSet($collection_id, $image_set_id);

    if (!isset($image_set) || !isset($image_set['id'])) {
        wp_send_json_error([
            'error' => true,
            'message' => __('Error getting image set', 'glycerine')
        ], 400);
    }

    wp_send_json_success($image_set);
}
add_action('wp_ajax_glycerine_get_image_set', 'glycerine_get_image_set');
add_action('wp_ajax_nopriv_glycerine_get_image_set', 'glycerine_get_image_set');


// Proxy request for get image IIIF data
function glycerine_proxy_request($request)
{
    $api_key = get_option('glycerine_api_key');

    $target_url = $request->get_param('target');
    if (empty($target_url)) {
        return new WP_Error('missing_target', 'Missing target parameter', ['status' => 400]);
    }


    $response = wp_remote_get($target_url, [
        'timeout' => 30,
        'headers' => [
            'Authorization' => 'Bearer ' . $api_key,
            'Accept' => 'application/json',
        ],
    ]);

    if (is_wp_error($response)) {
        return new WP_Error('proxy_failed', 'Proxy request failed', ['status' => 500]);
    }

    $body = wp_remote_retrieve_body($response);

    return rest_ensure_response(json_decode($body));
}