document.addEventListener("DOMContentLoaded", async function () {
    document.querySelectorAll(".glycerine-table").forEach((tableContainer) => {
        let currentPage = 1;

        let paging = parseInt(
            tableContainer.querySelector("#annotation-table-paging").value,
            10
        );

        const allRows = Array.from(
            tableContainer.querySelectorAll("#annotation-table-body tr")
        );

        let totalPages = Math.ceil(allRows.length / paging);

        const table = tableContainer.querySelector("table");
        const searchInput = tableContainer.querySelector("#annotation-search");
        const filterDropdowns =
            tableContainer.querySelectorAll(".column-filter");

        const prevButton = tableContainer.querySelector("#prev-page");
        const nextButton = tableContainer.querySelector("#next-page");
        const pageButtons = Array.from(
            tableContainer.querySelectorAll(".page-number")
        );

        const columnHeaders = Array.from(
            table.querySelector("thead").rows[0].cells
        ).map((cell) => cell.textContent.trim());

        filterDropdowns.forEach((select) => {
            new Choices(select, {
                removeItemButton: true,
                allowHTML: false,
                itemSelectText: "",
            });
        });

        // 🔄 Image loading queue and concurrency pool
        const maxConcurrent = 30;
        const taskQueue = [];
        const seenRows = new Set();
        let activeWorkers = 0;
        const width = parseInt(
            tableContainer.querySelector("#annotation-image-width").value,
            10
        );

        async function processRow(row) {
            if (!row || seenRows.has(row)) return;
            seenRows.add(row);

            const imageElement = row.querySelector(".annotation-img");
            const hiddenInput = row.querySelector(".annotation-data");

            if (
                !imageElement ||
                !hiddenInput ||
                imageElement.dataset.loaded === "true"
            )
                return;

            try {
                const annotation = JSON.parse(hiddenInput.value);
                const imageLoader = new ImageLoader(annotation.source, width);
                await imageLoader.load();

                if (imageLoader.hasLoaded()) {
                    const src = AnnotationCropper.cropAnnotationImage(
                        annotation,
                        imageLoader
                    );
                    imageElement.src = src;
                    imageElement.dataset.loaded = "true";

                    //remove class "loading" from imageElement
                    imageElement.classList.remove("loading");
                }
            } catch (e) {
                console.error("Error loading image", e);
            }
        }

        async function runWorkerLoop() {
            if (activeWorkers >= maxConcurrent) return;
            activeWorkers++;

            while (taskQueue.length > 0) {
                const row = taskQueue.shift();
                await processRow(row);
            }

            activeWorkers--;
        }

        function scheduleVisibleRows() {
            const allTableRows = Array.from(
                tableContainer.querySelectorAll("#annotation-table-body tr")
            );
            const visibleRows = allTableRows.filter(
                (row) => row.style.display !== "none"
            );
            const hiddenRows = allTableRows.filter(
                (row) => row.style.display === "none"
            );

            // Prioritize visible rows
            [...visibleRows].reverse().forEach((row) => {
                if (!seenRows.has(row)) {
                    taskQueue.unshift(row); // 👈 Now first visible row ends up first in queue
                }
            });

            // Load hidden ones eventually
            hiddenRows.forEach((row) => {
                if (!seenRows.has(row)) {
                    taskQueue.push(row); // Low priority
                }
            });

            runWorkerLoop(); // Start worker if not already running
        }

        function updateTable() {
            const searchQuery = searchInput.value.toLowerCase();
            let activeFilters = {};

            filterDropdowns.forEach((select) => {
                const column = select.getAttribute("data-column");
                const selectedValues = Array.from(select.selectedOptions).map(
                    (option) => option.value
                );
                if (selectedValues.length > 0 && selectedValues[0] !== "") {
                    activeFilters[column] = selectedValues;
                }
            });

            let visibleRows = [];
            allRows.forEach((row) => {
                let textMatch = false;
                let filterMatch = true;
                const cells = row.querySelectorAll("td");

                cells.forEach((cell, index) => {
                    if (index >= columnHeaders.length) return;
                    const columnName = columnHeaders[index];
                    const cellText = cell.textContent.trim();

                    if (
                        searchQuery &&
                        cellText.toLowerCase().includes(searchQuery)
                    ) {
                        textMatch = true;
                    }

                    if (activeFilters[columnName]) {
                        const selectedValues = activeFilters[columnName];

                        if (columnName === "Tags") {
                            const tagElements =
                                cell.querySelectorAll(".annotation-tag");
                            const tagValues = Array.from(tagElements).map(
                                (tag) => tag.textContent.trim()
                            );

                            if (
                                !selectedValues.some((tag) =>
                                    tagValues.includes(tag)
                                )
                            ) {
                                filterMatch = false;
                            }
                        } else {
                            if (!selectedValues.includes(cellText)) {
                                filterMatch = false;
                            }
                        }
                    }
                });

                const shouldShow =
                    (textMatch || searchQuery === "") && filterMatch;
                row.style.display = shouldShow ? "" : "none";
                if (shouldShow) visibleRows.push(row);
            });

            updatePagination(visibleRows);
            scheduleVisibleRows();
        }

        function updatePagination(visibleRows) {
            totalPages = Math.ceil(visibleRows.length / paging);
            currentPage = 1;

            pageButtons.forEach((button, index) => {
                button.style.display =
                    index < totalPages ? "inline-block" : "none";
                button.disabled = index === 0;
                button.dataset.page = index + 1;
            });

            prevButton.disabled = true;
            nextButton.disabled = totalPages <= 1;

            visibleRows.forEach((row, index) => {
                row.style.display =
                    Math.floor(index / paging) === 0 ? "" : "none";
                row.dataset.page = Math.floor(index / paging) + 1;
            });

            updatePageButtons();
        }

        function updatePageButtons() {
            pageButtons.forEach((button) => {
                button.disabled = parseInt(button.dataset.page) === currentPage;
            });

            prevButton.disabled = currentPage === 1;
            nextButton.disabled = currentPage === totalPages;
        }

        function showPage(page) {
            currentPage = page;
            allRows.forEach((row) => {
                row.style.display =
                    parseInt(row.dataset.page) === currentPage ? "" : "none";
            });

            updatePageButtons();
            tableContainer.scrollIntoView({
                behavior: "smooth",
                block: "start",
            });
            scheduleVisibleRows(); // 🟢 queue rows shown on this page
        }

        prevButton.addEventListener("click", () => {
            if (currentPage > 1) {
                currentPage--;
                showPage(currentPage);
            }
        });

        nextButton.addEventListener("click", () => {
            if (currentPage < totalPages) {
                currentPage++;
                showPage(currentPage);
            }
        });

        pageButtons.forEach((button) => {
            button.addEventListener("click", function () {
                currentPage = parseInt(this.dataset.page);
                showPage(currentPage);
            });
        });

        searchInput.addEventListener("keypress", function (event) {
            if (event.key === "Enter") {
                event.preventDefault();
                updateTable();
            }
        });

        filterDropdowns.forEach((select) => {
            select.addEventListener("change", updateTable);
        });

        updateTable(); // Initial render
    });
});

// Click to open image set
jQuery(document).ready(function ($) {
    const block_id = document.getElementById("annotation-table-block-id").value;
    const container = document.getElementById(block_id);
    const ajaxurl = document.getElementById("admin-url").value;
    const loadingWheel = document.getElementById(block_id + "_loadingWheel");

    function showLoadingWheel() {
        loadingWheel.style.display = "block";
    }

    function hideLoadingWheel() {
        loadingWheel.style.display = "none";
    }

    if (container) {
        const images = container.querySelectorAll(".annotation-img");

        images.forEach((img) => {
            img.addEventListener("click", function () {
                const imageSetId = this.getAttribute("data-image-set-id");
                const collectionId = this.getAttribute("data-collection-id");
                if (!imageSetId || !collectionId) return;

                showLoadingWheel();

                $.post(
                    ajaxurl,
                    {
                        action: "glycerine_get_image_set",
                        image_set_id: imageSetId,
                        collection_id: collectionId,
                    },
                    function (response) {
                        hideLoadingWheel();

                        if (response.success) {
                            const imageSet = response.data;
                            const url = imageSet?.publication?.persistent_url;

                            if (imageSet?.publication?.active && url) {
                                const publicationUrl = url.replace(
                                    "https://w3id.org/iaw/data/",
                                    "https://w3id.org/iaw/"
                                );
                                window.open(publicationUrl, "_blank");
                            } else {
                                alert("No images found in the image set.");
                            }
                        } else {
                            alert(
                                response.data.message || "An error occurred."
                            );
                        }
                    }
                );
            });
        });
    }
});
