class ImageLoader {
    iifSize;
    loaded;
    imageElement;
    iiif_url;
    ratio;

    constructor(iiif_url, width = null) {
        let size = "max";
        if (width) {
            size = `${width},`;
        }
        this.iifSize = size;

        // Force https if http is used
        if (iiif_url.startsWith("http:")) {
            iiif_url = iiif_url.replace(/^http:/, "https:");
        }

        this.iiif_url = iiif_url;
        this.loaded = false;
    }

    async load() {
        if (this.loaded) {
            return;
        }

        let imageURL, info;

        if (this.iiif_url.startsWith("https://heidicon")) {
            imageURL = `${this.iiif_url}/full/${this.iifSize}/0/default.jpg`;

            info = await fetch(`${this.iiif_url}/info.json`).then((response) =>
                response.json()
            );
        } else if (this.iiif_url.startsWith("https://iaw-image-server")) {
            imageURL = `${this.iiif_url}/full/${this.iifSize}/0/default.jpg`;
            info = await fetch(`${this.iiif_url}`).then((response) =>
                response.json()
            );
        } else {
            info = await fetch(
                `/wp-json/glycerine/v1/proxy?target=${this.iiif_url}`
            )
                .then((response) => {
                    if (!response.ok)
                        throw new Error("Network response was not ok");
                    return response.json();
                })
                .catch((error) => {
                    console.error("Fetch error:", error);
                });

            imageURL =
                info["items"][0]["items"][0]["body"]["service"]["0"]["id"] +
                "/full/" +
                this.iifSize +
                "/0/default.jpg";
        }

        const imageWidth = info.width;

        this.imageElement = new Image();
        this.imageElement.crossOrigin = "anonymous";
        this.imageElement.src = imageURL;
        await this.imageElement.decode();

        this.ratio = this.imageElement.width / imageWidth;
        this.loaded = true;

        return;
    }

    getImageElement() {
        return this.imageElement;
    }

    getRatio() {
        return this.ratio;
    }

    hasLoaded() {
        return this.loaded;
    }
}

class ImageCropper {
    /**
     * Constructor
     *
     * @param {HTMLElement} image
     *   The DOM element of the source image.
     */
    constructor(image) {
        this.image = image;
        this.canvas = document.createElement("canvas");
        this.canvas.width = this.image.width;
        this.canvas.height = this.image.height;
        this.canvasContext = this.canvas.getContext("2d");
    }

    /**
     * Crop a polygon area from the image.
     *
     * @param {Array} points
     *   Two dimension array with each element contains the x and y coordinates.
     * @returns {string}
     *   The base64 image data of the cropped image.
     */
    crop(points) {
        const cropAreaCords = this._calculateCropRectCoordinates(points);
        const cropAreaWidth = cropAreaCords[1][0] - cropAreaCords[0][0];
        const cropAreaHeight = cropAreaCords[1][1] - cropAreaCords[0][1];
        this.canvasContext.save();
        this.resetCanvas();
        this.canvasContext.beginPath();
        this.canvasContext.moveTo(points[0][0], points[0][1]);
        for (let i = 1; i < points.length; i++) {
            this.canvasContext.lineTo(points[i][0], points[i][1]);
        }
        this.canvasContext.closePath();
        this.canvasContext.clip();
        this.canvasContext.drawImage(this.image, 0, 0);
        this.canvasContext.restore();
        return this._createCropImage(
            cropAreaCords[0],
            cropAreaWidth,
            cropAreaHeight
        );
    }

    /**
     * Crop a circle area from the image.
     *
     * @param {number[]} center
     *   The x and y coordinates of the circle center.
     * @param {number} radius
     *   The circle radius.
     * @returns {string}
     *   The base64 image data of the cropped image.
     */
    cropCircle(center, radius) {
        const areaStartPoint = [center[0] - radius, center[1] - radius];
        this.canvasContext.save();
        this.resetCanvas();
        this.canvasContext.beginPath();
        this.canvasContext.arc(
            center[0],
            center[1],
            radius,
            0,
            2 * Math.PI,
            false
        );
        this.canvasContext.clip();
        this.canvasContext.drawImage(this.image, 0, 0);
        this.canvasContext.restore();
        return this._createCropImage(areaStartPoint, radius * 2, radius * 2);
    }

    /**
     * Crop an ellipse area from the image.
     *
     * @param {number[]} center
     *   The x and y coordinates of the ellipse center.
     * @param {number} radiusX
     *   The ellipse radius on the x axis.
     * @param {number} radiusY
     *   The ellipse radius on the y axis.
     * @returns {string}
     *   The base64 image data of the cropped image.
     */
    cropEllipse(center, radiusX, radiusY) {
        const areaStartPoint = [center[0] - radiusX, center[1] - radiusY];
        this.canvasContext.save();
        this.resetCanvas();
        this.canvasContext.beginPath();
        this.canvasContext.ellipse(
            center[0],
            center[1],
            radiusX,
            radiusY,
            0,
            0,
            2 * Math.PI,
            false
        );
        this.canvasContext.clip();
        this.canvasContext.drawImage(this.image, 0, 0);
        this.canvasContext.restore();
        return this._createCropImage(areaStartPoint, radiusX * 2, radiusY * 2);
    }

    /**
     * Clear and reset the canvas.
     */
    resetCanvas() {
        this.canvasContext.clearRect(0, 0, this.image.width, this.image.height);
    }

    /**
     * Create the crop image.
     *
     * @param {number[]} startCord
     *   The coordinates of the crop area start point.
     * @param {number} width
     *   The crop area width.
     * @param {number} height
     *   The crop area height.
     * @returns {string}
     *   The base64 image data of the cropped image.
     * @private
     */
    _createCropImage(startCord, width, height) {
        const transCanvas = document.createElement("canvas");
        const transContext = transCanvas.getContext("2d");
        transCanvas.width = width;
        transCanvas.height = height;
        transContext.drawImage(
            this.canvas,
            startCord[0],
            startCord[1],
            width,
            height,
            0,
            0,
            width,
            height
        );
        const cropImageURL = transCanvas.toDataURL();
        transCanvas.remove();
        return cropImageURL;
    }

    /**
     * Calculate the corner point of the rectangle crop area.
     *
     * @param {Array} points
     *   Two dimension array with each element contains the x and y coordinates.
     * @returns {Array}
     *   The array contains the x and y of the top left corner and the bottom right corner.
     * @private
     */
    _calculateCropRectCoordinates(points) {
        let minX = null;
        let minY = null;
        let maxX = null;
        let maxY = null;
        for (let i = 1; i < points.length; i++) {
            let x = points[i][0];
            let y = points[i][1];
            if (minX === null || x < minX) {
                minX = x;
            }
            if (minY === null || y < minY) {
                minY = y;
            }
            if (maxX === null || x > maxX) {
                maxX = x;
            }
            if (maxY === null || y > maxY) {
                maxY = y;
            }
        }
        return [
            [minX, minY],
            [maxX, maxY],
        ];
    }
}

function cropAnnotationImage(annotation, imageLoader) {
    if (!imageLoader.hasLoaded()) {
        return null;
    }
    const ratio = imageLoader.getRatio();
    const cropper = new ImageCropper(imageLoader.getImageElement());

    const selector = annotation.selector;

    if (selector.type === "SvgSelector") {
        if (selector.value.match(/<polygon points="([^"]+)"/)) {
            const matches = [
                ...selector.value.matchAll(/<polygon points="([^"]+)"/g),
            ];
            const value = matches[0][1];
            const pairs = value.split(" ");
            const points = [];
            pairs.forEach((pair) => {
                const coords = pair.split(",");
                const x = parseFloat(coords[0]) * ratio;
                const y = parseFloat(coords[1]) * ratio;
                points.push([x, y]);
            });
            return cropper.crop(points);
        } else if (
            selector.value.match(
                /<circle cx="([^"]+)" cy="([^"]+)" r="([^"]+)"/
            )
        ) {
            const matches = [
                ...selector.value.matchAll(
                    /<circle cx="([^"]+)" cy="([^"]+)" r="([^"]+)"/g
                ),
            ];
            const cx = parseFloat(matches[0][1]) * ratio;
            const cy = parseFloat(matches[0][2]) * ratio;
            const r = parseFloat(matches[0][3]) * ratio;
            return cropper.cropCircle([cx, cy], r);
        } else if (
            selector.value.match(
                /<ellipse cx="([^"]+)" cy="([^"]+)" rx="([^"]+)" ry="([^"]+)"/
            )
        ) {
            const matches = [
                ...selector.value.matchAll(
                    /<ellipse cx="([^"]+)" cy="([^"]+)" rx="([^"]+)" ry="([^"]+)"/g
                ),
            ];
            const cx = parseFloat(matches[0][1]) * ratio;
            const cy = parseFloat(matches[0][2]) * ratio;
            const rx = parseFloat(matches[0][3]) * ratio;
            const ry = parseFloat(matches[0][4]) * ratio;
            return cropper.cropEllipse([cx, cy], rx, ry);
        }
    } else if (selector.type === "FragmentSelector") {
        if (selector.value.match(/^xywh=pixel:([0-9\-.,]+)$/)) {
            const matches = [
                ...selector.value.matchAll(/^xywh=pixel:([0-9\-.,]+)$/g),
            ];
            const values = matches[0][1].split(",");
            const x = parseFloat(values[0]) * ratio;
            const y = parseFloat(values[1]) * ratio;
            const w = parseFloat(values[2]) * ratio;
            const h = parseFloat(values[3]) * ratio;
            const points = [
                [x, y],
                [x + w, y],
                [x + w, y + h],
                [x, y + h],
            ];
            return cropper.crop(points);
        }
    }
    return null;
}


export async function getAnnotationUrl(annotation) {
    let imageLoader = new ImageLoader(annotation.source);
    await imageLoader.load();

    if (imageLoader && imageLoader.hasLoaded()) {
        var src = cropAnnotationImage(annotation, imageLoader);
        return src;
    }
}
