<?php
/**
 * Helper Functions
 *
 * @package Glimbyte_SSL_Manager
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Get plugin option with default value.
 *
 * @param string $key Option key.
 * @param mixed  $default Default value.
 * @return mixed Option value.
 */
function glimbyte_ssl_get_option( $key, $default = '' ) {
	return get_option( 'glimbyte_ssl_' . $key, $default );
}

/**
 * Update plugin option.
 *
 * @param string $key Option key.
 * @param mixed  $value Option value.
 * @return bool True if updated, false otherwise.
 */
function glimbyte_ssl_update_option( $key, $value ) {
	return update_option( 'glimbyte_ssl_' . $key, $value );
}

/**
 * Delete plugin option.
 *
 * @param string $key Option key.
 * @return bool True if deleted, false otherwise.
 */
function glimbyte_ssl_delete_option( $key ) {
	return delete_option( 'glimbyte_ssl_' . $key );
}

/**
 * Check if current request is HTTPS.
 *
 * @return bool True if HTTPS, false otherwise.
 */
function glimbyte_ssl_is_https() {
	// Check standard HTTPS.
	if ( isset( $_SERVER['HTTPS'] ) && 'on' === strtolower( sanitize_text_field( wp_unslash( $_SERVER['HTTPS'] ) ) ) ) {
		return true;
	}

	// Check if port 443 (standard HTTPS port).
	if ( isset( $_SERVER['SERVER_PORT'] ) && '443' === $_SERVER['SERVER_PORT'] ) {
		return true;
	}

	// Check for reverse proxy/load balancer.
	if ( isset( $_SERVER['HTTP_X_FORWARDED_PROTO'] ) && 'https' === strtolower( sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_FORWARDED_PROTO'] ) ) ) ) {
		return true;
	}

	// Check CloudFlare.
	if ( isset( $_SERVER['HTTP_CF_VISITOR'] ) ) {
		$cf_visitor = json_decode( sanitize_text_field( wp_unslash( $_SERVER['HTTP_CF_VISITOR'] ) ), true );
		if ( isset( $cf_visitor['scheme'] ) && 'https' === $cf_visitor['scheme'] ) {
			return true;
		}
	}

	return false;
}

/**
 * Get current URL.
 *
 * @return string Current URL.
 */
function glimbyte_ssl_get_current_url() {
	$protocol = glimbyte_ssl_is_https() ? 'https://' : 'http://';
	$host     = isset( $_SERVER['HTTP_HOST'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_HOST'] ) ) : '';
	$uri      = isset( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';

	return $protocol . $host . $uri;
}

/**
 * Check if URL should be excluded from HTTPS redirect.
 *
 * @param string $url URL to check.
 * @return bool True if excluded, false otherwise.
 */
function glimbyte_ssl_is_url_excluded( $url ) {
	$excluded_urls = glimbyte_ssl_get_option( 'excluded_urls', '' );

	if ( empty( $excluded_urls ) ) {
		return false;
	}

	// Parse excluded URLs (one per line).
	$excluded_list = array_filter( array_map( 'trim', explode( "\n", $excluded_urls ) ) );

	foreach ( $excluded_list as $pattern ) {
		// Support wildcards.
		$pattern = str_replace( '*', '.*', preg_quote( $pattern, '/' ) );
		if ( preg_match( '/' . $pattern . '/i', $url ) ) {
			return true;
		}
	}

	return false;
}

/**
 * Log debug message.
 *
 * @param string $message Log message.
 * @param string $type Log type (info, warning, error).
 */
function glimbyte_ssl_log( $message, $type = 'info' ) {
	if ( ! glimbyte_ssl_get_option( 'debug_mode', false ) ) {
		return;
	}

	Glimbyte_SSL_Logger::log( $message, $type );
}

/**
 * Format date for display.
 *
 * @param string|int $date Date string or timestamp.
 * @return string Formatted date.
 */
function glimbyte_ssl_format_date( $date ) {
	if ( is_numeric( $date ) ) {
		$timestamp = $date;
	} else {
		$timestamp = strtotime( $date );
	}

	return date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $timestamp );
}

/**
 * Get admin page URL.
 *
 * @param string $tab Tab name.
 * @return string Admin page URL.
 */
function glimbyte_ssl_get_admin_url( $tab = '' ) {
	$url = admin_url( 'admin.php?page=glimbyte-ssl-guardian' );

	if ( ! empty( $tab ) ) {
		$url .= '&tab=' . $tab;
	}

	return $url;
}

/**
 * Display admin notice.
 *
 * @param string $message Notice message.
 * @param string $type Notice type (success, error, warning, info).
 */
function glimbyte_ssl_add_admin_notice( $message, $type = 'success' ) {
	set_transient( 'glimbyte_ssl_admin_notice', array(
		'message' => $message,
		'type'    => $type,
	), 30 );
}

/**
 * Sanitize checkbox input.
 *
 * @param mixed $value Input value.
 * @return bool Sanitized value.
 */
function glimbyte_ssl_sanitize_checkbox( $value ) {
	return ( isset( $value ) && '1' === $value ) ? true : false;
}

/**
 * Get SSL certificate info.
 *
 * @param string $domain Domain to check.
 * @return array|WP_Error Certificate info or error.
 */
function glimbyte_ssl_get_cert_info( $domain = '' ) {
	if ( empty( $domain ) ) {
		$parsed_url = wp_parse_url( home_url() );
		$domain = isset( $parsed_url['host'] ) ? $parsed_url['host'] : '';
	}

	// Check transient cache first.
	$cache_key = 'glimbyte_ssl_cert_info_' . md5( $domain );
	$cert_info = get_transient( $cache_key );

	if ( false !== $cert_info ) {
		return $cert_info;
	}

	// Fetch certificate info.
	$cert_info = Glimbyte_SSL_Checker::get_certificate_info( $domain );

	// Cache for 24 hours.
	if ( ! is_wp_error( $cert_info ) ) {
		set_transient( $cache_key, $cert_info, DAY_IN_SECONDS );
	}

	return $cert_info;
}

/**
 * Clear certificate cache.
 */
function glimbyte_ssl_clear_cert_cache() {
	$parsed_url = wp_parse_url( home_url() );
	$domain     = isset( $parsed_url['host'] ) ? $parsed_url['host'] : '';
	$cache_key  = 'glimbyte_ssl_cert_info_' . md5( $domain );
	delete_transient( $cache_key );
}

/**
 * Check if site is localhost.
 *
 * @return bool True if localhost, false otherwise.
 */
function glimbyte_ssl_is_localhost() {
	$host = isset( $_SERVER['HTTP_HOST'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_HOST'] ) ) : '';

	$localhost_patterns = array( 'localhost', '127.0.0.1', '::1', '.local', '.test', '.dev' );

	foreach ( $localhost_patterns as $pattern ) {
		if ( false !== stripos( $host, $pattern ) ) {
			return true;
		}
	}

	return false;
}
