<?php
/**
 * SSL Certificate Checker
 *
 * @package Glimbyte_SSL_Manager
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * SSL Checker class.
 */
class Glimbyte_SSL_Checker {

	/**
	 * Class instance.
	 *
	 * @var Glimbyte_SSL_Checker
	 */
	private static $instance = null;

	/**
	 * Get class instance.
	 *
	 * @return Glimbyte_SSL_Checker
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor.
	 */
	private function __construct() {
		add_action( 'glimbyte_ssl_check_certificate', array( $this, 'check_and_notify' ) );
	}

	/**
	 * Get SSL certificate information for a domain.
	 *
	 * @param string $domain Domain to check.
	 * @return array|WP_Error Certificate information or error.
	 */
	public static function get_certificate_info( $domain ) {
		// Remove protocol if present.
		$domain = preg_replace( '/^https?:\/\//', '', $domain );

		// Remove path if present.
		$domain = preg_replace( '/\/.*$/', '', $domain );

		$context = stream_context_create( array(
			'ssl' => array(
				'capture_peer_cert' => true,
				'verify_peer'       => false,
				'verify_peer_name'  => false,
			),
		) );

		// Attempt to connect.
		$stream = @stream_socket_client(
			'ssl://' . $domain . ':443',
			$errno,
			$errstr,
			30,
			STREAM_CLIENT_CONNECT,
			$context
		);

		if ( false === $stream ) {
			return new WP_Error(
				'connection_failed',
				sprintf(
					/* translators: %s: Error message */
					__( 'Failed to connect: %s', 'glimbyte-ssl-guardian' ),
					$errstr
				)
			);
		}

		$params = stream_context_get_params( $stream );

		if ( ! isset( $params['options']['ssl']['peer_certificate'] ) ) {
			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- Required for socket streams
			fclose( $stream );
			return new WP_Error( 'no_certificate', __( 'No certificate found.', 'glimbyte-ssl-guardian' ) );
		}

		$cert = $params['options']['ssl']['peer_certificate'];
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- Required for socket streams
		fclose( $stream );

		// Parse certificate.
		$cert_data = openssl_x509_parse( $cert );

		if ( false === $cert_data ) {
			return new WP_Error( 'parse_failed', __( 'Failed to parse certificate.', 'glimbyte-ssl-guardian' ) );
		}

		// Extract relevant information.
		$info = array(
			'issuer'       => isset( $cert_data['issuer']['CN'] ) ? $cert_data['issuer']['CN'] : __( 'Unknown', 'glimbyte-ssl-guardian' ),
			'subject'      => isset( $cert_data['subject']['CN'] ) ? $cert_data['subject']['CN'] : __( 'Unknown', 'glimbyte-ssl-guardian' ),
			'valid_from'   => $cert_data['validFrom_time_t'],
			'valid_to'     => $cert_data['validTo_time_t'],
			'days_left'    => self::calculate_days_until_expiry( $cert_data['validTo_time_t'] ),
			'status'       => self::get_certificate_status( $cert_data['validTo_time_t'] ),
			'is_valid'     => self::is_certificate_valid( $cert_data['validTo_time_t'] ),
			'signature'    => isset( $cert_data['signatureTypeSN'] ) ? $cert_data['signatureTypeSN'] : __( 'Unknown', 'glimbyte-ssl-guardian' ),
			'san'          => self::extract_san( $cert_data ),
		);

		return $info;
	}

	/**
	 * Calculate days until certificate expires.
	 *
	 * @param int $expiry_timestamp Expiry timestamp.
	 * @return int Days until expiry.
	 */
	private static function calculate_days_until_expiry( $expiry_timestamp ) {
		$now = time();
		$diff = $expiry_timestamp - $now;
		return max( 0, floor( $diff / DAY_IN_SECONDS ) );
	}

	/**
	 * Get certificate status.
	 *
	 * @param int $expiry_timestamp Expiry timestamp.
	 * @return string Status (valid, expiring_soon, expired).
	 */
	private static function get_certificate_status( $expiry_timestamp ) {
		$days_left = self::calculate_days_until_expiry( $expiry_timestamp );

		if ( $days_left <= 0 ) {
			return 'expired';
		}

		if ( $days_left <= 30 ) {
			return 'expiring_soon';
		}

		return 'valid';
	}

	/**
	 * Check if certificate is valid.
	 *
	 * @param int $expiry_timestamp Expiry timestamp.
	 * @return bool True if valid, false otherwise.
	 */
	private static function is_certificate_valid( $expiry_timestamp ) {
		return time() < $expiry_timestamp;
	}

	/**
	 * Extract Subject Alternative Names (SAN) from certificate.
	 *
	 * @param array $cert_data Certificate data.
	 * @return array SANs.
	 */
	private static function extract_san( $cert_data ) {
		$sans = array();

		if ( isset( $cert_data['extensions']['subjectAltName'] ) ) {
			$san_string = $cert_data['extensions']['subjectAltName'];
			$parts = explode( ',', $san_string );

			foreach ( $parts as $part ) {
				$part = trim( $part );
				if ( 0 === strpos( $part, 'DNS:' ) ) {
					$sans[] = substr( $part, 4 );
				}
			}
		}

		return $sans;
	}

	/**
	 * Check certificate and send notifications if needed.
	 */
	public function check_and_notify() {
		// Only check if alerts are enabled.
		if ( ! glimbyte_ssl_get_option( 'cert_alert_enabled', true ) ) {
			return;
		}

		$parsed_url = wp_parse_url( home_url() );
		$domain = isset( $parsed_url['host'] ) ? $parsed_url['host'] : '';
		$cert_info = self::get_certificate_info( $domain );

		if ( is_wp_error( $cert_info ) ) {
			glimbyte_ssl_log( 'Certificate check failed: ' . $cert_info->get_error_message(), 'error' );
			return;
		}

		$days_left = $cert_info['days_left'];
		$alert_days = glimbyte_ssl_get_option( 'cert_alert_days', '30,14,7' );
		$alert_thresholds = array_map( 'intval', explode( ',', $alert_days ) );

		// Check if we should send notification.
		foreach ( $alert_thresholds as $threshold ) {
			if ( $days_left <= $threshold ) {
				// Check if we've already sent notification for this threshold.
				$sent_key = 'glimbyte_ssl_alert_sent_' . $threshold;
				$already_sent = get_transient( $sent_key );

				if ( ! $already_sent ) {
					$this->send_expiry_notification( $cert_info, $days_left );
					// Mark as sent for 24 hours.
					set_transient( $sent_key, true, DAY_IN_SECONDS );
					break;
				}
			}
		}

		// Log the check.
		glimbyte_ssl_log(
			sprintf(
				/* translators: %d: Days left */
				__( 'Certificate checked. Days until expiry: %d', 'glimbyte-ssl-guardian' ),
				$days_left
			),
			'info'
		);
	}

	/**
	 * Send certificate expiry notification.
	 *
	 * @param array $cert_info Certificate information.
	 * @param int   $days_left Days until expiry.
	 */
	private function send_expiry_notification( $cert_info, $days_left ) {
		$to = glimbyte_ssl_get_option( 'cert_alert_email', get_option( 'admin_email' ) );
		$site_name = get_bloginfo( 'name' );
		$site_url = home_url();

		$subject = sprintf(
			/* translators: 1: Site name, 2: Days left */
			__( '[%1$s] SSL Certificate Expiring in %2$d Days', 'glimbyte-ssl-guardian' ),
			$site_name,
			$days_left
		);

		$message = sprintf(
			/* translators: 1: Site name, 2: Site URL, 3: Days left, 4: Expiry date */
			__(
				"Hello,

Your SSL certificate for %1\$s (%2\$s) will expire in %3\$d days.

Certificate Details:
- Issuer: %5\$s
- Expires: %4\$s

Please renew your SSL certificate as soon as possible to avoid any service interruption.

---
This is an automated notification from Glimbyte SSL Guardian.
Manage your SSL settings: %6\$s",
				'glimbyte-ssl-guardian'
			),
			$site_name,
			$site_url,
			$days_left,
			glimbyte_ssl_format_date( $cert_info['valid_to'] ),
			$cert_info['issuer'],
			glimbyte_ssl_get_admin_url()
		);

		wp_mail( $to, $subject, $message );

		glimbyte_ssl_log(
			sprintf(
				/* translators: %s: Email address */
				__( 'Expiry notification sent to %s', 'glimbyte-ssl-guardian' ),
				$to
			),
			'info'
		);
	}
}
