<?php
/**
 * Multisite Support
 *
 * @package Glimbyte_SSL_Manager
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Multisite class.
 */
class Glimbyte_SSL_Multisite {

	/**
	 * Class instance.
	 *
	 * @var Glimbyte_SSL_Multisite
	 */
	private static $instance = null;

	/**
	 * Get class instance.
	 *
	 * @return Glimbyte_SSL_Multisite
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor.
	 */
	private function __construct() {
		add_action( 'network_admin_menu', array( $this, 'add_network_admin_menu' ) );
		add_action( 'network_admin_edit_glimbyte_ssl_network_settings', array( $this, 'save_network_settings' ) );
	}

	/**
	 * Add network admin menu.
	 */
	public function add_network_admin_menu() {
		add_submenu_page(
			'settings.php',
			__( 'Glimbyte SSL Guardian', 'glimbyte-ssl-guardian' ),
			__( 'SSL Guardian', 'glimbyte-ssl-guardian' ),
			'manage_network_options',
			'glimbyte-ssl-network',
			array( $this, 'render_network_admin_page' )
		);
	}

	/**
	 * Render network admin page.
	 */
	public function render_network_admin_page() {
		if ( ! current_user_can( 'manage_network_options' ) ) {
			return;
		}

		$network_force_https = get_site_option( 'glimbyte_ssl_network_force_https', false );
		$allow_site_override = get_site_option( 'glimbyte_ssl_allow_site_override', true );

		?>
		<div class="wrap">
			<h1><?php esc_html_e( 'Glimbyte SSL Guardian - Network Settings', 'glimbyte-ssl-guardian' ); ?></h1>

			<form method="post" action="edit.php?action=glimbyte_ssl_network_settings">
				<?php wp_nonce_field( 'glimbyte_ssl_network_settings' ); ?>

				<table class="form-table">
					<tr>
						<th scope="row">
							<?php esc_html_e( 'Force HTTPS Network-Wide', 'glimbyte-ssl-guardian' ); ?>
						</th>
						<td>
							<label>
								<input type="checkbox" name="network_force_https" value="1" <?php checked( $network_force_https, true ); ?>>
								<?php esc_html_e( 'Force HTTPS on all sites in the network', 'glimbyte-ssl-guardian' ); ?>
							</label>
							<p class="description">
								<?php esc_html_e( 'When enabled, all sites in the network will automatically redirect to HTTPS.', 'glimbyte-ssl-guardian' ); ?>
							</p>
						</td>
					</tr>

					<tr>
						<th scope="row">
							<?php esc_html_e( 'Allow Site Override', 'glimbyte-ssl-guardian' ); ?>
						</th>
						<td>
							<label>
								<input type="checkbox" name="allow_site_override" value="1" <?php checked( $allow_site_override, true ); ?>>
								<?php esc_html_e( 'Allow individual sites to override network settings', 'glimbyte-ssl-guardian' ); ?>
							</label>
							<p class="description">
								<?php esc_html_e( 'When enabled, site administrators can configure their own SSL settings.', 'glimbyte-ssl-guardian' ); ?>
							</p>
						</td>
					</tr>
				</table>

				<?php submit_button( __( 'Save Network Settings', 'glimbyte-ssl-guardian' ) ); ?>
			</form>

			<hr>

			<h2><?php esc_html_e( 'Network Sites SSL Status', 'glimbyte-ssl-guardian' ); ?></h2>

			<?php $this->display_network_sites_status(); ?>
		</div>
		<?php
	}

	/**
	 * Display SSL status for all network sites.
	 */
	private function display_network_sites_status() {
		$sites = get_sites( array( 'number' => 100 ) );

		if ( empty( $sites ) ) {
			echo '<p>' . esc_html__( 'No sites found.', 'glimbyte-ssl-guardian' ) . '</p>';
			return;
		}

		echo '<table class="wp-list-table widefat fixed striped">';
		echo '<thead>';
		echo '<tr>';
		echo '<th>' . esc_html__( 'Site', 'glimbyte-ssl-guardian' ) . '</th>';
		echo '<th>' . esc_html__( 'URL', 'glimbyte-ssl-guardian' ) . '</th>';
		echo '<th>' . esc_html__( 'HTTPS Status', 'glimbyte-ssl-guardian' ) . '</th>';
		echo '<th>' . esc_html__( 'SSL Certificate', 'glimbyte-ssl-guardian' ) . '</th>';
		echo '</tr>';
		echo '</thead>';
		echo '<tbody>';

		foreach ( $sites as $site ) {
			switch_to_blog( $site->blog_id );

			$site_name = get_bloginfo( 'name' );
			$site_url = get_home_url();
			$https_enabled = glimbyte_ssl_get_option( 'https_enabled', false );
			$is_https = ( 0 === strpos( $site_url, 'https://' ) );

			echo '<tr>';
			echo '<td>' . esc_html( $site_name ) . '</td>';
			echo '<td><a href="' . esc_url( $site_url ) . '" target="_blank">' . esc_html( $site_url ) . '</a></td>';
			echo '<td>';
			if ( $https_enabled ) {
				echo '<span style="color: #00a32a;">' . esc_html__( 'Enabled', 'glimbyte-ssl-guardian' ) . '</span>';
			} else {
				echo '<span style="color: #646970;">' . esc_html__( 'Disabled', 'glimbyte-ssl-guardian' ) . '</span>';
			}
			echo '</td>';
			echo '<td>';
			if ( $is_https ) {
				$parsed_url = wp_parse_url( $site_url );
				$domain = isset( $parsed_url['host'] ) ? $parsed_url['host'] : '';
				$cert_info = glimbyte_ssl_get_cert_info( $domain );
				if ( ! is_wp_error( $cert_info ) ) {
					$days_left = absint( $cert_info['days_left'] );
					if ( $days_left > 30 ) {
						$color = '#00a32a';
					} elseif ( $days_left > 7 ) {
						$color = '#dba617';
					} else {
						$color = '#d63638';
					}
					echo '<span style="color: ' . esc_attr( $color ) . ';">';
					printf(
						/* translators: %d: Days until expiry */
						esc_html__( '%d days remaining', 'glimbyte-ssl-guardian' ),
						absint( $days_left )
					);
					echo '</span>';
				} else {
					echo '<span style="color: #d63638;">' . esc_html__( 'Error checking certificate', 'glimbyte-ssl-guardian' ) . '</span>';
				}
			} else {
				echo '<span style="color: #646970;">' . esc_html__( 'Not using HTTPS', 'glimbyte-ssl-guardian' ) . '</span>';
			}
			echo '</td>';
			echo '</tr>';

			restore_current_blog();
		}

		echo '</tbody>';
		echo '</table>';
	}

	/**
	 * Save network settings.
	 */
	public function save_network_settings() {
		check_admin_referer( 'glimbyte_ssl_network_settings' );

		if ( ! current_user_can( 'manage_network_options' ) ) {
			wp_die( esc_html__( 'You do not have permission to access this page.', 'glimbyte-ssl-guardian' ) );
		}

		$network_force_https = isset( $_POST['network_force_https'] ) ? true : false;
		$allow_site_override = isset( $_POST['allow_site_override'] ) ? true : false;

		update_site_option( 'glimbyte_ssl_network_force_https', $network_force_https );
		update_site_option( 'glimbyte_ssl_allow_site_override', $allow_site_override );

		wp_safe_redirect( add_query_arg( array(
			'page'    => 'glimbyte-ssl-network',
			'updated' => 'true',
		), network_admin_url( 'settings.php' ) ) );
		exit;
	}
}
