<?php
/**
 * Mixed Content Handler
 *
 * @package Glimbyte_SSL_Manager
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Mixed Content class.
 */
class Glimbyte_SSL_Mixed_Content {

	/**
	 * Class instance.
	 *
	 * @var Glimbyte_SSL_Mixed_Content
	 */
	private static $instance = null;

	/**
	 * Get class instance.
	 *
	 * @return Glimbyte_SSL_Mixed_Content
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Track if output buffering is active.
	 *
	 * @var bool
	 */
	private $buffer_active = false;

	/**
	 * Constructor.
	 */
	private function __construct() {
		add_action( 'template_redirect', array( $this, 'start_output_buffer' ), 0 );
		add_action( 'shutdown', array( $this, 'end_output_buffer' ), 999 );
	}

	/**
	 * Start output buffering to fix mixed content.
	 */
	public function start_output_buffer() {
		// Only fix on HTTPS pages.
		if ( ! glimbyte_ssl_is_https() ) {
			return;
		}

		$this->buffer_active = true;
		ob_start( array( $this, 'fix_mixed_content' ) );
	}

	/**
	 * End output buffering explicitly.
	 */
	public function end_output_buffer() {
		if ( $this->buffer_active && ob_get_level() > 0 ) {
			ob_end_flush();
		}
	}

	/**
	 * Fix mixed content in HTML output.
	 *
	 * @param string $buffer HTML buffer.
	 * @return string Fixed HTML.
	 */
	public function fix_mixed_content( $buffer ) {
		// Get fixing method.
		$method = glimbyte_ssl_get_option( 'mixed_content_method', 'https' );

		// Get site URL.
		$site_url = home_url();
		$http_site_url = preg_replace( '/^https:/i', 'http:', $site_url );

		// Count fixes for logging.
		$fixes_count = 0;

		// Fix internal URLs.
		if ( 'https' === $method ) {
			// Replace http:// with https:// for internal resources.
			$buffer = str_replace( $http_site_url, str_replace( 'http:', 'https:', $http_site_url ), $buffer, $count );
			$fixes_count += $count;
		} else {
			// Replace with protocol-relative URLs.
			$buffer = str_replace( array( 'http:' . $site_url, 'https:' . $site_url ), $site_url, $buffer, $count );
			$fixes_count += $count;
		}

		// Fix common external HTTP resources.
		$patterns = array(
			// Images.
			'/<img[^>]+src=["\']http:\/\/([^"\']+)["\']/i',
			// Scripts.
			'/<script[^>]+src=["\']http:\/\/([^"\']+)["\']/i',
			// Stylesheets.
			'/<link[^>]+href=["\']http:\/\/([^"\']+)["\']/i',
			// Iframes.
			'/<iframe[^>]+src=["\']http:\/\/([^"\']+)["\']/i',
			// Background images in inline CSS.
			'/url\(["\']?http:\/\/([^"\')\s]+)["\']?\)/i',
		);

		foreach ( $patterns as $pattern ) {
			if ( 'https' === $method ) {
				$buffer = preg_replace_callback( $pattern, function( $matches ) {
					return str_replace( 'http://', 'https://', $matches[0] );
				}, $buffer, -1, $count );
			} else {
				$buffer = preg_replace_callback( $pattern, function( $matches ) {
					return str_replace( 'http://', '//', $matches[0] );
				}, $buffer, -1, $count );
			}
			$fixes_count += $count;
		}

		// Log fixes if any.
		if ( $fixes_count > 0 ) {
			glimbyte_ssl_log(
				sprintf(
					/* translators: %d: Number of fixes */
					__( 'Fixed %d mixed content issues on this page', 'glimbyte-ssl-guardian' ),
					$fixes_count
				),
				'info'
			);
		}

		return $buffer;
	}

	/**
	 * Scan content for mixed content issues.
	 *
	 * @return array Scan results.
	 */
	public static function scan_content() {
		global $wpdb;

		$results = array(
			'total_posts'   => 0,
			'posts_with_issues' => 0,
			'total_issues'  => 0,
			'issues_by_type' => array(
				'images'      => 0,
				'scripts'     => 0,
				'stylesheets' => 0,
				'iframes'     => 0,
				'other'       => 0,
			),
			'affected_posts' => array(),
		);

		// Get all published posts and pages.
		$posts = get_posts( array(
			'post_type'   => array( 'post', 'page' ),
			'post_status' => 'publish',
			'numberposts' => -1,
		) );

		$results['total_posts'] = count( $posts );

		foreach ( $posts as $post ) {
			$content = $post->post_content;
			$post_issues = array();

			// Check for images.
			if ( preg_match_all( '/<img[^>]+src=["\']http:\/\/([^"\']+)["\']/i', $content, $matches ) ) {
				$count = count( $matches[0] );
				$results['issues_by_type']['images'] += $count;
				$results['total_issues'] += $count;
				$post_issues['images'] = $count;
			}

			// Check for scripts.
			if ( preg_match_all( '/<script[^>]+src=["\']http:\/\/([^"\']+)["\']/i', $content, $matches ) ) {
				$count = count( $matches[0] );
				$results['issues_by_type']['scripts'] += $count;
				$results['total_issues'] += $count;
				$post_issues['scripts'] = $count;
			}

			// Check for stylesheets.
			if ( preg_match_all( '/<link[^>]+href=["\']http:\/\/([^"\']+)["\']/i', $content, $matches ) ) {
				$count = count( $matches[0] );
				$results['issues_by_type']['stylesheets'] += $count;
				$results['total_issues'] += $count;
				$post_issues['stylesheets'] = $count;
			}

			// Check for iframes.
			if ( preg_match_all( '/<iframe[^>]+src=["\']http:\/\/([^"\']+)["\']/i', $content, $matches ) ) {
				$count = count( $matches[0] );
				$results['issues_by_type']['iframes'] += $count;
				$results['total_issues'] += $count;
				$post_issues['iframes'] = $count;
			}

			// Check for other HTTP URLs.
			if ( preg_match_all( '/http:\/\/[^"\'\s<>]+/i', $content, $matches ) ) {
				$count = count( $matches[0] ) - array_sum( $post_issues );
				if ( $count > 0 ) {
					$results['issues_by_type']['other'] += $count;
					$results['total_issues'] += $count;
					$post_issues['other'] = $count;
				}
			}

			// If post has issues, add to affected posts.
			if ( ! empty( $post_issues ) ) {
				$results['posts_with_issues']++;
				$results['affected_posts'][] = array(
					'id'     => $post->ID,
					'title'  => $post->post_title,
					'type'   => $post->post_type,
					'url'    => get_permalink( $post->ID ),
					'issues' => $post_issues,
				);
			}
		}

		return $results;
	}

	/**
	 * Fix mixed content in database.
	 *
	 * @return array Fix results.
	 */
	public static function fix_database_content() {
		global $wpdb;

		$results = array(
			'posts_fixed' => 0,
			'fixes_count' => 0,
		);

		// Get all published posts and pages.
		$posts = get_posts( array(
			'post_type'   => array( 'post', 'page' ),
			'post_status' => 'publish',
			'numberposts' => -1,
		) );

		$method = glimbyte_ssl_get_option( 'mixed_content_method', 'https' );

		foreach ( $posts as $post ) {
			$content = $post->post_content;
			$original_content = $content;

			if ( 'https' === $method ) {
				// Replace HTTP with HTTPS.
				$content = preg_replace( '/http:\/\//i', 'https://', $content, -1, $count );
			} else {
				// Replace with protocol-relative URLs.
				$content = preg_replace( '/(https?):\/\//i', '//', $content, -1, $count );
			}

			// Update post if content changed.
			if ( $content !== $original_content ) {
				wp_update_post( array(
					'ID'           => $post->ID,
					'post_content' => $content,
				) );

				$results['posts_fixed']++;
				$results['fixes_count'] += $count;
			}
		}

		// Log the fix.
		glimbyte_ssl_log(
			sprintf(
				/* translators: 1: Posts fixed, 2: Total fixes */
				__( 'Database fix completed. Posts fixed: %1$d, Total fixes: %2$d', 'glimbyte-ssl-guardian' ),
				$results['posts_fixed'],
				$results['fixes_count']
			),
			'info'
		);

		return $results;
	}
}
