<?php
/**
 * Logger
 *
 * @package Glimbyte_SSL_Manager
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Logger class.
 */
class Glimbyte_SSL_Logger {

	/**
	 * Maximum log entries.
	 */
	const MAX_LOG_ENTRIES = 1000;

	/**
	 * Log a message.
	 *
	 * @param string $message Log message.
	 * @param string $type Log type (info, warning, error).
	 */
	public static function log( $message, $type = 'info' ) {
		$logs = self::get_logs();

		// Add new log entry.
		$logs[] = array(
			'timestamp' => time(),
			'type'      => $type,
			'message'   => $message,
		);

		// Limit log size.
		if ( count( $logs ) > self::MAX_LOG_ENTRIES ) {
			$logs = array_slice( $logs, -self::MAX_LOG_ENTRIES );
		}

		// Save logs.
		update_option( 'glimbyte_ssl_logs', $logs, false );
	}

	/**
	 * Get all logs.
	 *
	 * @return array Logs.
	 */
	public static function get_logs() {
		return get_option( 'glimbyte_ssl_logs', array() );
	}

	/**
	 * Get recent logs.
	 *
	 * @param int $limit Number of logs to retrieve.
	 * @return array Recent logs.
	 */
	public static function get_recent_logs( $limit = 100 ) {
		$logs = self::get_logs();
		return array_slice( array_reverse( $logs ), 0, $limit );
	}

	/**
	 * Clear all logs.
	 *
	 * @return bool True if cleared, false otherwise.
	 */
	public static function clear_logs() {
		return delete_option( 'glimbyte_ssl_logs' );
	}

	/**
	 * Export logs as text.
	 *
	 * @return string Logs as text.
	 */
	public static function export_logs() {
		$logs = self::get_logs();
		$output = '';

		foreach ( $logs as $log ) {
			$output .= sprintf(
				'[%s] [%s] %s' . PHP_EOL,
				gmdate( 'Y-m-d H:i:s', $log['timestamp'] ),
				strtoupper( $log['type'] ),
				$log['message']
			);
		}

		return $output;
	}

	/**
	 * Get log count by type.
	 *
	 * @return array Log counts.
	 */
	public static function get_log_counts() {
		$logs = self::get_logs();
		$counts = array(
			'info'    => 0,
			'warning' => 0,
			'error'   => 0,
		);

		foreach ( $logs as $log ) {
			$type = isset( $log['type'] ) ? $log['type'] : 'info';
			if ( isset( $counts[ $type ] ) ) {
				$counts[ $type ]++;
			}
		}

		return $counts;
	}
}
