<?php
/**
 * HTTPS Redirect Handler
 *
 * @package Glimbyte_SSL_Manager
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * HTTPS Redirect class.
 */
class Glimbyte_SSL_HTTPS_Redirect {

	/**
	 * Class instance.
	 *
	 * @var Glimbyte_SSL_HTTPS_Redirect
	 */
	private static $instance = null;

	/**
	 * Get class instance.
	 *
	 * @return Glimbyte_SSL_HTTPS_Redirect
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor.
	 */
	private function __construct() {
		add_action( 'template_redirect', array( $this, 'force_https' ), 1 );
	}

	/**
	 * Force HTTPS redirect.
	 */
	public function force_https() {
		// Skip if already on HTTPS.
		if ( glimbyte_ssl_is_https() ) {
			return;
		}

		// Skip for localhost.
		if ( glimbyte_ssl_is_localhost() ) {
			glimbyte_ssl_log( 'HTTPS redirect skipped: localhost detected', 'info' );
			return;
		}

		// Get current URL.
		$current_url = glimbyte_ssl_get_current_url();

		// Check if URL is excluded.
		if ( glimbyte_ssl_is_url_excluded( $current_url ) ) {
			glimbyte_ssl_log( 'HTTPS redirect skipped: URL excluded - ' . $current_url, 'info' );
			return;
		}

		// Check if we should force HTTPS.
		if ( ! $this->should_force_https() ) {
			return;
		}

		// Build HTTPS URL.
		$https_url = $this->get_https_url( $current_url );

		// Get redirect type.
		$redirect_type = glimbyte_ssl_get_option( 'redirect_type', '301' );
		$redirect_code = $this->get_redirect_code( $redirect_type );

		// Log redirect.
		glimbyte_ssl_log(
			sprintf(
				/* translators: 1: HTTP URL, 2: HTTPS URL, 3: Redirect code */
				__( 'Redirecting %1$s to %2$s (HTTP %3$d)', 'glimbyte-ssl-guardian' ),
				$current_url,
				$https_url,
				$redirect_code
			),
			'info'
		);

		// Perform redirect.
		wp_safe_redirect( $https_url, $redirect_code );
		exit;
	}

	/**
	 * Check if HTTPS should be forced for current request.
	 *
	 * @return bool True if HTTPS should be forced, false otherwise.
	 */
	private function should_force_https() {
		// Check if forcing all pages.
		if ( glimbyte_ssl_get_option( 'force_all', true ) ) {
			return true;
		}

		// Add custom logic for specific pages if needed.
		// This can be extended with filters.
		return apply_filters( 'glimbyte_ssl_should_force_https', false );
	}

	/**
	 * Convert HTTP URL to HTTPS.
	 *
	 * @param string $url HTTP URL.
	 * @return string HTTPS URL.
	 */
	private function get_https_url( $url ) {
		return preg_replace( '/^http:/i', 'https:', $url );
	}

	/**
	 * Get HTTP redirect code.
	 *
	 * @param string $type Redirect type (301, 302, 307).
	 * @return int Redirect code.
	 */
	private function get_redirect_code( $type ) {
		$codes = array(
			'301' => 301, // Permanent.
			'302' => 302, // Temporary.
			'307' => 307, // Temporary (preserving method).
		);

		return isset( $codes[ $type ] ) ? $codes[ $type ] : 301;
	}
}
