<?php
/**
 * HSTS Manager
 *
 * @package Glimbyte_SSL_Manager
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * HSTS Manager class.
 */
class Glimbyte_SSL_HSTS_Manager {

	/**
	 * Class instance.
	 *
	 * @var Glimbyte_SSL_HSTS_Manager
	 */
	private static $instance = null;

	/**
	 * Get class instance.
	 *
	 * @return Glimbyte_SSL_HSTS_Manager
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor.
	 */
	private function __construct() {
		add_action( 'send_headers', array( $this, 'send_hsts_header' ) );
	}

	/**
	 * Send HSTS header.
	 */
	public function send_hsts_header() {
		// Only send HSTS header on HTTPS.
		if ( ! glimbyte_ssl_is_https() ) {
			return;
		}

		// Build HSTS header value.
		$header_value = $this->build_hsts_header();

		if ( empty( $header_value ) ) {
			return;
		}

		// Send header.
		header( 'Strict-Transport-Security: ' . $header_value );

		// Log if debug mode is enabled.
		glimbyte_ssl_log(
			sprintf(
				/* translators: %s: HSTS header value */
				__( 'HSTS header sent: %s', 'glimbyte-ssl-guardian' ),
				$header_value
			),
			'info'
		);
	}

	/**
	 * Build HSTS header value.
	 *
	 * @return string HSTS header value.
	 */
	private function build_hsts_header() {
		$max_age = glimbyte_ssl_get_option( 'hsts_max_age', '31536000' );
		$include_subdomains = glimbyte_ssl_get_option( 'hsts_subdomains', false );
		$preload = glimbyte_ssl_get_option( 'hsts_preload', false );

		// Validate max-age.
		$max_age = absint( $max_age );
		if ( $max_age < 1 ) {
			$max_age = 31536000; // Default to 1 year.
		}

		// Build header.
		$parts = array( 'max-age=' . $max_age );

		if ( $include_subdomains ) {
			$parts[] = 'includeSubDomains';
		}

		if ( $preload ) {
			$parts[] = 'preload';
		}

		return implode( '; ', $parts );
	}

	/**
	 * Get HSTS max-age options.
	 *
	 * @return array Max-age options.
	 */
	public static function get_max_age_options() {
		return array(
			'2592000'  => __( '1 Month (2592000 seconds)', 'glimbyte-ssl-guardian' ),
			'15768000' => __( '6 Months (15768000 seconds)', 'glimbyte-ssl-guardian' ),
			'31536000' => __( '1 Year (31536000 seconds) - Recommended', 'glimbyte-ssl-guardian' ),
			'63072000' => __( '2 Years (63072000 seconds)', 'glimbyte-ssl-guardian' ),
		);
	}
}
