<?php
/**
 * Admin Interface
 *
 * @package Glimbyte_SSL_Manager
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Admin class.
 */
class Glimbyte_SSL_Admin {

	/**
	 * Class instance.
	 *
	 * @var Glimbyte_SSL_Admin
	 */
	private static $instance = null;

	/**
	 * Get class instance.
	 *
	 * @return Glimbyte_SSL_Admin
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor.
	 */
	private function __construct() {
		add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
		add_action( 'admin_init', array( $this, 'register_settings' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
		add_action( 'admin_notices', array( $this, 'display_admin_notices' ) );
		add_action( 'wp_ajax_glimbyte_ssl_scan_content', array( $this, 'ajax_scan_content' ) );
		add_action( 'wp_ajax_glimbyte_ssl_fix_content', array( $this, 'ajax_fix_content' ) );
		add_action( 'wp_ajax_glimbyte_ssl_test_email', array( $this, 'ajax_test_email' ) );
		add_action( 'wp_ajax_glimbyte_ssl_submit_support', array( $this, 'ajax_submit_support' ) );
	}

	/**
	 * Add admin menu.
	 */
	public function add_admin_menu() {
		add_menu_page(
			__( 'Glimbyte SSL Guardian', 'glimbyte-ssl-guardian' ),
			__( 'SSL Guardian', 'glimbyte-ssl-guardian' ),
			'manage_options',
			'glimbyte-ssl-guardian',
			array( $this, 'render_admin_page' ),
			'dashicons-lock',
			30
		);
	}

	/**
	 * Register plugin settings.
	 */
	public function register_settings() {
		// HTTPS Settings.
		register_setting( 'glimbyte_ssl_https', 'glimbyte_ssl_https_enabled', array(
			'type'              => 'boolean',
			'sanitize_callback' => 'glimbyte_ssl_sanitize_checkbox',
		) );

		register_setting( 'glimbyte_ssl_https', 'glimbyte_ssl_redirect_type', array(
			'type'              => 'string',
			'default'           => '301',
			'sanitize_callback' => 'sanitize_text_field',
		) );

		register_setting( 'glimbyte_ssl_https', 'glimbyte_ssl_force_all', array(
			'type'              => 'boolean',
			'sanitize_callback' => 'glimbyte_ssl_sanitize_checkbox',
		) );

		register_setting( 'glimbyte_ssl_https', 'glimbyte_ssl_excluded_urls', array(
			'type'              => 'string',
			'sanitize_callback' => 'sanitize_textarea_field',
		) );

		// Mixed Content Settings.
		register_setting( 'glimbyte_ssl_mixed_content', 'glimbyte_ssl_mixed_content_fix_enabled', array(
			'type'              => 'boolean',
			'sanitize_callback' => 'glimbyte_ssl_sanitize_checkbox',
		) );

		register_setting( 'glimbyte_ssl_mixed_content', 'glimbyte_ssl_mixed_content_method', array(
			'type'              => 'string',
			'default'           => 'https',
			'sanitize_callback' => 'sanitize_text_field',
		) );

		// Certificate Settings.
		register_setting( 'glimbyte_ssl_certificate', 'glimbyte_ssl_cert_alert_enabled', array(
			'type'              => 'boolean',
			'sanitize_callback' => 'glimbyte_ssl_sanitize_checkbox',
		) );

		register_setting( 'glimbyte_ssl_certificate', 'glimbyte_ssl_cert_alert_days', array(
			'type'              => 'string',
			'default'           => '30,14,7',
			'sanitize_callback' => 'sanitize_text_field',
		) );

		register_setting( 'glimbyte_ssl_certificate', 'glimbyte_ssl_cert_alert_email', array(
			'type'              => 'string',
			'sanitize_callback' => 'sanitize_email',
		) );

		// HSTS Settings.
		register_setting( 'glimbyte_ssl_hsts', 'glimbyte_ssl_hsts_enabled', array(
			'type'              => 'boolean',
			'sanitize_callback' => 'glimbyte_ssl_sanitize_checkbox',
		) );

		register_setting( 'glimbyte_ssl_hsts', 'glimbyte_ssl_hsts_max_age', array(
			'type'              => 'string',
			'default'           => '31536000',
			'sanitize_callback' => 'sanitize_text_field',
		) );

		register_setting( 'glimbyte_ssl_hsts', 'glimbyte_ssl_hsts_subdomains', array(
			'type'              => 'boolean',
			'sanitize_callback' => 'glimbyte_ssl_sanitize_checkbox',
		) );

		register_setting( 'glimbyte_ssl_hsts', 'glimbyte_ssl_hsts_preload', array(
			'type'              => 'boolean',
			'sanitize_callback' => 'glimbyte_ssl_sanitize_checkbox',
		) );

		// Debug Settings.
		register_setting( 'glimbyte_ssl_debug', 'glimbyte_ssl_debug_mode', array(
			'type'              => 'boolean',
			'sanitize_callback' => 'glimbyte_ssl_sanitize_checkbox',
		) );
	}

	/**
	 * Enqueue admin assets.
	 *
	 * @param string $hook Current admin page hook.
	 */
	public function enqueue_admin_assets( $hook ) {
		// Load on our admin page - try both hook patterns.
		if ( strpos( $hook, 'glimbyte-ssl' ) === false ) {
			return;
		}

		wp_enqueue_style(
			'glimbyte-ssl-admin',
			GLIMBYTE_SSL_PLUGIN_URL . 'admin/css/admin.css',
			array(),
			GLIMBYTE_SSL_VERSION
		);

		wp_enqueue_script(
			'glimbyte-ssl-admin',
			GLIMBYTE_SSL_PLUGIN_URL . 'admin/js/admin.js',
			array( 'jquery' ),
			GLIMBYTE_SSL_VERSION,
			true
		);

		wp_localize_script( 'glimbyte-ssl-admin', 'glimbyteSSL', array(
			'ajax_url' => admin_url( 'admin-ajax.php' ),
			'nonce'    => wp_create_nonce( 'glimbyte_ssl_ajax' ),
			'strings'  => array(
				'scanning'         => __( 'Scanning...', 'glimbyte-ssl-guardian' ),
				'fixing'           => __( 'Fixing...', 'glimbyte-ssl-guardian' ),
				'fix_complete'     => __( 'Fix complete!', 'glimbyte-ssl-guardian' ),
				'error'            => __( 'An error occurred. Please try again.', 'glimbyte-ssl-guardian' ),
			),
		) );
	}

	/**
	 * Display admin notices.
	 */
	public function display_admin_notices() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Check if HTTPS is enabled but site is not using HTTPS.
		if ( glimbyte_ssl_get_option( 'https_enabled', false ) && ! glimbyte_ssl_is_https() ) {
			?>
			<div class="notice notice-warning">
				<p>
					<?php esc_html_e( 'HTTPS redirect is enabled but your site is not using HTTPS. Please check your SSL certificate configuration.', 'glimbyte-ssl-guardian' ); ?>
				</p>
			</div>
			<?php
		}
	}

	/**
	 * Render admin page.
	 */
	public function render_admin_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'glimbyte-ssl-guardian' ) );
		}

		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Tab parameter is only used for display, not data processing.
		$active_tab = isset( $_GET['tab'] ) ? sanitize_key( $_GET['tab'] ) : 'dashboard';
		?>
		<div class="wrap glimbyte-ssl-wrap">
			<h1><?php esc_html_e( 'Glimbyte SSL Guardian', 'glimbyte-ssl-guardian' ); ?></h1>

			<nav class="nav-tab-wrapper">
				<a href="?page=glimbyte-ssl-guardian&tab=dashboard" class="nav-tab <?php echo 'dashboard' === $active_tab ? 'nav-tab-active' : ''; ?>">
					<?php esc_html_e( 'Dashboard', 'glimbyte-ssl-guardian' ); ?>
				</a>
				<a href="?page=glimbyte-ssl-guardian&tab=https" class="nav-tab <?php echo 'https' === $active_tab ? 'nav-tab-active' : ''; ?>">
					<?php esc_html_e( 'HTTPS Settings', 'glimbyte-ssl-guardian' ); ?>
				</a>
				<a href="?page=glimbyte-ssl-guardian&tab=mixed-content" class="nav-tab <?php echo 'mixed-content' === $active_tab ? 'nav-tab-active' : ''; ?>">
					<?php esc_html_e( 'Mixed Content', 'glimbyte-ssl-guardian' ); ?>
				</a>
				<a href="?page=glimbyte-ssl-guardian&tab=certificate" class="nav-tab <?php echo 'certificate' === $active_tab ? 'nav-tab-active' : ''; ?>">
					<?php esc_html_e( 'Certificate Monitor', 'glimbyte-ssl-guardian' ); ?>
				</a>
				<a href="?page=glimbyte-ssl-guardian&tab=hsts" class="nav-tab <?php echo 'hsts' === $active_tab ? 'nav-tab-active' : ''; ?>">
					<?php esc_html_e( 'HSTS', 'glimbyte-ssl-guardian' ); ?>
				</a>
				<a href="?page=glimbyte-ssl-guardian&tab=debug" class="nav-tab <?php echo 'debug' === $active_tab ? 'nav-tab-active' : ''; ?>">
					<?php esc_html_e( 'Debug', 'glimbyte-ssl-guardian' ); ?>
				</a>
				<a href="?page=glimbyte-ssl-guardian&tab=support" class="nav-tab <?php echo 'support' === $active_tab ? 'nav-tab-active' : ''; ?>">
					<?php esc_html_e( 'Support', 'glimbyte-ssl-guardian' ); ?>
				</a>
			</nav>

			<div class="glimbyte-ssl-content">
				<?php
				switch ( $active_tab ) {
					case 'https':
						$this->render_tab( 'https' );
						break;
					case 'mixed-content':
						$this->render_tab( 'mixed-content' );
						break;
					case 'certificate':
						$this->render_tab( 'certificate' );
						break;
					case 'hsts':
						$this->render_tab( 'hsts' );
						break;
					case 'debug':
						$this->render_tab( 'debug' );
						break;
					case 'support':
						$this->render_tab( 'support' );
						break;
					case 'dashboard':
					default:
						$this->render_tab( 'dashboard' );
						break;
				}
				?>
			</div>
		</div>
		<?php
	}

	/**
	 * Render tab content.
	 *
	 * @param string $tab Tab name.
	 */
	private function render_tab( $tab ) {
		$file = GLIMBYTE_SSL_PLUGIN_DIR . 'admin/views/' . $tab . '.php';
		
		if ( file_exists( $file ) ) {
			include $file;
		}
	}

	/**
	 * AJAX: Scan content for mixed content issues.
	 */
	public function ajax_scan_content() {
		check_ajax_referer( 'glimbyte_ssl_ajax', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'glimbyte-ssl-guardian' ) ) );
		}

		$results = Glimbyte_SSL_Mixed_Content::scan_content();

		if ( is_wp_error( $results ) ) {
			wp_send_json_error( array( 'message' => $results->get_error_message() ) );
		}

		// Store results in transient for display.
		set_transient( 'glimbyte_ssl_scan_results', $results, WEEK_IN_SECONDS );

		wp_send_json_success( $results );
	}

	/**
	 * AJAX: Fix mixed content issues in database.
	 */
	public function ajax_fix_content() {
		check_ajax_referer( 'glimbyte_ssl_ajax', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'glimbyte-ssl-guardian' ) ) );
		}

		$fixed = Glimbyte_SSL_Mixed_Content::fix_database_content();

		if ( is_wp_error( $fixed ) ) {
			wp_send_json_error( array( 'message' => $fixed->get_error_message() ) );
		}

		// Clear scan results.
		delete_transient( 'glimbyte_ssl_scan_results' );

		wp_send_json_success( array( 'posts_fixed' => $fixed ) );
	}

	/**
	 * AJAX: Test email notification.
	 */
	public function ajax_test_email() {
		check_ajax_referer( 'glimbyte_ssl_ajax', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'glimbyte-ssl-guardian' ) ) );
		}

		$to = glimbyte_ssl_get_option( 'cert_alert_email', get_option( 'admin_email' ) );
		$subject = __( '[Test] SSL Certificate Alert', 'glimbyte-ssl-guardian' );
		$message = __( 'This is a test email from Glimbyte SSL Guardian. If you received this, email notifications are working correctly.', 'glimbyte-ssl-guardian' );

		$sent = wp_mail( $to, $subject, $message );

		if ( $sent ) {
			wp_send_json_success( array( 'message' => __( 'Test email sent successfully!', 'glimbyte-ssl-guardian' ) ) );
		} else {
			wp_send_json_error( array( 'message' => __( 'Failed to send test email. Please check your email configuration.', 'glimbyte-ssl-guardian' ) ) );
		}
	}

	/**
	 * AJAX: Submit support request.
	 */
	public function ajax_submit_support() {
		check_ajax_referer( 'glimbyte_ssl_ajax', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'glimbyte-ssl-guardian' ) ) );
		}

		// Sanitize inputs.
		$name = isset( $_POST['name'] ) ? sanitize_text_field( wp_unslash( $_POST['name'] ) ) : '';
		$email = isset( $_POST['email'] ) ? sanitize_email( wp_unslash( $_POST['email'] ) ) : '';
		$phone = isset( $_POST['phone'] ) ? sanitize_text_field( wp_unslash( $_POST['phone'] ) ) : '';
		$message = isset( $_POST['message'] ) ? sanitize_textarea_field( wp_unslash( $_POST['message'] ) ) : '';
		$consent = isset( $_POST['consent'] ) ? true : false;

		// Validate required fields.
		if ( empty( $name ) || empty( $email ) || empty( $message ) ) {
			wp_send_json_error( array( 'message' => __( 'Please fill in all required fields.', 'glimbyte-ssl-guardian' ) ) );
		}

		if ( ! $consent ) {
			wp_send_json_error( array( 'message' => __( 'Please provide consent to contact you.', 'glimbyte-ssl-guardian' ) ) );
		}

		// Validate email.
		if ( ! is_email( $email ) ) {
			wp_send_json_error( array( 'message' => __( 'Please provide a valid email address.', 'glimbyte-ssl-guardian' ) ) );
		}

		// Gather system information.
		global $wp_version;
		$site_url = home_url();
		$cert_info = glimbyte_ssl_get_cert_info();
		$is_https = glimbyte_ssl_is_https();
		$php_version = PHP_VERSION;
		$wp_version_info = $wp_version;

		// Get SSL status.
		$ssl_status = 'Unknown';
		if ( is_wp_error( $cert_info ) ) {
			$ssl_status = 'Error: ' . $cert_info->get_error_message();
		} elseif ( ! $is_https ) {
			$ssl_status = 'Not using HTTPS';
		} elseif ( $cert_info['is_valid'] ) {
			$ssl_status = 'Valid (expires in ' . $cert_info['days_left'] . ' days)';
		} else {
			$ssl_status = 'Invalid or Expired';
		}

		// Prepare data to send to Glimbyte API.
		$api_data = array(
			'name'            => $name,
			'email'           => $email,
			'phone'           => $phone,
			'message'         => $message,
			'site_url'        => $site_url,
			'wp_version'      => $wp_version_info,
			'php_version'     => $php_version,
			'plugin_version'  => GLIMBYTE_SSL_VERSION,
			'ssl_status'      => $ssl_status,
			'is_https'        => $is_https ? 'yes' : 'no',
			'consent'         => true,
			'source'          => 'glimbyte-ssl-guardian',
		);

		// Try sending via API first (most reliable).
		$api_sent = $this->send_to_glimbyte_api( $api_data );

		// Fallback to wp_mail if API fails.
		if ( ! $api_sent ) {
			$mail_sent = $this->send_via_wp_mail( $api_data );
			
			if ( $mail_sent ) {
				glimbyte_ssl_log(
					sprintf(
						/* translators: %s: Email address */
						__( 'Support request submitted by %s (via wp_mail fallback)', 'glimbyte-ssl-guardian' ),
						$email
					),
					'info'
				);

				wp_send_json_success( array( 
					'message' => __( 'Thank you! Your support request has been sent. We\'ll contact you shortly at the email address provided.', 'glimbyte-ssl-guardian' ) 
				) );
			} else {
				wp_send_json_error( array( 
					'message' => __( 'Unable to send support request. Please contact us directly at support@glimbyte.ie', 'glimbyte-ssl-guardian' ) 
				) );
			}
		} else {
			// API success.
			glimbyte_ssl_log(
				sprintf(
					/* translators: %s: Email address */
					__( 'Support request submitted by %s', 'glimbyte-ssl-guardian' ),
					$email
				),
				'info'
			);

			wp_send_json_success( array( 
				'message' => __( 'Thank you! Your support request has been sent. We\'ll contact you shortly at the email address provided.', 'glimbyte-ssl-guardian' ) 
			) );
		}
	}

	/**
	 * Send support request to Glimbyte API endpoint.
	 *
	 * @param array $data Form data.
	 * @return bool True if sent successfully, false otherwise.
	 */
	private function send_to_glimbyte_api( $data ) {
		// Glimbyte API endpoint for support requests.
		$api_url = 'https://glimbyte.ie/wp-json/glimbyte/v1/support-request';

		// Send POST request.
		$response = wp_remote_post(
			$api_url,
			array(
				'timeout'     => 15,
				'headers'     => array(
					'Content-Type' => 'application/json',
				),
				'body'        => wp_json_encode( $data ),
				'data_format' => 'body',
			)
		);

		// Check for errors.
		if ( is_wp_error( $response ) ) {
			glimbyte_ssl_log( 'API request failed: ' . $response->get_error_message(), 'warning' );
			return false;
		}

		$response_code = wp_remote_retrieve_response_code( $response );
		if ( 200 !== $response_code && 201 !== $response_code ) {
			glimbyte_ssl_log( 'API returned error code: ' . $response_code, 'warning' );
			return false;
		}

		return true;
	}

	/**
	 * Send support request via wp_mail as fallback.
	 *
	 * @param array $data Form data.
	 * @return bool True if sent successfully, false otherwise.
	 */
	private function send_via_wp_mail( $data ) {
		$to = 'support@glimbyte.ie';
		$subject = '[Glimbyte SSL Guardian] Support Request from ' . $data['name'];
		
		// HTML email message.
		// Note: Inline styles are required for HTML emails as email clients don't support external stylesheets.
		// This is standard practice for email templates and not a wp_enqueue issue.
		// phpcs:disable WordPress.WP.EnqueuedResources.NonEnqueuedStylesheet
		$email_message = '
		<html>
		<head>
			<style>
				body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
				.header { background: #2271b1; color: white; padding: 20px; }
				.section { margin: 20px 0; padding: 15px; background: #f5f5f5; border-left: 4px solid #2271b1; }
				.section h3 { margin-top: 0; color: #2271b1; }
				.info-row { margin: 8px 0; }
				.label { font-weight: bold; display: inline-block; width: 150px; }
				.message-box { background: white; padding: 15px; border: 1px solid #ddd; margin-top: 10px; }
			</style>
		</head>
		<body>
			<div class="header">
				<h2 style="margin:0;">New Support Request - Glimbyte SSL Guardian</h2>
			</div>
			
			<div class="section">
				<h3>Contact Information</h3>
				<div class="info-row"><span class="label">Name:</span> ' . esc_html( $data['name'] ) . '</div>
				<div class="info-row"><span class="label">Email:</span> <a href="mailto:' . esc_attr( $data['email'] ) . '">' . esc_html( $data['email'] ) . '</a></div>
				<div class="info-row"><span class="label">Phone:</span> ' . ( ! empty( $data['phone'] ) ? esc_html( $data['phone'] ) : 'Not provided' ) . '</div>
			</div>
			
			<div class="section">
				<h3>Website Information</h3>
				<div class="info-row"><span class="label">Site URL:</span> <a href="' . esc_url( $data['site_url'] ) . '">' . esc_html( $data['site_url'] ) . '</a></div>
				<div class="info-row"><span class="label">WordPress Version:</span> ' . esc_html( $data['wp_version'] ) . '</div>
				<div class="info-row"><span class="label">PHP Version:</span> ' . esc_html( $data['php_version'] ) . '</div>
				<div class="info-row"><span class="label">Plugin Version:</span> ' . esc_html( $data['plugin_version'] ) . '</div>
			</div>
			
			<div class="section">
				<h3>SSL Status</h3>
				<div class="info-row"><span class="label">Current Status:</span> ' . esc_html( $data['ssl_status'] ) . '</div>
				<div class="info-row"><span class="label">Using HTTPS:</span> ' . esc_html( $data['is_https'] ) . '</div>
			</div>
			
			<div class="section">
				<h3>Message</h3>
				<div class="message-box">' . nl2br( esc_html( $data['message'] ) ) . '</div>
			</div>
			
			<div style="margin-top: 30px; padding: 15px; background: #e8f4f8; border-left: 4px solid #00a32a;">
				<strong>Consent:</strong> User has provided consent to be contacted.
			</div>
			
			<div style="margin-top: 30px; padding: 15px; font-size: 12px; color: #666; border-top: 2px solid #ddd;">
				<p>This request was submitted via Glimbyte SSL Guardian plugin.</p>
				<p>Sent from: ' . esc_html( $data['site_url'] ) . '</p>
			</div>
		</body>
		</html>
		';
		// phpcs:enable WordPress.WP.EnqueuedResources.NonEnqueuedStylesheet

		// Improved headers for HTML email.
		$admin_email = get_option( 'admin_email' );
		$site_name = get_bloginfo( 'name' );
		
		$headers = array(
			'From: ' . $site_name . ' <' . $admin_email . '>',
			'Reply-To: ' . $data['email'],
			'Content-Type: text/html; charset=UTF-8',
			'X-Mailer: Glimbyte-SSL-Guardian/' . GLIMBYTE_SSL_VERSION,
		);

		return wp_mail( $to, $subject, $email_message, $headers );
	}
}
