<?php
/**
 * Plugin Name: Glimbyte SSL Guardian
 * Description: Complete SSL management: force HTTPS, fix mixed content, monitor certificates. Lightweight and focused.
 * Version: 1.1.2
 * Author: Ardit Caushi
 * Author URI: https://glimbyte.ie
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: glimbyte-ssl-guardian
 * Domain Path: /languages
 * Requires at least: 5.0
 * Requires PHP: 7.4
 *
 * @package Glimbyte_SSL_Guardian
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Define plugin constants.
define( 'GLIMBYTE_SSL_VERSION', '1.1.2' );
define( 'GLIMBYTE_SSL_PLUGIN_FILE', __FILE__ );
define( 'GLIMBYTE_SSL_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'GLIMBYTE_SSL_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'GLIMBYTE_SSL_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );

/**
 * Main plugin class.
 */
class Glimbyte_SSL_Manager {

	/**
	 * Plugin instance.
	 *
	 * @var Glimbyte_SSL_Manager
	 */
	private static $instance = null;

	/**
	 * Get plugin instance.
	 *
	 * @return Glimbyte_SSL_Manager
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor.
	 */
	private function __construct() {
		$this->load_dependencies();
		$this->init_hooks();
	}

	/**
	 * Load required files.
	 */
	private function load_dependencies() {
		// Helper functions.
		require_once GLIMBYTE_SSL_PLUGIN_DIR . 'includes/functions.php';

		// Core classes.
		require_once GLIMBYTE_SSL_PLUGIN_DIR . 'includes/class-ssl-checker.php';
		require_once GLIMBYTE_SSL_PLUGIN_DIR . 'includes/class-https-redirect.php';
		require_once GLIMBYTE_SSL_PLUGIN_DIR . 'includes/class-mixed-content.php';
		require_once GLIMBYTE_SSL_PLUGIN_DIR . 'includes/class-hsts-manager.php';
		require_once GLIMBYTE_SSL_PLUGIN_DIR . 'includes/class-logger.php';

		// Admin classes.
		if ( is_admin() ) {
			require_once GLIMBYTE_SSL_PLUGIN_DIR . 'includes/class-admin.php';
			require_once GLIMBYTE_SSL_PLUGIN_DIR . 'includes/class-dashboard-widget.php';
		}

		// Multisite class.
		if ( is_multisite() ) {
			require_once GLIMBYTE_SSL_PLUGIN_DIR . 'includes/class-multisite.php';
		}
	}

	/**
	 * Initialize WordPress hooks.
	 */
	private function init_hooks() {
		// Initialize components.
		add_action( 'plugins_loaded', array( $this, 'init_components' ) );

		// Activation/deactivation hooks.
		register_activation_hook( __FILE__, array( $this, 'activate' ) );
		register_deactivation_hook( __FILE__, array( $this, 'deactivate' ) );
	}

	/**
	 * Initialize plugin components.
	 */
	public function init_components() {
		// Initialize HTTPS redirect.
		if ( glimbyte_ssl_get_option( 'https_enabled', false ) ) {
			Glimbyte_SSL_HTTPS_Redirect::get_instance();
		}

		// Initialize mixed content fixer.
		if ( glimbyte_ssl_get_option( 'mixed_content_fix_enabled', false ) ) {
			Glimbyte_SSL_Mixed_Content::get_instance();
		}

		// Initialize HSTS manager.
		if ( glimbyte_ssl_get_option( 'hsts_enabled', false ) ) {
			Glimbyte_SSL_HSTS_Manager::get_instance();
		}

		// Initialize SSL checker (for monitoring).
		Glimbyte_SSL_Checker::get_instance();

		// Initialize admin interface.
		if ( is_admin() ) {
			Glimbyte_SSL_Admin::get_instance();
			Glimbyte_SSL_Dashboard_Widget::get_instance();
		}

		// Initialize multisite if applicable.
		if ( is_multisite() ) {
			Glimbyte_SSL_Multisite::get_instance();
		}
	}

	/**
	 * Plugin activation.
	 */
	public function activate() {
		// Set default options.
		$defaults = array(
			'https_enabled'              => false,
			'redirect_type'              => '301',
			'force_all'                  => true,
			'excluded_urls'              => '',
			'mixed_content_fix_enabled'  => false,
			'mixed_content_method'       => 'https',
			'hsts_enabled'               => false,
			'hsts_max_age'               => '31536000',
			'hsts_subdomains'            => false,
			'hsts_preload'               => false,
			'cert_alert_enabled'         => true,
			'cert_alert_days'            => '30,14,7',
			'cert_alert_email'           => get_option( 'admin_email' ),
			'debug_mode'                 => false,
		);

		foreach ( $defaults as $key => $value ) {
			if ( false === get_option( 'glimbyte_ssl_' . $key ) ) {
				add_option( 'glimbyte_ssl_' . $key, $value );
			}
		}

		// Schedule certificate check cron.
		if ( ! wp_next_scheduled( 'glimbyte_ssl_check_certificate' ) ) {
			wp_schedule_event( time(), 'daily', 'glimbyte_ssl_check_certificate' );
		}

		// Flush rewrite rules.
		flush_rewrite_rules();
	}

	/**
	 * Plugin deactivation.
	 */
	public function deactivate() {
		// Clear scheduled cron jobs.
		wp_clear_scheduled_hook( 'glimbyte_ssl_check_certificate' );

		// Flush rewrite rules.
		flush_rewrite_rules();
	}
}

/**
 * Initialize the plugin.
 */
function glimbyte_ssl_init() {
	return Glimbyte_SSL_Manager::get_instance();
}

// Start the plugin.
glimbyte_ssl_init();
