<?php
/**
 * Dashboard Tab View
 *
 * @package Glimbyte_SSL_Manager
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Template variables - passed from parent scope, not globals.
// phpcs:disable WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound

// Get SSL certificate info.
$cert_info = glimbyte_ssl_get_cert_info();
$is_https = glimbyte_ssl_is_https();
$https_enabled = glimbyte_ssl_get_option( 'https_enabled', false );
$mixed_content_enabled = glimbyte_ssl_get_option( 'mixed_content_fix_enabled', false );
$hsts_enabled = glimbyte_ssl_get_option( 'hsts_enabled', false );

// Determine if we should highlight support section.
$show_support_alert = false;
$support_message = '';

if ( is_wp_error( $cert_info ) ) {
	$show_support_alert = true;
	$support_message = __( 'Having SSL certificate issues? Our team can help you install and configure SSL properly.', 'glimbyte-ssl-guardian' );
} elseif ( ! $is_https ) {
	$show_support_alert = true;
	$support_message = __( 'Need help setting up SSL for your website? Get professional SSL installation from our team.', 'glimbyte-ssl-guardian' );
} elseif ( ! empty( $cert_info ) && $cert_info['days_left'] < 30 ) {
	$show_support_alert = true;
	$support_message = __( 'Your SSL certificate is expiring soon. Let us help you renew it before it expires.', 'glimbyte-ssl-guardian' );
}

if ( ! $show_support_alert ) {
	$support_message = __( 'Need help with SSL management, security, or WordPress development? Contact our expert team.', 'glimbyte-ssl-guardian' );
}

// phpcs:enable WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound
?>


<div class="ssl-dashboard-container">
	
	<!-- SSL Status Overview -->
	<div class="ssl-dashboard-card" style="<?php echo ( $is_https && ! is_wp_error( $cert_info ) && $cert_info['is_valid'] ) ? 'border-left: 4px solid #00a32a;' : 'border-left: 4px solid #d63638;'; ?>">
		<h2 style="margin: 0 0 10px; font-size: 24px;">
			<?php
			if ( $is_https && ! is_wp_error( $cert_info ) && $cert_info['is_valid'] ) {
				esc_html_e( 'SSL Active & Valid', 'glimbyte-ssl-guardian' );
			} elseif ( is_wp_error( $cert_info ) ) {
				esc_html_e( 'SSL Error Detected', 'glimbyte-ssl-guardian' );
			} else {
				esc_html_e( 'SSL Not Active', 'glimbyte-ssl-guardian' );
			}
			?>
		</h2>
		<p style="margin: 0; color: #50575e;">
			<?php
			if ( $is_https && ! is_wp_error( $cert_info ) && $cert_info['is_valid'] ) {
				printf(
					/* translators: %d: Number of days (numeric value) until the SSL certificate expires */
					esc_html__( 'Your SSL certificate is valid and will expire in %d days.', 'glimbyte-ssl-guardian' ),
					absint( $cert_info['days_left'] )
				);
			} elseif ( is_wp_error( $cert_info ) ) {
				echo esc_html( $cert_info->get_error_message() );
			} else {
				esc_html_e( 'Your site is not using HTTPS. Enable HTTPS redirect below to secure your site.', 'glimbyte-ssl-guardian' );
			}
			?>
		</p>
	</div>

	<!-- Two Column Grid -->
	<div class="ssl-dashboard-grid">
		
		<!-- Left Column -->
		<div>
			
			<!-- Quick Stats -->
			<div class="ssl-dashboard-card">
				<h3><?php esc_html_e( 'Quick Stats', 'glimbyte-ssl-guardian' ); ?></h3>
				<div style="display: flex; flex-direction: column; gap: 10px;">
					<div style="display: flex; justify-content: space-between; padding: 10px; background: #f6f7f7; border-radius: 4px;">
						<strong><?php esc_html_e( 'HTTPS Redirect', 'glimbyte-ssl-guardian' ); ?></strong>
						<span class="ssl-status-badge <?php echo $https_enabled ? 'enabled' : 'disabled'; ?>">
							<?php echo $https_enabled ? esc_html__( 'Enabled', 'glimbyte-ssl-guardian' ) : esc_html__( 'Disabled', 'glimbyte-ssl-guardian' ); ?>
						</span>
					</div>
					<div style="display: flex; justify-content: space-between; padding: 10px; background: #f6f7f7; border-radius: 4px;">
						<strong><?php esc_html_e( 'Mixed Content Fix', 'glimbyte-ssl-guardian' ); ?></strong>
						<span class="ssl-status-badge <?php echo $mixed_content_enabled ? 'enabled' : 'disabled'; ?>">
							<?php echo $mixed_content_enabled ? esc_html__( 'Enabled', 'glimbyte-ssl-guardian' ) : esc_html__( 'Disabled', 'glimbyte-ssl-guardian' ); ?>
						</span>
					</div>
					<div style="display: flex; justify-content: space-between; padding: 10px; background: #f6f7f7; border-radius: 4px;">
						<strong><?php esc_html_e( 'HSTS Enabled', 'glimbyte-ssl-guardian' ); ?></strong>
						<span class="ssl-status-badge <?php echo $hsts_enabled ? 'enabled' : 'disabled'; ?>">
							<?php echo $hsts_enabled ? esc_html__( 'Enabled', 'glimbyte-ssl-guardian' ) : esc_html__( 'Disabled', 'glimbyte-ssl-guardian' ); ?>
						</span>
					</div>
				</div>
			</div>

			<!-- Certificate Information -->
			<?php if ( ! is_wp_error( $cert_info ) ) : ?>
			<div class="ssl-dashboard-card" style="margin-top: 20px;">
				<h3><?php esc_html_e( 'Certificate Information', 'glimbyte-ssl-guardian' ); ?></h3>
				<table style="width: 100%;">
					<tr>
						<td style="padding: 8px 0; font-weight: 500;"><?php esc_html_e( 'Issued by:', 'glimbyte-ssl-guardian' ); ?></td>
						<td style="padding: 8px 0;"><?php echo esc_html( $cert_info['issuer'] ); ?></td>
					</tr>
					<tr>
						<td style="padding: 8px 0; font-weight: 500;"><?php esc_html_e( 'Valid until:', 'glimbyte-ssl-guardian' ); ?></td>
						<td style="padding: 8px 0;"><?php echo esc_html( glimbyte_ssl_format_date( $cert_info['valid_to'] ) ); ?></td>
					</tr>
					<tr>
						<td style="padding: 8px 0; font-weight: 500;"><?php esc_html_e( 'Days remaining:', 'glimbyte-ssl-guardian' ); ?></td>
						<td style="padding: 8px 0;">
							<?php
							$days_left = absint( $cert_info['days_left'] ) ; // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound
							if ( $days_left > 30 ) {
								echo '<strong style="color: #00a32a;">' . esc_html( $days_left ) . ' ' . esc_html__( 'days', 'glimbyte-ssl-guardian' ) . '</strong>';
							} elseif ( $days_left > 7 ) {
								echo '<strong style="color: #dba617;">' . esc_html( $days_left ) . ' ' . esc_html__( 'days', 'glimbyte-ssl-guardian' ) . '</strong>';
							} else {
								echo '<strong style="color: #d63638;">' . esc_html( $days_left ) . ' ' . esc_html__( 'days', 'glimbyte-ssl-guardian' ) . '</strong>';
							}
							?>
						</td>
					</tr>
				</table>
			</div>
			<?php endif; ?>

		</div>

		<!-- Right Column -->
		<div>
			
			<!-- Quick Actions -->
			<div class="ssl-dashboard-card">
				<h3><?php esc_html_e( 'Quick Actions', 'glimbyte-ssl-guardian' ); ?></h3>
				<form method="post" action="options.php">
					<?php settings_fields( 'glimbyte_ssl_https' ); ?>
					<input type="hidden" name="glimbyte_ssl_https_enabled" value="<?php echo $https_enabled ? '0' : '1'; ?>">
					<?php if ( $https_enabled ) : ?>
						<button type="submit" class="button button-secondary ssl-button-block">
							<span class="dashicons dashicons-lock"></span>
							<?php esc_html_e( 'Disable HTTPS Redirect', 'glimbyte-ssl-guardian' ); ?>
						</button>
					<?php else : ?>
						<button type="submit" class="button button-primary ssl-button-block">
							<span class="dashicons dashicons-lock"></span>
							<?php esc_html_e( 'Enable HTTPS Redirect', 'glimbyte-ssl-guardian' ); ?>
						</button>
					<?php endif; ?>
				</form>

				<a href="<?php echo esc_url( glimbyte_ssl_get_admin_url( 'mixed-content' ) ); ?>" class="button ssl-button-block">
					<span class="dashicons dashicons-search"></span>
					<?php esc_html_e( 'Scan for Mixed Content', 'glimbyte-ssl-guardian' ); ?>
				</a>

				<a href="<?php echo esc_url( glimbyte_ssl_get_admin_url( 'certificate' ) ); ?>" class="button ssl-button-block">
					<span class="dashicons dashicons-admin-network"></span>
					<?php esc_html_e( 'View Certificate Details', 'glimbyte-ssl-guardian' ); ?>
				</a>
			</div>

			<!-- Get Professional Help -->
			<div class="ssl-dashboard-card" style="margin-top: 20px; <?php echo $show_support_alert ? 'border-left: 4px solid #d63638;' : ''; ?>">
				<h3>
					<span class="dashicons dashicons-sos"></span>
					<?php esc_html_e( 'Get Professional Help', 'glimbyte-ssl-guardian' ); ?>
				</h3>
				
				<?php if ( $show_support_alert ) : ?>
				<div class="ssl-support-alert">
					<strong><?php echo esc_html( $support_message ); ?></strong>
				</div>
				<?php else : ?>
				<p><?php echo esc_html( $support_message ); ?></p>
				<?php endif; ?>

				<button type="button" class="button button-primary ssl-button-block" id="ssl-show-support-btn">
					<span class="dashicons dashicons-email"></span>
					<?php esc_html_e( 'Request Support Now', 'glimbyte-ssl-guardian' ); ?>
				</button>

				<div id="ssl-support-form-container" style="display: none;">
					<form id="ssl-support-form">
						<label style="font-weight: 600;"><?php esc_html_e( 'Your Name *', 'glimbyte-ssl-guardian' ); ?></label>
						<input type="text" name="name" required>

						<label style="font-weight: 600;"><?php esc_html_e( 'Email Address *', 'glimbyte-ssl-guardian' ); ?></label>
						<input type="email" name="email" required>

						<label style="font-weight: 600;"><?php esc_html_e( 'Phone Number', 'glimbyte-ssl-guardian' ); ?></label>
						<input type="tel" name="phone">

						<label style="font-weight: 600;"><?php esc_html_e( 'How Can We Help? *', 'glimbyte-ssl-guardian' ); ?></label>
						<textarea name="message" rows="4" required></textarea>

						<label>
							<input type="checkbox" name="consent" required style="width: auto;">
							<?php esc_html_e( 'I consent to Glimbyte Ltd contacting me regarding this support request *', 'glimbyte-ssl-guardian' ); ?>
						</label>

						<div style="background: #e7f5fe; border-left: 4px solid #2271b1; padding: 10px; margin: 15px 0; border-radius: 4px;">
							<small><?php esc_html_e( 'Your website URL, WordPress version, SSL status, and technical details will be automatically included to help us assist you better.', 'glimbyte-ssl-guardian' ); ?></small>
						</div>

						<button type="submit" class="button button-primary" style="margin-right: 10px;">
							<span class="dashicons dashicons-email"></span>
							<?php esc_html_e( 'Send Support Request', 'glimbyte-ssl-guardian' ); ?>
						</button>
						<button type="button" class="button" id="ssl-cancel-support-btn">
							<?php esc_html_e( 'Cancel', 'glimbyte-ssl-guardian' ); ?>
						</button>
					</form>
					<div id="ssl-support-form-result" style="display: none; margin-top: 15px; padding: 12px; border-radius: 4px;"></div>
				</div>
			</div>

		</div>
	</div>

</div>

