<?php
/**
 * Settings page
 *
 * @package Gift_Card_Manager
 */

namespace GIFT_CARD_MANAGER;

/**
 * Handler for "admin_menu" action
 */
function gift_card_manager_add_settings_page() {
	add_submenu_page( 'woocommerce-marketing', 'Gift Card Manager for WooCommerce', 'Gift Cards', 'manage_options', 'gift-card-manager', gift_card_manager_concat( __NAMESPACE__, '\\gift_card_manager_render_plugin_settings_page' ) );
}

/**
 * Generate Gift Card claimed log rows
 *
 * @param string $search Search query string to filter results.
 * @param int    $rpp Rows per page.
 * @param int    $page Page number.
 */
function gift_card_manager_claimed_log( $search, $rpp, $page ) {
	$gift_cards = gift_card_manager_get_gift_cards_claimed();
	$search     = strtolower( trim( $search ) );
	$rpp        = max( 10, (int) $rpp );
	$page       = max( 1, (int) $page );
	$results    = array();
	foreach ( $gift_cards as $gift_card ) {
		// Put as much as possible non-HTML data into $data below, to maximize available data for the search function.
		$data = array(
			0 => $gift_card->code,
			1 => $gift_card->currency . ' ' . $gift_card->value,
			2 => $gift_card->group ?? 'N/A',
			3 => gift_card_manager_coalesce( strtolower( get_user_by( 'id', $gift_card->claim_user_id )->user_email ?? '' ), 'N/A' ),
			4 => $gift_card->create_time ? wp_date( 'Y-m-d', $gift_card->create_time ) : '',
			5 => $gift_card->claim_time ? wp_date( 'Y-m-d', $gift_card->claim_time ) : '',
		);

		if ( $search && ! count( array_filter( $data, fn( $data ) => strpos( strtolower( $data ), $search ) !== false ) ) ) {
			continue; // Filter result out.
		}

		$results[] = gift_card_manager_concat(
			'<tr><td>',
			$data[0],
			'</td><td>',
			$data[1],
			'</td><td>',
			$data[2],
			'</td><td>',
			$data[3],
			'</td><td>',
			$data[4],
			'</td><td>',
			$data[5],
			'</td></tr>'
		);
	}
	if ( count( $results ) ) {
		$offset = max( 0, min( floor( count( $results ) / $rpp ) * $rpp, ( $page - 1 ) * $rpp ) );
		return array(
			'data'        => implode( "\n", array_slice( $results, $offset, $rpp ) ),
			'page'        => (int) ( $offset / $rpp + 1 ),
			'total_pages' => (int) ( ( count( $results ) - 1 ) / $rpp + 1 ),
		);
	} else {
		return array(
			'data'        => '<tr><td colspan="99">No Claimed Gift Cards in the specified range.</td></tr>',
			'page'        => 1,
			'total_pages' => 1,
		);
	}
}

/**
 * Handler for Ajax action - Claimed Gift Cards log pagination
 */
function gift_card_manager_claimed_log_ajax() {
	check_ajax_referer( 'ajax' );
	gift_card_manager_validate_ajax();

	$request_id = sanitize_text_field( wp_unslash( $_POST['request_id'] ?? '' ) ); // phpcs:ignore
	$search     = sanitize_text_field( wp_unslash( $_POST['search'] ?? '' ) ); // phpcs:ignore
	$rpp        = sanitize_text_field( wp_unslash( $_POST['rpp'] ?? '' ) ); // phpcs:ignore
	$page       = sanitize_text_field( wp_unslash( $_POST['page'] ?? '' ) ); // phpcs:ignore

	header( 'Content-Type: application/json' );
	die( wp_json_encode( array_merge( array( 'request_id' => $request_id ), gift_card_manager_claimed_log( $search, $rpp, $page ) ) ) );
}

/**
 * Handler for Ajax action - Delete Gift Card
 */
function gift_card_manager_delete_ajax() {
	check_ajax_referer( 'ajax' );
	gift_card_manager_validate_ajax();

	$id = sanitize_text_field( wp_unslash( $_POST['id'] ?? '' ) ); // phpcs:ignore

	if ( $id ) {
		gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( $wpdb->prepare( "UPDATE {$wpdb->prefix}fg_gift_card_manager_gift_cards SET delete_time = %d WHERE id = %d AND delete_time IS NULL", gift_card_manager_time(), $id ) ) ); // phpcs:ignore
		die( 'Gift Card deleted' );
	}
	http_response_code( 500 );
	die( 'Invalid Gift Card ID' );
}

/**
 * Handler for Ajax action - Delete All (Selected in UI) Gift Cards
 */
function gift_card_manager_delete_all_ajax() {
	check_ajax_referer( 'ajax' );
	gift_card_manager_validate_ajax();

	$ids = explode( ',', sanitize_text_field( wp_unslash( $_POST['ids'] ?? '' ) ) ); // phpcs:ignore

	if ( $ids ) {
		foreach ( $ids as $id ) {
			gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( $wpdb->prepare( "UPDATE {$wpdb->prefix}fg_gift_card_manager_gift_cards SET delete_time = %d WHERE id = %d AND delete_time IS NULL", gift_card_manager_time(), $id ) ) ); // phpcs:ignore
		}
		die( 'Gift Cards deleted' );
	}
	http_response_code( 500 );
	die( 'Invalid Gift Card IDs' );
}

/**
 * Generate Gift Card expired log rows
 *
 * @param string $search Search query string to filter results.
 * @param int    $rpp Rows per page.
 * @param int    $page Page number.
 */
function gift_card_manager_expired_log( $search, $rpp, $page ) {
	$gift_cards = gift_card_manager_get_gift_cards_expired();
	$search     = strtolower( trim( $search ) );
	$rpp        = max( 10, (int) $rpp );
	$page       = max( 1, (int) $page );
	$results    = array();
	foreach ( $gift_cards as $gift_card ) {
		// Put as much as possible non-HTML data into $data below, to maximize available data for the search function.
		$data = array(
			0 => $gift_card->code,
			1 => $gift_card->currency . ' ' . $gift_card->value,
			2 => $gift_card->group ?? 'N/A',
			3 => $gift_card->create_time ? wp_date( 'Y-m-d', $gift_card->create_time ) : '',
			4 => $gift_card->expire_time ? wp_date( 'Y-m-d H:i:s', $gift_card->expire_time ) : '',
		);

		if ( $search && ! count( array_filter( $data, fn( $data ) => strpos( strtolower( $data ), $search ) !== false ) ) ) {
			continue; // Filter result out.
		}

		$results[] = gift_card_manager_concat(
			'<tr><td class="check"><input type="checkbox" class="fg-gcm-checkbox">',
			'</td><td>',
			$data[0],
			'</td><td>',
			$data[1],
			'</td><td>',
			$data[2],
			'</td><td>',
			$data[3],
			'</td><td>',
			$data[4],
			'</td><td>',
			'<button class="btnDeleteExpired fg-gcm-button fg-gcm-button--x" title="Delete Gift Card" data-id="',
			$gift_card->id,
			'">x</button>&nbsp;<span class="status"></span>',
			'</td></tr>'
		);
	}
	if ( count( $results ) ) {
		$offset = max( 0, min( floor( count( $results ) / $rpp ) * $rpp, ( $page - 1 ) * $rpp ) );
		return array(
			'data'        => implode( "\n", array_slice( $results, $offset, $rpp ) ),
			'page'        => (int) ( $offset / $rpp + 1 ),
			'total_pages' => (int) ( ( count( $results ) - 1 ) / $rpp + 1 ),
		);
	} else {
		return array(
			'data'        => '<tr><td colspan="99">No Expired Gift Cards in the specified range.</td></tr>',
			'page'        => 1,
			'total_pages' => 1,
		);
	}
}

/**
 * Handler for Ajax action - Expired Gift Cards log pagination
 */
function gift_card_manager_expired_log_ajax() {
	check_ajax_referer( 'ajax' );
	gift_card_manager_validate_ajax();

	$request_id = sanitize_text_field( wp_unslash( $_POST['request_id'] ?? '' ) ); // phpcs:ignore
	$search     = sanitize_text_field( wp_unslash( $_POST['search'] ?? '' ) ); // phpcs:ignore
	$rpp        = sanitize_text_field( wp_unslash( $_POST['rpp'] ?? '' ) ); // phpcs:ignore
	$page       = sanitize_text_field( wp_unslash( $_POST['page'] ?? '' ) ); // phpcs:ignore

	header( 'Content-Type: application/json' );
	die( wp_json_encode( array_merge( array( 'request_id' => $request_id ), gift_card_manager_expired_log( $search, $rpp, $page ) ) ) );
}

/**
 * Generate Gift Cards
 */
function gift_card_manager_generate_ajax() {
	global $wpdb;

	check_ajax_referer( 'ajax' );
	gift_card_manager_validate_ajax();

	$account_limit = sanitize_text_field( wp_unslash( $_POST['account_limit'] ?? '' ) ); // phpcs:ignore
	$amount        = sanitize_text_field( wp_unslash( $_POST['amount'] ?? '' ) ); // phpcs:ignore
	$currency      = sanitize_text_field( wp_unslash( $_POST['currency'] ?? '' ) ); // phpcs:ignore
	$expire        = sanitize_text_field( wp_unslash( $_POST['expire'] ?? '' ) ); // phpcs:ignore
	$format        = trim( preg_replace( '/^-+|-+$/', '', sanitize_text_field( wp_unslash( $_POST['format'] ?? '' ) ) ) ); // phpcs:ignore
	$group         = trim( sanitize_text_field( wp_unslash( $_POST['group'] ?? '' ) ) ); // phpcs:ignore
	$value         = sanitize_text_field( wp_unslash( $_POST['value'] ?? '' ) ); // phpcs:ignore

	$expire_components = explode( '-', $expire ?? '' );

	if ( ! $format ) {
		$format = 'aaaa-aaaa-aaaa';
	}
	if ( ! $expire ) {
		$expire = null;
	}
	if ( ! $amount ) {
		$amount = 1;
	} elseif ( strlen( $format ) < 8 || strlen( $format ) > 32 ) {
		die( 'Invalid format length, must be between 8-32 characters.' );
	} elseif ( preg_match( '/[^alnA-Z0-9\-]/', $format ) ) {
		die( 'Invalid format, must only contain the following characters: l n a A-Z 0-9 -' );
	} elseif ( strlen( preg_replace( '/[^\-]/', '', $format ) ) > 3 ) {
		die( 'Invalid format, must not contain more than 3 dash (-) characters.' );
	} elseif ( strpos( $format, '--' ) !== false ) {
		die( 'Invalid format, must not contain 2 consecutive dash characters.' );
	} elseif ( ! $value || ! is_numeric( $value ) || $value <= 0 ) {
		die( 'Invalid value, must be positive number.' );
	} elseif ( preg_match( '/\D/', (string) $amount ) || ! is_numeric( $amount ) || $amount < 0 ) {
		die( 'Invalid amount, must be positive integer number.' );
	} elseif ( $expire && $expire_components && ( 3 !== count( $expire_components ) || ! checkdate( $expire_components[1], $expire_components[2], $expire_components[0] ) ) ) {
		die( 'Invalid expire date, format is YYYY-MM-DD.' );
	} elseif ( $group && strlen( trim( $group ) ) > 128 ) {
		die( 'Invalid group name, must be between 1-128 characters.' );
	} elseif ( $account_limit && ( preg_match( '/\D/', (string) $amount ) || ! is_numeric( $account_limit ) || $account_limit < 0 ) ) {
		die( 'Invalid account limit, must be empty or positive integer number.' );
	}

	$options_json                 = gift_card_manager_options();
	$options_json['accountLimit'] = $account_limit;
	$options_json['amount']       = $amount;
	$options_json['currency']     = $currency;
	$options_json['expire']       = $expire;
	$options_json['format']       = $format;
	$options_json['group']        = $group;
	$options_json['value']        = $value;
	gift_card_manager_set_config( 'options', wp_json_encode( $options_json ) );

	$group_id = null;
	if ( $group || $account_limit ) {
		if ( ! $group ) {
			$group = 'Group_' . wp_date( 'Y_m_d_H_i_s' );
		}
		gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( $wpdb->prepare( "INSERT INTO {$wpdb->prefix}fg_gift_card_manager_gift_card_groups (name, account_limit, create_time) VALUES (%s, %d, %d)", $group, $account_limit, gift_card_manager_time() ) ) ); // phpcs:ignore
		$group_id = $wpdb->insert_id;
	}

	for ( $i = 0; $i < $amount; ++$i ) {
		gift_card_manager_generate( $format, $currency, $value, $group_id, $expire );
	}

	$suffix = $amount > 1 ? 's' : '';
	die( esc_html( $wpdb->last_error ? ( "Failed to generate Gift Card$suffix. See server logs for more information." /*. " {$wpdb->last_error}" */ ) : "$amount Gift Card$suffix generated successfully. See Unclaimed Gift Cards below." ) );
}

/**
 * Check if group name already exists
 */
function gift_card_manager_group_name_exists_ajax() {
	check_ajax_referer( 'ajax' );
	gift_card_manager_validate_ajax();

	$group = strtolower( trim( sanitize_text_field( wp_unslash( $_POST['group'] ?? '' ) ) ) ); // phpcs:ignore
	$existing_group = gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( $wpdb->prepare( "SELECT name FROM {$wpdb->prefix}fg_gift_card_manager_gift_card_groups WHERE LOWER(name) = %s", $group ) ) )[0]->name ?? ''; // phpcs:ignore
	die( esc_html( $existing_group ) );
}

/**
 * Check if current user is an administrator and can manage options
 *
 * @param bool $is_ajax True if current request is Ajax.
 */
function gift_card_manager_is_admin( $is_ajax ) {
	return in_array( 'administrator', wp_get_current_user()->roles ?? array(), true ) && ( $is_ajax || current_user_can( 'manage_options' ) );
}

/**
 * Handler for call to "add_submenu_page"
 */
function gift_card_manager_render_plugin_settings_page() {
	if ( ! gift_card_manager_is_admin( false ) ) {
		wp_die();
	}

	$options                 = gift_card_manager_options();
	$checked                 = fn( $property ) => ( $options[ $property ] ?? true ) ? ' checked' : '';
	$display                 = fn( $property ) => ( $options[ $property ] ?? true ) ? '' : ' style="display: none"';
	$option                  = fn( $property, $default_value ) => ( $options[ $property ] ?? null ) ?? $default_value;
	$plugin_dir              = plugin_dir_url( __FILE__ );
	$gift_card_unclaimed_log = gift_card_manager_unclaimed_log( '', 10, 1 );
	$gift_card_claimed_log   = gift_card_manager_claimed_log( '', 10, 1 );
	$gift_card_expired_log   = gift_card_manager_expired_log( '', 10, 1 );
	$report_send_time        = gift_card_manager_get_config( 'last_report_send_time' );

	echo '',
		'<div class="fg-gcm-body">',
			'<div class="fg-gcm-header">',
				'<img class="fg-gcm-plugin-logo" src="',
					esc_attr( plugin_dir_url( __FILE__ ) . 'img/logo.png' ),
				'">',
				'<nav class="fg-gcm-quick-nav">',
					'<ul class="fg-gcm-quick-nav__list">',
						'<li><a class="fg-gcm-quick-nav__link" href="#generate">Generate Gift Cards</a></li>',
						'<li><a class="fg-gcm-quick-nav__link" href="#settings">Settings</a></li>',
						'<li><a class="fg-gcm-quick-nav__link" href="#preview">Preview shortcode</a></li>',
						'<li><a class="fg-gcm-quick-nav__link" href="#unclaimed">Unclaimed Gift Cards</a></li>',
						'<li><a class="fg-gcm-quick-nav__link" href="#expired">Expired Gift Cards</a></li>',
						'<li><a class="fg-gcm-quick-nav__link" href="#claimed">Claimed Gift Cards</a></li>',
					'</ul>',
				'</nav>',

				'<div class="fg-gcm-quick-nav-save">',
					'<p class="saveStatus"></p>',
				'</div>',
			'</div>',

			'<section class="fg-gcm-header__intro">',
				'<h2 class="fg-gcm-header__sub-title"></h2>',
				'<p>Welcome to the settings page of our WordPress plugin! Here, you can configure various options to manage Gift Cards on your WooCommerce site. The settings available include:</p>',
				'<ul>',
					'<li><a class="fg-gcm-quick-nav__link" href="#generate">Generate Gift Cards:</a> Create new Gift Cards for WooCommerce.</li>',
					'<li><a class="fg-gcm-quick-nav__link" href="#settings">Settings:</a> Configure plugin options.</li>',
					'<li><a class="fg-gcm-quick-nav__link" href="#preview">Preview shortcode:</a> Preview shortcode.</li>',
					'<li><a class="fg-gcm-quick-nav__link" href="#unclaimed">Unclaimed Gift Cards:</a> Maintain and manage Unclaimed Gift Cards.</li>',
					'<li><a class="fg-gcm-quick-nav__link" href="#claimed">Claimed Gift Cards:</a> View Claimed Gift Cards.</li>',
					'<li><a class="fg-gcm-quick-nav__link" href="#expired">Expired Gift Cards:</a> Maintain and manage Expired Gift Cards.</li>',
				'</ul>',
				'<p>Use these settings to effectively manage Gift Cards for your WooCommerce site. For more info, <a href="https://woogiftcard.com" target="_blank">visit our website</a>.</p>',
			'</section>',

			'<section id="generate" class="fg-gcm-back-panel">',
				'<h2 class="fg-gcm-section-header">Create new Gift Cards for WooCommerce</h2>',
				'<p>Customise automatic generation of Gift Cards. Options include Code Format, Gift Card Value, Expire Date and Amount of new cards to generate.</p>',
				'<p>The Gift Card Code Format can contain the following format characters, each having the indicated meaning: l (capital letter), n (number), a (l or n), - (dash character), other A-Z and 0-9 characters are inserted as is</p>',
				'<table class="fg-gcm-table">',
					'<tr><td class="fg-gcm-row-title">Code Format:</td><td><input type="text" id="format" class="format fg-gcm-input fg-gcm-input--code-format" placeholder="Default aaaa-aaaa-aaaa, length from 8 to 32, allowed: l n a A-Z 0-9 - (max 3 dashes)" onkeyup="giftCardManagerValidate( \'format\' )" value="',
					esc_attr( gift_card_manager_option( 'format' ) ),
					'"></td></tr>',
					'<tr><td class="fg-gcm-row-title">Gift Card Currency:</td><td class="fg-gcm-input"><select id="currency"><option disabled>Select a currency</option>',
					implode( '', array_map( fn( $currency ) => gift_card_manager_concat( '<option value="', esc_attr( $currency ), '"', esc_attr( gift_card_manager_option( 'currency' ) == $currency ? ' selected' : ''), '>', esc_html( $currency ), ' (', esc_html( get_woocommerce_currencies()[ $currency ] ), ')', '</option>' ), array_keys( get_woocommerce_currencies() ) ) ), // phpcs:ignore
					'</select></td></tr>',
					'<tr><td class="fg-gcm-row-title">Gift Card Value:</td><td><input type="text" id="value" class="value fg-gcm-input" placeholder="E.g. 100" onkeyup="giftCardManagerValidate( \'value\' )" value="',
					esc_attr( gift_card_manager_option( 'value', 1000 ) ),
					'"></td></tr>',
					'<tr><td class="fg-gcm-row-title">Expire Date:</td><td><input type="text" id="expire" class="expire fg-gcm-input" placeholder="YYYY-MM-DD" onkeyup="giftCardManagerValidate( \'expire\' )" value="',
					esc_attr( gift_card_manager_option( 'expire' ) ),
					'"></td></tr>',
					'<tr><td class="fg-gcm-row-title">Amount of cards to generate:</td><td><input type="text" id="amount" class="amount fg-gcm-input" placeholder="E.g. 50" onkeyup="giftCardManagerValidate( \'amount\' )" value="',
					esc_attr( gift_card_manager_option( 'amount', 10 ) ),
					'"></td></tr>',
					'<tr><td class="fg-gcm-row-title">Group Name:</td><td><input type="text" id="group" class="group fg-gcm-input" placeholder="Group Name, or leave empty" onkeyup="giftCardManagerValidate( \'group\' )" value="',
					esc_attr( gift_card_manager_option( 'group', '' ) ),
					'"><br><span id="groupExists" class="fg-gcm-validate-error"></span></td></tr>',
					'<tr><td class="fg-gcm-row-title">Account Limit:</td><td><input type="text" id="account-limit" class="group fg-gcm-input" placeholder="E.g. 1 (means max 1 claim from this group for any user), or leave empty" onkeyup="giftCardManagerValidate( \'accountLimit\' )" value="',
					esc_attr( gift_card_manager_option( 'accountLimit', '' ) ),
					'"></td></tr>',
				'</table>',
				'<p><button id="btnGenerate" class="fg-gcm-button fg-gcm-button--save fg-gcm__button">Generate Gift Cards</button>&nbsp;<span id="generateStatus"></p>',
			'</section>',

			'<section id="settings" class="fg-gcm-back-panel">',
				'<h2 class="fg-gcm-section-header">Settings</h2>',
				'<p>Configure plugin options.</p>',
				'<table class="fg-gcm-table">',
					'<tr>',
						'<td>Default email address:</td><td><input type="text" class="fg-gcm-input fg-gcm-input--default-email" value="',
						esc_attr( strtolower( gift_card_manager_admin_email() ) ),
						'" disabled></td>',
					'</tr>',
					'<tr>',
						'<td>Custom report email address:</td><td><input type="text" placeholder="Enter optional alternative email address to use instead of default email address above" class="monitor-email fg-gcm-input fg-gcm-input--monitor-email" onkeyup="giftCardManagerValidate( \'monitorEmail\' )" value="',
						esc_attr( strtolower( $option( 'monitorEmail', '' ) ) ),
						'">',
						'</td>',
					'</tr>',
					'<tr>',
						'<td>Send daily claim report email:</td><td><input id="reportEnabled" type="checkbox" class="report-enabled fg-gcm-checkbox fg-gcm-checkbox--report-enabled"',
						esc_html( ( true === ( $options['reportEnabled'] ?? true ) ? ' checked' : '' ) ),
						'>',
						'</td>',
					'</tr>',
					'<tr>',
						'<td>Force daily report email:</td><td><input id="reportForce" type="checkbox" class="report-force fg-gcm-checkbox fg-gcm-checkbox--report-force"',
							esc_html( ( true === ( $options['reportForce'] ?? true ) ? ' checked' : '' ) ),
							'>',
						'</td>',
					'</tr>',
					'<tr>',
						'<td>Last report email:</td><td>',
							esc_html( $report_send_time ? wp_date( 'Y-m-d H:i', $report_send_time ) : 'N/A' ),
						'</td>',
					'</tr>',
					'<tr>',
						'<td>Next report email:</td><td>',
							esc_html( wp_date( 'Y-m-d H:i', wp_next_scheduled( 'gift_card_manager_cron_job_report_email' ) ) ),
						'</td>',
					'</tr>',
				'</table>',
				'<p>',
					'<button id="btnSaveSettings" class="fg-gcm-button fg-gcm-gift_card_manager__button">Save Settings</button>&nbsp;<button id="btnReportEmail" class="fg-gcm-button fg-gcm-gift_card_manager__button">Send report email now</button>',
				'</p>',
				'<p>',
					'<span class="saveStatus"></span>&nbsp;<span id="reportEmailStatus" class="fg-gcm-status"></span>',
				'</p>',
			'</section>',

			'<section id="preview" class="fg-gcm-back-panel">',
				'<h2 class="fg-gcm-section-header">Preview Shortcode</h2>',
				'<p>Preview and test the functionality of the [gcm-claim] shortcode on your site.</p>',
				'<div id="previewArea"></div>
				<button id="btnPreview" class="fg-gcm-button fg-gcm-gift_card_manager__button">Click to preview shortcode</button></p>',
			'</section>',

			'<section id="unclaimed" class="fg-gcm-back-panel">',
				'<div class="fg-gcm-row-header">',
					'<h2 class="fg-gcm-section-header">Unclaimed Gift Cards</h2>',
					'<p>Maintain and manage Unclaimed Gift Cards.',
				'</div>',

				'<div class="fg-gcm-row">',
					'<table class="fg-gcm-table">',
						'<tr>',
							'<td class="fg-gcm-row-title" style="text-align: left">Rows per page: <select class="ddRpp"><option value="10">10</option><option value="20">20</option><option value="50">50</option><option value="100">100</option><option value="200">200</option><option value="all">All</option></select>&nbsp;Go to page: <input class="txtPage fg-gcm-input" size="5"></td>',
							'<td style="text-align: right">Search: <input class="txtSearch" class="fg-gcm-input"></td>',
						'</tr>',
					'</table>',
				'</div>',

				'<div>&nbsp;</div>',

				'<div class="fg-gcm-row">',
					'<table class="fg-gcm-table log">',
						'<thead>',
							'<tr>',
								'<th></th>',
								'<th>Code</th>',
								'<th>Value</th>',
								'<th>Group</th>',
								'<th>Create Date</th>',
								'<th>Expire Date</th>',
								'<th>Actions</th>',
							'</tr>',
						'</thead>',
						'<tbody>',
							wp_kses_post( $gift_card_unclaimed_log['data'] ),
						'</tbody>',
					'</table>',
				'</div>',

				'<div>&nbsp;</div>',

				'<div class="fg-gcm-row">',
					'<table class="fg-gcm-table">',
						'<tr>',
							'<td class="fg-gcm-row-title" style="text-align: left">',
								'<div class="page">Page 1 of ',
									esc_html( $gift_card_unclaimed_log['total_pages'] ),
								'</div>',
							'</td>',
							'<td style="text-align: right">',
								'<div class="fg-gcm-table-pagination-nav">',
									'<p id="unclaimedLogLoading" class="feedback" style="display: none">Loading...</p>',
									'<button class="btnDeleteSelected fg-gcm-button fg-gcm-gift_card_manager__button">Delete selected</button>',
									'<button class="btnPrev fg-gcm-button fg-gcm-gift_card_manager__button">Previous</button>',
									'<button class="btnNext fg-gcm-button fg-gcm-gift_card_manager__button">Next</button>',
								'</div>',
							'</td>',
						'</tr>',
					'</table>',
				'</div>',
			'</section>',

			'<section id="claimed" class="fg-gcm-back-panel">',
				'<div class="fg-gcm-row-header">',
					'<h2 class="fg-gcm-section-header">Claimed Gift Cards</h2>',
					'<p>View Claimed Gift Cards.',
				'</div>',

				'<div class="fg-gcm-row">',
					'<table class="fg-gcm-table">',
						'<tr>',
							'<td class="fg-gcm-row-title" style="text-align: left">Rows per page: <select class="ddRpp"><option value="10">10</option><option value="20">20</option><option value="50">50</option><option value="100">100</option><option value="200">200</option><option value="all">All</option></select>&nbsp;Go to page: <input class="txtPage fg-gcm-input" size="5"></td>',
							'<td style="text-align: right">Search: <input class="txtSearch" class="fg-gcm-input"></td>',
						'</tr>',
					'</table>',
				'</div>',

				'<div>&nbsp;</div>',

				'<div class="fg-gcm-row">',
					'<table class="fg-gcm-table log">',
						'<thead>',
							'<tr>',
								'<th>Code</th>',
								'<th>Value</th>',
								'<th>Group</th>',
								'<th>Email</th>',
								'<th>Create Date</th>',
								'<th>Claim Date</th>',
							'</tr>',
						'</thead>',
						'<tbody>',
							wp_kses_post( $gift_card_claimed_log['data'] ),
						'</tbody>',
					'</table>',
				'</div>',

				'<div>&nbsp;</div>',

				'<div class="fg-gcm-row">',
					'<table class="fg-gcm-table">',
						'<tr>',
							'<td class="fg-gcm-row-title" style="text-align: left">',
								'<div class="page">Page 1 of ',
									esc_html( $gift_card_claimed_log['total_pages'] ),
								'</div>',
							'</td>',
							'<td style="text-align: right">',
								'<div class="fg-gcm-table-pagination-nav">',
									'<p class="feedback" style="display: none">Loading...</p>',
									'<button class="btnPrev fg-gcm-button fg-gcm-gift_card_manager__button">Previous</button>',
									'<button class="btnNext fg-gcm-button fg-gcm-gift_card_manager__button">Next</button>',
								'</div>',
							'</td>',
						'</tr>',
					'</table>',
				'</div>',
			'</section>',

			'<section id="expired" class="fg-gcm-back-panel">',
				'<div class="fg-gcm-row-header">',
					'<h2 class="fg-gcm-section-header">Expired Gift Cards</h2>',
					'<p>Maintain and manage Expired Gift Cards.',
				'</div>',

				'<div class="fg-gcm-row">',
					'<table class="fg-gcm-table">',
						'<tr>',
							'<td class="fg-gcm-row-title" style="text-align: left">Rows per page: <select class="ddRpp"><option value="10">10</option><option value="20">20</option><option value="50">50</option><option value="100">100</option><option value="200">200</option><option value="all">All</option></select>&nbsp;Go to page: <input class="txtPage fg-gcm-input" size="5"></td>',
							'<td style="text-align: right">Search: <input class="txtSearch" class="fg-gcm-input"></td>',
						'</tr>',
					'</table>',
				'</div>',

				'<div>&nbsp;</div>',

				'<div class="fg-gcm-row">',
					'<table class="fg-gcm-table log">',
						'<thead>',
							'<tr>',
								'<th></th>',
								'<th>Code</th>',
								'<th>Value</th>',
								'<th>Group</th>',
								'<th>Create Date</th>',
								'<th>Expire Date</th>',
								'<th>Actions</th>',
							'</tr>',
						'</thead>',
						'<tbody>',
							wp_kses_post( $gift_card_expired_log['data'] ),
						'</tbody>',
					'</table>',
				'</div>',

				'<div>&nbsp;</div>',

				'<div class="fg-gcm-row">',
					'<table class="fg-gcm-table">',
						'<tr>',
							'<td class="fg-gcm-row-title" style="text-align: left">',
								'<div class="page">Page 1 of ',
									esc_html( $gift_card_expired_log['total_pages'] ),
								'</div>',
							'</td>',
							'<td style="text-align: right">',
								'<div class="fg-gcm-table-pagination-nav">',
									'<p class="feedback" style="display: none">Loading...</p>',
									'<button class="btnDeleteSelected fg-gcm-button fg-gcm-gift_card_manager__button">Delete selected</button>',
									'<button class="btnPrev fg-gcm-button fg-gcm-gift_card_manager__button">Previous</button>',
									'<button class="btnNext fg-gcm-button fg-gcm-gift_card_manager__button">Next</button>',
								'</div>',
							'</td>',
						'</tr>',
					'</table>',
				'</div>',
			'</section>',

			'<script>',
				'jQuery( \'td.check\' ).each( ( idx, el ) => { el = jQuery( el ); if ( ! el.html() ) el.html( \'<input type="checkbox" class="fg-gcm-checkbox">\' ); } );', // because wp_kses_post() strips <input type="checkbox"> from output.
			'</script>',
		'</div>';
}

/**
 * Handler for Ajax action - Send report email
 */
function gift_card_manager_report_email_ajax() {
	check_ajax_referer( 'ajax' );
	gift_card_manager_validate_ajax();

	die( esc_html( gift_card_manager_report_email( true ) ) );
}

/**
 * Handler for Ajax action - Save settings
 */
function gift_card_manager_save_settings_ajax() {
	check_ajax_referer( 'ajax' );
	gift_card_manager_validate_ajax();

	$options_json = sanitize_text_field( wp_unslash( $_POST['options'] ?? '' ) ); // phpcs:ignore
	if ( ! json_decode( $options_json ) ) {
		http_response_code( 500 );
		die( 'Invalid options JSON' );
	}

	gift_card_manager_set_config( 'options', $options_json );
	gift_card_manager_cron_job_reschedule();
	die( 'Options saved' );
}

/**
 * Generate Gift Card unclaimed log rows
 *
 * @param string $search Search query string to filter results.
 * @param int    $rpp Rows per page.
 * @param int    $page Page number.
 */
function gift_card_manager_unclaimed_log( $search, $rpp, $page ) {
	$gift_cards = gift_card_manager_get_gift_cards_unclaimed();
	$search     = strtolower( trim( $search ) );
	$rpp        = max( 10, (int) $rpp );
	$page       = max( 1, (int) $page );
	$results    = array();
	foreach ( $gift_cards as $gift_card ) {
		// Put as much as possible non-HTML data into $data below, to maximize available data for the search function.
		$data = array(
			0 => $gift_card->code,
			1 => $gift_card->currency . ' ' . $gift_card->value,
			2 => $gift_card->group ?? 'N/A',
			3 => $gift_card->create_time ? wp_date( 'Y-m-d', $gift_card->create_time ) : '',
			4 => $gift_card->expire_time ? wp_date( 'Y-m-d H:i:s', $gift_card->expire_time ) : 'No expiry',
		);

		if ( $search && ! count( array_filter( $data, fn( $data ) => strpos( strtolower( $data ), $search ) !== false ) ) ) {
			continue; // Filter result out.
		}

		$results[] = gift_card_manager_concat(
			'<tr><td class="check"><input type="checkbox" class="fg-gcm-checkbox">',
			'</td><td>',
			$data[0],
			'</td><td>',
			$data[1],
			'</td><td>',
			$data[2],
			'</td><td>',
			$data[3],
			'</td><td>',
			$data[4],
			'</td><td><button class="btnDeleteUnclaimed fg-gcm-button fg-gcm-button--x" title="Delete Gift Card" data-id="',
			$gift_card->id,
			'">x</button>&nbsp;<span class="status"></span>',
			'</td></tr>'
		);
	}
	if ( count( $results ) ) {
		$offset = max( 0, min( floor( count( $results ) / $rpp ) * $rpp, ( $page - 1 ) * $rpp ) );
		return array(
			'data'        => implode( "\n", array_slice( $results, $offset, $rpp ) ),
			'page'        => (int) ( $offset / $rpp + 1 ),
			'total_pages' => (int) ( ( count( $results ) - 1 ) / $rpp + 1 ),
		);
	} else {
		return array(
			'data'        => '<tr><td colspan="99">No Unclaimed Gift Cards in the specified range.</td></tr>',
			'page'        => 1,
			'total_pages' => 1,
		);
	}
}

/**
 * Handler for Ajax action - Unclaimed Gift Cards log pagination
 */
function gift_card_manager_unclaimed_log_ajax() {
	check_ajax_referer( 'ajax' );
	gift_card_manager_validate_ajax();

	$request_id = sanitize_text_field( wp_unslash( $_POST['request_id'] ?? '' ) ); // phpcs:ignore
	$search     = sanitize_text_field( wp_unslash( $_POST['search'] ?? '' ) ); // phpcs:ignore
	$rpp        = sanitize_text_field( wp_unslash( $_POST['rpp'] ?? '' ) ); // phpcs:ignore
	$page       = sanitize_text_field( wp_unslash( $_POST['page'] ?? '' ) ); // phpcs:ignore

	header( 'Content-Type: application/json' );
	die( wp_json_encode( array_merge( array( 'request_id' => $request_id ), gift_card_manager_unclaimed_log( $search, $rpp, $page ) ) ) );
}

/**
 * Validate Ajax admin requests
 *
 * @param bool $require_admin True if admin login is required to process the Ajax request.
 */
function gift_card_manager_validate_ajax( $require_admin = true ) { //phpcs:ignore
	if ( wp_doing_ajax() ) {
		if ( $require_admin && ! gift_card_manager_is_admin( true ) ) {
			http_response_code( 500 );
			die( 'Invalid user' );
		}
		check_ajax_referer( 'ajax' );
	} else {
		http_response_code( 500 );
	}
}

require_once __DIR__ . '/lib.php';

add_action( 'admin_menu', gift_card_manager_concat( __NAMESPACE__, '\\gift_card_manager_add_settings_page' ) );
add_action( 'admin_enqueue_scripts', gift_card_manager_concat( __NAMESPACE__, '\\gift_card_manager_enqueue_scripts' ) );

add_action( 'wp_ajax_nopriv_gift_card_manager_claim_ajax', gift_card_manager_concat( __NAMESPACE__, '\\gift_card_manager_claim_ajax' ) );
add_action( 'wp_ajax_gift_card_manager_claim_ajax', gift_card_manager_concat( __NAMESPACE__, '\\gift_card_manager_claim_ajax' ) );

add_action( 'wp_ajax_nopriv_gift_card_manager_claimed_log_ajax', gift_card_manager_concat( __NAMESPACE__, '\\gift_card_manager_claimed_log_ajax' ) );
add_action( 'wp_ajax_gift_card_manager_claimed_log_ajax', gift_card_manager_concat( __NAMESPACE__, '\\gift_card_manager_claimed_log_ajax' ) );

add_action( 'wp_ajax_nopriv_gift_card_manager_delete_ajax', gift_card_manager_concat( __NAMESPACE__, '\\gift_card_manager_delete_ajax' ) );
add_action( 'wp_ajax_gift_card_manager_delete_ajax', gift_card_manager_concat( __NAMESPACE__, '\\gift_card_manager_delete_ajax' ) );

add_action( 'wp_ajax_nopriv_gift_card_manager_delete_all_ajax', gift_card_manager_concat( __NAMESPACE__, '\\gift_card_manager_delete_all_ajax' ) );
add_action( 'wp_ajax_gift_card_manager_delete_all_ajax', gift_card_manager_concat( __NAMESPACE__, '\\gift_card_manager_delete_all_ajax' ) );

add_action( 'wp_ajax_nopriv_gift_card_manager_expired_log_ajax', gift_card_manager_concat( __NAMESPACE__, '\\gift_card_manager_expired_log_ajax' ) );
add_action( 'wp_ajax_gift_card_manager_expired_log_ajax', gift_card_manager_concat( __NAMESPACE__, '\\gift_card_manager_expired_log_ajax' ) );

add_action( 'wp_ajax_nopriv_gift_card_manager_generate_ajax', gift_card_manager_concat( __NAMESPACE__, '\\gift_card_manager_generate_ajax' ) );
add_action( 'wp_ajax_gift_card_manager_generate_ajax', gift_card_manager_concat( __NAMESPACE__, '\\gift_card_manager_generate_ajax' ) );

add_action( 'wp_ajax_nopriv_gift_card_manager_group_name_exists_ajax', gift_card_manager_concat( __NAMESPACE__, '\\gift_card_manager_group_name_exists_ajax' ) );
add_action( 'wp_ajax_gift_card_manager_group_name_exists_ajax', gift_card_manager_concat( __NAMESPACE__, '\\gift_card_manager_group_name_exists_ajax' ) );

add_action( 'wp_ajax_nopriv_gift_card_manager_report_email_ajax', gift_card_manager_concat( __NAMESPACE__, '\\gift_card_manager_report_email_ajax' ) );
add_action( 'wp_ajax_gift_card_manager_report_email_ajax', gift_card_manager_concat( __NAMESPACE__, '\\gift_card_manager_report_email_ajax' ) );

add_action( 'wp_ajax_nopriv_gift_card_manager_save_settings_ajax', gift_card_manager_concat( __NAMESPACE__, '\\gift_card_manager_save_settings_ajax' ) );
add_action( 'wp_ajax_gift_card_manager_save_settings_ajax', gift_card_manager_concat( __NAMESPACE__, '\\gift_card_manager_save_settings_ajax' ) );

add_action( 'wp_ajax_nopriv_gift_card_manager_unclaimed_log_ajax', gift_card_manager_concat( __NAMESPACE__, '\\gift_card_manager_unclaimed_log_ajax' ) );
add_action( 'wp_ajax_gift_card_manager_unclaimed_log_ajax', gift_card_manager_concat( __NAMESPACE__, '\\gift_card_manager_unclaimed_log_ajax' ) );
