<?php
/**
 * Plugin function library
 *
 * @package Gift_Card_Manager
 */

namespace GIFT_CARD_MANAGER;

/**
 * Get site administrator email address
 */
function gift_card_manager_admin_email() {
	return get_bloginfo( 'admin_email' );
}

/**
 * Get first non-falsy argument, or null if none found
 */
function gift_card_manager_coalesce() {
	foreach ( func_get_args() as $arg ) {
		if ( $arg ) {
			return $arg;
		}
	}
	return null;
}

/**
 * Concatenate arguments into one string
 */
function gift_card_manager_concat() {
	$ret = '';
	foreach ( func_get_args() as $arg ) {
		$ret .= $arg;
	}
	return $ret;
}

/**
 * Do WP-Cron actions for this plugin - send daily report email
 */
function gift_card_manager_cron_job_report_email() {
	static $run_yet = false;
	if ( $run_yet ) {
		return;
	}
	$run_yet = true;

	$last_send_time         = gift_card_manager_get_config( 'last_report_send_time' );
	$last_send_attempt_time = gift_card_manager_get_config( 'last_report_send_attempt_time' );
	if ( gift_card_manager_option( 'reportEnabled', true ) && ( ( ! $last_send_time && ! $last_send_attempt_time ) || ( wp_date( 'd' ) !== wp_date( 'd', $last_send_time ) && gift_card_manager_time() >= $last_send_attempt_time + 300 ) ) ) {
		gift_card_manager_report_email( gift_card_manager_option( 'reportForce', true ) );
	}
	gift_card_manager_cron_job_reschedule();
}

/**
 * Re-schedule WP-Cron jobs
 */
function gift_card_manager_cron_job_reschedule() {
	gift_card_manager_cron_job_unschedule();
	gift_card_manager_cron_job_schedule();
}

/**
 * Schedule WP-Cron jobs
 */
function gift_card_manager_cron_job_schedule() {
	static $done = false;
	if ( $done ) {
		return;
	}

	$unmod        = fn( $dividend, $mod ) => $dividend - ( $dividend % $mod );
	$install_time = gift_card_manager_get_config( 'plugin_install_time' );

	// Schedule "daily report email" job.
	$interval          = 86400;
	$last_run_time     = gift_card_manager_get_config( 'last_report_send_time' );
	$last_attempt_time = gift_card_manager_get_config( 'last_report_send_attempt_time' );
	if ( gift_card_manager_time() >= $unmod( max( $install_time, $last_run_time, $last_attempt_time ), $interval ) + $interval + 300 ) {
		gift_card_manager_cron_job_report_email();
		$last_attempt_time = gift_card_manager_get_config( 'last_report_send_attempt_time' );
		$last_run_time     = gift_card_manager_get_config( 'last_report_send_time' );
	}
	do {
		$next = wp_next_scheduled( 'gift_card_manager_cron_job_report_email' );
		$bad  = $next && ( gift_card_manager_time() >= $unmod( min( $next, max( $install_time, $last_run_time, $last_attempt_time ) ), $interval ) + $interval + 300 || $next - gift_card_manager_time() >= $interval + 300 );
		if ( $bad ) {
			wp_unschedule_event( $next, 'gift_card_manager_cron_job_report_email' );
			gift_card_manager_cron_job_report_email();
			$last_attempt_time = gift_card_manager_get_config( 'last_report_send_attempt_time' );
			$last_run_time     = gift_card_manager_get_config( 'last_report_send_time' );
		}
	} while ( $bad );
	if ( ! $next ) {
		$daily_offset        = 60 * 60 * 23 + 55 * 60 - gift_card_manager_timezone_offset();
		$time_of_day         = gift_card_manager_time() % 86400;
		$report_send_time    = gift_card_manager_get_config( 'last_report_send_time' );
		$schedule_time_today = gift_card_manager_time() - $time_of_day + $daily_offset;
		wp_schedule_single_event( $schedule_time_today + ( $time_of_day >= $daily_offset && $report_send_time >= $schedule_time_today ? 86400 : 0 ), 'gift_card_manager_cron_job_report_email' );
	}

	$done = true;
}

/**
 * Unschedule/de-schedule any existing WP-Cron jobs
 */
function gift_card_manager_cron_job_unschedule() {
	while ( $next = wp_next_scheduled( 'gift_card_manager_cron_job_report_email' ) ) { // phpcs:ignore
		wp_unschedule_event( $next, 'gift_card_manager_cron_job_report_email' );
	}
}

/**
 * Deactivate plugin
 */
function gift_card_manager_deactivate() {
	gift_card_manager_cron_job_unschedule();
}

/**
 * Handler for "admin_enqueue_scripts" and "wp_enqueue_scripts" actions
 */
function gift_card_manager_enqueue_scripts() {
	// Admin scripts.
	if ( 'admin.php' === basename( gift_card_manager_server( 'SCRIPT_FILENAME' ) ) && 'gift-card-manager' === ( $_GET['page'] ?? '' ) ) { // phpcs:ignore
		wp_register_script( 'gift_card_manager_admin_settings_js', gift_card_manager_concat( plugin_dir_url( __FILE__ ), 'js/settings.js' ), array( 'jquery' ), gift_card_manager_time(), true );
		wp_enqueue_script( 'gift_card_manager_admin_settings_js' );
		wp_localize_script(
			'gift_card_manager_admin_settings_js',
			'gift_card_manager_php_admin_settings',
			array(
				'ajax_nonce'   => wp_create_nonce( 'ajax' ),
				'ajax_url'     => admin_url( 'admin-ajax.php' ),
				'options_json' => gift_card_manager_options_json(),
			)
		);
	}

	// Front-facing scripts.
	wp_register_script( 'gift_card_manager_claim_js', gift_card_manager_concat( plugin_dir_url( __FILE__ ), 'js/claim.js' ), array( 'jquery' ), gift_card_manager_time(), true );
	wp_enqueue_script( 'gift_card_manager_claim_js' );
	wp_localize_script(
		'gift_card_manager_claim_js',
		'gift_card_manager_php_claim_settings',
		array(
			'ajax_nonce' => wp_create_nonce( 'ajax' ),
			'ajax_url'   => admin_url( 'admin-ajax.php' ),
		)
	);

	// Styles.
	wp_enqueue_style( 'gift_card_manager_settings_css', gift_card_manager_concat( plugin_dir_url( __FILE__ ), 'css/settings.css' ), array(), gift_card_manager_time() );
}

/**
 * Generate a gift card
 *
 * @param string $format Gift Card Code Format.
 * @param string $currency Gift Card Currency.
 * @param string $value Gift Card Value.
 * @param string $group_id Group ID (optional).
 * @param string $expire Expire Date (optional).
 */
function gift_card_manager_generate( $format, $currency, $value, $group_id = null, $expire = null ) {
	global $wpdb;

	if ( ! $format ) {
		$format = 'aaaa-aaaa-aaaa';
	}
	if ( $value <= 0 ) {
		$value = 0;
	}
	$value = round( $value, 2 );

	$code = '';
	for ( $j = 0, $len = max( 8, min( 32, strlen( $format ) ) ); $j < $len; ++$j ) {
		switch ( $format[ $j ] ?? 'a' ) {
			case 'a':
				$code .= 'ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789'[ random_int( 0, 35 ) ];
				break;

			case 'l':
				$code .= 'ABCDEFGHIJKLMNOPQRSTUVWXYZ'[ random_int( 0, 25 ) ];
				break;

			case 'n':
				$code .= '0123456789'[ random_int( 0, 9 ) ];
				break;

			case '-':
				$code .= '-';
				break;

			default:
				$code .= $format[ $j ] ?? ( 'ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789'[ random_int( 0, 35 ) ] );
				break;
		}
	}
	if ( $group_id ) {
		gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( $wpdb->prepare( "INSERT INTO {$wpdb->prefix}fg_gift_card_manager_gift_cards (code, currency, value, group_id, create_time, expire_time) VALUES (%s, %s, %f, %d, %d, %d)", $code, $currency, $value, $group_id, gift_card_manager_time(), $expire ? ( gift_card_manager_strtotime( $expire ) + 60 * 60 * 24 - 1 ) : null ) ) ); // phpcs:ignore
	} else {
		gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( $wpdb->prepare( "INSERT INTO {$wpdb->prefix}fg_gift_card_manager_gift_cards (code, currency, value, create_time, expire_time) VALUES (%s, %s, %f, %d, %d)", $code, $currency, $value, gift_card_manager_time(), $expire ? ( gift_card_manager_strtotime( $expire ) + 60 * 60 * 24 - 1 ) : null ) ) ); // phpcs:ignore
	}

	return $code;
}

/**
 * Get a plugin configuration value
 *
 * @param string $name Name of configuration data.
 */
function gift_card_manager_get_config( $name ) {
	return gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( $wpdb->prepare( "SELECT value FROM {$wpdb->prefix}fg_gift_card_manager_config WHERE name = %s", $name ) ) )[0]->value ?? null; // phpcs:ignore
}

/**
 * Get claimed Gift Cards
 */
function gift_card_manager_get_gift_cards_claimed() {
	static $data = null;
	if ( null === $data ) {
		$data = gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( "SELECT {$wpdb->prefix}fg_gift_card_manager_gift_cards.*, {$wpdb->prefix}fg_gift_card_manager_gift_card_groups.name `group` FROM {$wpdb->prefix}fg_gift_card_manager_gift_cards LEFT JOIN {$wpdb->prefix}fg_gift_card_manager_gift_card_groups ON {$wpdb->prefix}fg_gift_card_manager_gift_cards.group_id = {$wpdb->prefix}fg_gift_card_manager_gift_card_groups.id WHERE claim_time IS NOT NULL AND delete_time IS NULL ORDER BY claim_time DESC, `group`, code" ) ); // phpcs:ignore
	}
	return $data;
}

/**
 * Get Gift Cards claimed yesterday by this plugin
 */
function gift_card_manager_get_gift_cards_claimed_yesterday() {
	static $gift_cards = null;
	if ( null === $gift_cards ) {
		$last_report_send_time = ( gift_card_manager_get_config( 'last_report_send_time' ) ?? 0 ) - 5;
		$now                   = gift_card_manager_time();
		$gift_cards = gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( $wpdb->prepare( "SELECT {$wpdb->prefix}fg_gift_card_manager_gift_cards.*, {$wpdb->prefix}fg_gift_card_manager_gift_card_groups.name `group` FROM {$wpdb->prefix}fg_gift_card_manager_gift_cards LEFT JOIN {$wpdb->prefix}fg_gift_card_manager_gift_card_groups ON {$wpdb->prefix}fg_gift_card_manager_gift_cards.group_id = {$wpdb->prefix}fg_gift_card_manager_gift_card_groups.id WHERE claim_time IS NOT NULL AND delete_time IS NULL AND claim_time BETWEEN %d AND %d ORDER BY claim_time", $last_report_send_time, $now ) ) ); // phpcs:ignore
	}
	return $gift_cards;
}

/**
 * Get expired Gift Cards
 */
function gift_card_manager_get_gift_cards_expired() {
	static $data = null;
	if ( null === $data ) {
		$time = gift_card_manager_time();
		$data = gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( $wpdb->prepare( "SELECT {$wpdb->prefix}fg_gift_card_manager_gift_cards.*, {$wpdb->prefix}fg_gift_card_manager_gift_card_groups.name `group` FROM {$wpdb->prefix}fg_gift_card_manager_gift_cards LEFT JOIN {$wpdb->prefix}fg_gift_card_manager_gift_card_groups ON {$wpdb->prefix}fg_gift_card_manager_gift_cards.group_id = {$wpdb->prefix}fg_gift_card_manager_gift_card_groups.id WHERE claim_time IS NULL AND delete_time IS NULL AND COALESCE(expire_time, 0) BETWEEN 1 AND %d ORDER BY {$wpdb->prefix}fg_gift_card_manager_gift_cards.create_time DESC, code", $time ) ) ); // phpcs:ignore
	}
	return $data;
}

/**
 * Get unclaiemd Gift Cards
 */
function gift_card_manager_get_gift_cards_unclaimed() {
	static $data = null;
	if ( null === $data ) {
		$time = gift_card_manager_time();
		$data = gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( $wpdb->prepare( "SELECT {$wpdb->prefix}fg_gift_card_manager_gift_cards.*, {$wpdb->prefix}fg_gift_card_manager_gift_card_groups.name `group` FROM {$wpdb->prefix}fg_gift_card_manager_gift_cards LEFT JOIN {$wpdb->prefix}fg_gift_card_manager_gift_card_groups ON {$wpdb->prefix}fg_gift_card_manager_gift_cards.group_id = {$wpdb->prefix}fg_gift_card_manager_gift_card_groups.id WHERE claim_time IS NULL AND delete_time IS NULL AND (COALESCE(expire_time, 0) = 0 OR expire_time > %d) ORDER BY {$wpdb->prefix}fg_gift_card_manager_gift_cards.create_time DESC, code", $time ) ) ); // phpcs:ignore
	}
	return $data;
}

/**
 * Load WordPress framework
 */
function gift_card_manager_init() {
	$original_dir = getcwd();
	$current_dir  = realpath( __DIR__ );
	chdir( $current_dir );
	do {
		if ( file_exists( './wp-load.php' ) ) {
			break;
		}

		$prev_current_dir = $current_dir;
		$current_dir      = realpath( '..' );
		chdir( '..' );
	} while ( realpath( $prev_current_dir ) !== realpath( $current_dir ) );
	if ( file_exists( './wp-load.php' ) ) {
		include_once './wp-load.php';
	} else {
		die( 'Please run this script file from a WordPress plugin directory on your file system.' );
	}

	chdir( $original_dir );
}

/**
 * Initialize plugin's database tables
 */
function gift_card_manager_init_db() {
	static $already_init = false;
	if ( $already_init ) {
		return;
	}
	$already_init  = true;
	$table_version = null;
	$create        = function () {
		gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( "CREATE TABLE IF NOT EXISTS {$wpdb->prefix}fg_gift_card_manager_config (name VARCHAR(512) PRIMARY KEY, value TEXT)" ) ); // phpcs:ignore
		gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( "CREATE TABLE IF NOT EXISTS {$wpdb->prefix}fg_gift_card_manager_gift_cards (id BIGINT PRIMARY KEY AUTO_INCREMENT, code VARCHAR(25) UNIQUE NOT NULL, value DECIMAL(20, 2) NOT NULL, group_id BIGINT, claim_user_id BIGINT, create_time BIGINT NOT NULL, claim_time BIGINT, expire_time BIGINT, delete_time BIGINT)" ) ); // phpcs:ignore
		gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( "CREATE TABLE IF NOT EXISTS {$wpdb->prefix}fg_gift_card_manager_gift_card_groups (id BIGINT PRIMARY KEY AUTO_INCREMENT, name VARCHAR(128) NOT NULL, account_limit INT NOT NULL, create_time BIGINT NOT NULL)" ) ); // phpcs:ignore
		gift_card_manager_set_config( 'table_version', GIFT_CARD_MANAGER_LATEST_TABLE_VERSION );
	};
	define( 'GIFT_CARD_MANAGER_LATEST_TABLE_VERSION', 4 );
	while ( ( $table_version = (int) gift_card_manager_get_config( 'table_version' ) ) !== GIFT_CARD_MANAGER_LATEST_TABLE_VERSION ) { // phpcs:ignore
		switch ( $table_version ) {
			case GIFT_CARD_MANAGER_LATEST_TABLE_VERSION:
				// Already latest version.
				break;

			case 0:
				// Initialize to latest version.
				$create();
				gift_card_manager_set_config( 'plugin_install_time', gift_card_manager_time() );
				break;

			case 1:
				// Initialize to version 2.
				gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( "CREATE TABLE IF NOT EXISTS {$wpdb->prefix}fg_gift_card_manager_gift_card_groups (id BIGINT PRIMARY KEY AUTO_INCREMENT, name VARCHAR(128) NOT NULL, account_limit INT NOT NULL, create_time BIGINT NOT NULL)" ) ); // phpcs:ignore
				gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( "ALTER TABLE {$wpdb->prefix}fg_gift_card_manager_gift_cards ADD COLUMN group_id BIGINT AFTER value" ) ); // phpcs:ignore
				gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( "ALTER TABLE {$wpdb->prefix}fg_gift_card_manager_gift_cards ADD CONSTRAINT fk_group FOREIGN KEY (group_id) REFERENCES {$wpdb->prefix}fg_gift_card_manager_gift_card_groups(id)" ) ); // phpcs:ignore
				gift_card_manager_set_config( 'table_version', 2 );
				break;

			case 2:
				// Initialize to version 3.
				gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( "ALTER TABLE {$wpdb->prefix}fg_gift_card_manager_gift_cards CHANGE COLUMN code code VARCHAR(32)" ) ); // phpcs:ignore
				gift_card_manager_set_config( 'table_version', 3 );
				break;

			case 3:
				// Initialize to version 4.
				gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( "SET @old_sql_mode = @@sql_mode; SET SESSION sql_mode = REPLACE(@@sql_mode, 'STRICT_TRANS_TABLES', '')" ) ); // phpcs:ignore
				gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( "ALTER TABLE {$wpdb->prefix}fg_gift_card_manager_gift_cards CONVERT TO CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci" ) ); // phpcs:ignore
				gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( "ALTER TABLE {$wpdb->prefix}fg_gift_card_manager_gift_cards ADD COLUMN currency TEXT AFTER code" ) ); // phpcs:ignore
				gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( "UPDATE {$wpdb->prefix}fg_gift_card_manager_gift_cards SET currency = %s", get_woocommerce_currency() ) ); // phpcs:ignore
				gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( "SET SESSION sql_mode = @old_sql_mode" ) ); // phpcs:ignore
				gift_card_manager_set_config( 'table_version', 4 );
				break;

			default:
				// Initialize unsupported version to latest version.
				gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( "DROP TABLE {$wpdb->prefix}fg_gift_card_manager_config" ) ); // phpcs:ignore
				gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( "DROP TABLE {$wpdb->prefix}fg_gift_card_manager_gift_cards" ) ); // phpcs:ignore
				gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( "DROP TABLE {$wpdb->prefix}fg_gift_card_manager_gift_card_groups" ) ); // phpcs:ignore
				$create();
				break;
		}
	}

	// Save plugin version to database.
	gift_card_manager_set_config( 'plugin_version', gift_card_manager_version() );
}

/**
 * Get an option value as configured by user on Settings Page
 *
 * @param string $name Setting name.
 * @param string $default_value Default value.
 */
function gift_card_manager_option( $name, $default_value = null ) {
	return gift_card_manager_options()[$name] ?? $default_value; // phpcs:ignore
}

/**
 * Get plugin's options/settings
 */
function gift_card_manager_options() {
	return @json_decode( gift_card_manager_options_json(), true ); // phpcs:ignore
}

/**
 * Get plugin's options/settings as a JSON string
 */
function gift_card_manager_options_json() {
	return gift_card_manager_coalesce( gift_card_manager_get_config( 'options' ), '{}' );
}

/**
 * Perform a SQL database query
 *
 * @param string $callback A function that returns the result of executing a prepared statement.
 */
function gift_card_manager_query( $callback ) {
	global $wpdb;
	gift_card_manager_init_db();
	$results = $callback( $wpdb );
	return $results && $wpdb->last_error ? array() : $results;
}

/**
 * Send daily report email
 *
 * @param bool $force Force sending email regardless of any settings or reason to not send the daily report email (e.g. for test purposes).
 */
function gift_card_manager_report_email( $force = false ) {
	gift_card_manager_set_config( 'last_report_send_attempt_time', gift_card_manager_time() );

	$default_email = gift_card_manager_admin_email();
	$gift_cards    = gift_card_manager_get_gift_cards_claimed_yesterday();
	$email         = gift_card_manager_coalesce( gift_card_manager_option( 'monitorEmail' ), $default_email );
	if ( $email && ( $force || count( $gift_cards ) ) ) {
		( function_exists( 'wp_mail' ) ? 'wp_mail' : 'mail' )(
			$email,
			'Gift Card Manager for WooCommerce - Claim Report on ' . wp_date( 'Y-m-d' ),
			gift_card_manager_concat(
				'<!DOCTYPE html><html><head><style>table,th,td{border:solid 1px #000}</style></head><body><img src="',
				plugin_dir_url( __FILE__ ),
				'img/logo.png"><p>Daily Claim Report for ',
				site_url(),
				' on ',
				wp_date( 'Y-m-d' ),
				'.</p><table cellpadding="10" cellspacing="0">',
				count( $gift_cards ) ? '<tr><th>Code</th><th>Value</th><th>Group</th><th>Create Time</th><th>Claim Time</th></tr>' : '',
				implode(
					"\n",
					! count( $gift_cards )
						? array( '<tr><td colspan="99">Nothing to report.</td></tr>' )
						: array_map(
							function ( $gift_card ) {
								return gift_card_manager_concat(
									'<tr><td>',
									$gift_card->code,
									'</td><td>',
									htmlentities( $gift_card->currency ),
									' ',
									number_format( $gift_card->value, 2 ),
									'</td><td>',
									$gift_card->group ?? 'N/A',
									'</td><td>',
									$gift_card->create_time ? wp_date( 'Y-m-d H:i', $gift_card->create_time ) : '',
									'</td><td>',
									$gift_card->claim_time ? wp_date( 'Y-m-d H:i', $gift_card->claim_time ) : '',
									'</td></tr>',
								);
							},
							$gift_cards
						)
				),
				'</table>',
				"\n\n",
				'<hr>To stop receiving this email, please disable claim reports on the <a href="',
				trim( site_url(), '/' ),
				'/wp-admin/admin.php?page=gift-card-manager">Gift Card Manager for WooCommerce settings page</a> on your site.',
				'</body></html>'
			),
			function_exists( 'wp_mail' ) ? array( 'Content-Type: text/html; charset=UTF-8' ) : array( 'Content-Type' => 'text/html; charset=UTF-8' )
		);

		gift_card_manager_set_config( 'last_report_send_time', gift_card_manager_time() );

		return "Email sent to $email on " . wp_date( 'Y-m-d H:i:s' ) . '.';
	} elseif ( ! $email ) {
		return 'Email not sent, no email address available.';
	}

	return 'Email not sent, nothing to report.';
}

/**
 * Get the first $_SERVER value for which its name matches one of the provided arguments to this function
 *
 * @return $_SERVER value
 */
function gift_card_manager_server() {
	$normalize = fn( $str ) => strtolower( preg_replace( '/(^-+)|(-+\$)/', '', preg_replace( '/[^A-Za-z0-9]+/', '-', $str ?? '' ) ) );
	foreach ( func_get_args() as $arg ) {
		$arg = $normalize( $arg );
		if ( $arg ) {
			foreach ( $_SERVER as $name => $value ) {
				if ( $normalize( $name ) === $arg ) {
					return wp_kses_post( $value );
				}
			}
		}
	}
}

/**
 * Set a plugin configuration value
 *
 * @param string $name Name of configuration data.
 * @param string $value Value of configuration data.
 */
function gift_card_manager_set_config( $name, $value ) {
	gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( $wpdb->prepare( "INSERT INTO {$wpdb->prefix}fg_gift_card_manager_config (name, value) VALUES (%s, %s) ON DUPLICATE KEY UPDATE value = %s", $name, $value, $value ) ) ); // phpcs:ignore
}

/**
 * Get current UTC time as per strtotime()
 *
 * @param string $format strtotime() date format string (e.g. "Y-m-d H:i:s").
 */
function gift_card_manager_strtotime( $format ) {
	return strtotime( $format ) - gift_card_manager_timezone_offset();
}

/**
 * Get current UTC time
 */
function gift_card_manager_time() {
	return time();
}

/**
 * Get site timezone offset off of UTC in seconds
 */
function gift_card_manager_timezone_offset() {
	return wp_timezone()->getOffset( new \DateTime( 'now', wp_timezone() ) );
}
/**
 * Get plugin version.
 */
function gift_card_manager_version() {
	$matches = array();
	preg_match( '/Stable tag: (.+)\n/', file_get_contents( rtrim( __DIR__, '/' ) . '/readme.txt' ), $matches ); // phpcs:ignore
	return $matches[1] ?? '0';
}
