/**
 * Settings page
 *
 * @package Gift_Card_Manager
 */

( ( $ ) => {
	let buttonTimeout = {};
	function giftCardManagerDebounceButton( jq ) {
		jq.attr( "disabled", "disabled" ).addClass( "fg-gcm-button--disabled" );
		const id = giftCardManagerDebounceButton.caller.name; // Name of function that called giftCardManagerDebounceButton().
		clearTimeout( buttonTimeout[id] );
		const enable      = () => giftCardManagerEnableButton( jq );
		buttonTimeout[id] = setTimeout( enable, 5000 );
		return enable;
	}
	function giftCardManagerEnableButton( jq ) {
		jq.removeAttr( "disabled" ).removeClass( "fg-gcm-button--disabled" );
	}

	let claimedCurrentPage       = 1;
	let claimedPaginateRequestId = null;
	function giftCardManagerClaimedLogPaginate( jq, pageIncDecAmount ) {
		const enableButton = giftCardManagerDebounceButton( jq );

		$( "#claimed .feedback" ).css( "display", "" );
		claimedPaginateRequestId = window.crypto.randomUUID();
		const search             = $( "#claimed .txtSearch" ).val().trim();
		const rpp                = $( "#claimed .ddRpp" ).val() == "all" ? Number.MAX_SAFE_INTEGER : $( "#claimed .ddRpp" ).val();
		const page               = Math.max( 1, claimedCurrentPage + pageIncDecAmount );
		$.post(
			gift_card_manager_php_admin_settings.ajax_url,
			{
				action: "gift_card_manager_claimed_log_ajax",
				_ajax_nonce: gift_card_manager_php_admin_settings.ajax_nonce,
				request_id: claimedPaginateRequestId,
				search,
				rpp,
				page
			}
		)
		.done(
			function ( data ) {
				if ( data.request_id === claimedPaginateRequestId ) {
					claimedCurrentPage = data.page;
					$( "#claimed .feedback" ).css( "display", "none" );
					$( "#claimed table.log tbody" ).html( data.data );
					$( "#claimed .page" ).html( `Page ${claimedCurrentPage} of ${data.total_pages}` );
				}
			}
		)
		.always( enableButton );
	}

	function giftCardManagerDeleteExpired( jq ) {
		if ( ! confirm( "Really delete Gift Card?" ) ) {
			return;
		}

		const enableButton = giftCardManagerDebounceButton( jq );

		const id = jq.attr( "data-id" );
		$.post(
			gift_card_manager_php_admin_settings.ajax_url,
			{
				action: "gift_card_manager_delete_ajax",
				_ajax_nonce: gift_card_manager_php_admin_settings.ajax_nonce,
				id
			}
		)
		.done( () => giftCardManagerExpiredLogPaginate( $(), 0 ) )
		.fail( () => jq.closest( "tr" ).find( ".status" ).html( "<span>HTTP error</span>" ) )
		.always( enableButton );
	}

	function giftCardManagerDeleteUnclaimed( jq ) {
		if ( ! confirm( "Really delete Gift Card?" ) ) {
			return;
		}

		const enableButton = giftCardManagerDebounceButton( jq );

		const id = jq.attr( "data-id" );
		$.post(
			gift_card_manager_php_admin_settings.ajax_url,
			{
				action: "gift_card_manager_delete_ajax",
				_ajax_nonce: gift_card_manager_php_admin_settings.ajax_nonce,
				id
			}
		)
		.done( () => giftCardManagerUnclaimedLogPaginate( $(), 0 ) )
		.fail( () => jq.closest( "tr" ).find( ".status" ).html( "<span>HTTP error</span>" ) )
		.always( enableButton );
	}

	function giftCardManagerDeleteSelectedExpired( jq ) {
		const enableButton = giftCardManagerDebounceButton( jq );

		let ids = [];
		$( "#expired input:checkbox:checked" ).each( ( idx, value ) => ids.push( $( value ).closest( "tr" ).find( "button" ).attr( "data-id" ) ) );
		ids = ids.join( "," );

		if ( ! ids ) {
			alert( "No Gift Cards selected!" );
			return;
		}
		if ( ! confirm( "Really delete selected Gift Cards?" ) ) {
			return;
		}

		$.post(
			gift_card_manager_php_admin_settings.ajax_url,
			{
				action: "gift_card_manager_delete_all_ajax",
				_ajax_nonce: gift_card_manager_php_admin_settings.ajax_nonce,
				ids
			}
		)
		.done( () => giftCardManagerExpiredLogPaginate( $(), 0 ) )
		.fail( () => giftCardManagerExpiredLogPaginate( $(), 0 ) )
		.always( enableButton );
	}

	function giftCardManagerDeleteSelectedUnclaimed( jq ) {
		const enableButton = giftCardManagerDebounceButton( jq );

		let ids = [];
		$( "#unclaimed input:checkbox:checked" ).each( ( idx, value ) => ids.push( $( value ).closest( "tr" ).find( "button" ).attr( "data-id" ) ) );
		ids = ids.join( "," );

		if ( ! ids ) {
			alert( "No Gift Cards selected!" );
			return;
		}
		if ( ! confirm( "Really delete selected Gift Cards?" ) ) {
			return;
		}

		$.post(
			gift_card_manager_php_admin_settings.ajax_url,
			{
				action: "gift_card_manager_delete_all_ajax",
				_ajax_nonce: gift_card_manager_php_admin_settings.ajax_nonce,
				ids
			}
		)
		.done( () => giftCardManagerUnclaimedLogPaginate( $(), 0 ) )
		.fail( () => giftCardManagerUnclaimedLogPaginate( $(), 0 ) )
		.always( enableButton );
	}

	let expiredCurrentPage       = 1;
	let expiredPaginateRequestId = null;
	function giftCardManagerExpiredLogPaginate( jq, pageIncDecAmount ) {
		const enableButton = giftCardManagerDebounceButton( jq );

		$( "#expired .feedback" ).css( "display", "" );
		claimedPaginateRequestId = window.crypto.randomUUID();
		const search             = $( "#expired .txtSearch" ).val().trim();
		const rpp                = $( "#expired .ddRpp" ).val() == "all" ? Number.MAX_SAFE_INTEGER : $( "#expired .ddRpp" ).val();
		const page               = Math.max( 1, claimedCurrentPage + pageIncDecAmount );
		$.post(
			gift_card_manager_php_admin_settings.ajax_url,
			{
				action: "gift_card_manager_expired_log_ajax",
				_ajax_nonce: gift_card_manager_php_admin_settings.ajax_nonce,
				request_id: claimedPaginateRequestId,
				search,
				rpp,
				page
			}
		)
		.done(
			function ( data ) {
				if ( data.request_id === claimedPaginateRequestId ) {
					claimedCurrentPage = data.page;
					$( "#expired .feedback" ).css( "display", "none" );
					$( "#expired table.log tbody" ).html( data.data );
					$( "#expired .page" ).html( `Page ${claimedCurrentPage} of ${data.total_pages}` );
				}
			}
		)
		.always( enableButton );
	}

	function giftCardManagerGenerate( jq ) {
		$( "#generateStatus" ).html( "Generating..." );

		const enableButton = giftCardManagerDebounceButton( jq );

		$.post(
			gift_card_manager_php_admin_settings.ajax_url,
			{
				action: "gift_card_manager_generate_ajax",
				_ajax_nonce: gift_card_manager_php_admin_settings.ajax_nonce,
				account_limit: $( "#account-limit" ).val(),
				amount: $( "#amount" ).val(),
				currency: $( "#currency" ).val(),
				expire: $( "#expire" ).val(),
				format: $( "#format" ).val(),
				group: $( "#group" ).val(),
				value: $( "#value" ).val(),
			}
		)
		.done(
			function ( data ) {
				$( "#generateStatus" ).html( data );
				giftCardManagerUnclaimedLogPaginate( $(), 0 );
			}
		)
		.fail( () => $( "#generateStatus" ).html( "Could not generate Gift Cards." ) )
		.always( enableButton );
	}

	function giftCardManagerPreview( jq ) {
		const enableButton = giftCardManagerDebounceButton( jq );

		$.post(
			gift_card_manager_php_admin_settings.ajax_url,
			{
				action: "gift_card_manager_shortcode_test_ajax",
				_ajax_nonce: gift_card_manager_php_admin_settings.ajax_nonce
			}
		)
		.done( data => $( "#previewArea" ).append( "<p>" + data + "</p>" ) )
		.fail( () => $( "#previewArea" ).html( "Could not render Gift Card Manager shortcode." ) )
		.always( enableButton );
	}

	function giftCardManagerGetOptions() {
		const options         = {};
		options.accountLimit  = $( "#account-limit" )[ 0 ].value;
		options.amount        = $( "#amount" )[ 0 ].value;
		options.currency      = $( "#currency" )[ 0 ].value;
		options.expire        = $( "#expire" )[ 0 ].value;
		options.format        = $( "#format" )[ 0 ].value.replace( /^-+|-+$/g, "" ).trim();
		options.group         = $( "#group" )[ 0 ].value.trim();
		options.monitorEmail  = $( ".monitor-email" )[ 0 ].value;
		options.reportEnabled = ! ! $( ".report-enabled" )[ 0 ].checked;
		options.reportForce   = ! ! $( ".report-force" )[ 0 ].checked;
		options.value         = $( "#value" )[ 0 ].value;
		return options;
	}

	function giftCardManagerInit() {
		const options = JSON.parse( gift_card_manager_php_admin_settings.options_json );
		if ( ! options ) {
			options = {};
		}

		$( "#claimed .ddRpp" ).change( () => giftCardManagerClaimedLogPaginate( $(), 0 ) );
		$( "#expired .ddRpp" ).change( () => giftCardManagerExpiredLogPaginate( $(), 0 ) );
		$( "#unclaimed .ddRpp" ).change( () => giftCardManagerUnclaimedLogPaginate( $(), 0 ) );
		let pageDebounceTimeout   = null;
		let searchDebounceTimeout = null;
		$( "#claimed .txtSearch" ).on(
			"input",
			function () {
				if ( searchDebounceTimeout ) {
					clearTimeout( searchDebounceTimeout );
				}
				searchDebounceTimeout = setTimeout( () => giftCardManagerClaimedLogPaginate( $(), 0 ), 500 );
			}
		);
		$( "#expired .txtSearch" ).on(
			"input",
			function () {
				if ( searchDebounceTimeout ) {
					clearTimeout( searchDebounceTimeout );
				}
				searchDebounceTimeout = setTimeout( () => giftCardManagerExpiredLogPaginate( $(), 0 ), 500 );
			}
		);
		$( "#unclaimed .txtSearch" ).on(
			"input",
			function () {
				if ( searchDebounceTimeout ) {
					clearTimeout( searchDebounceTimeout );
				}
				searchDebounceTimeout = setTimeout( () => giftCardManagerUnclaimedLogPaginate( $(), 0 ), 500 );
			}
		);
		$( "#claimed .txtPage" ).on(
			"input",
			function () {
				const input = $( "#claimed .txtPage" ).val();
				if ( input && isFinite( input ) && input.match( /^\d+$/ ) ) {
					claimedCurrentPage = parseInt( input );
				}
				if ( pageDebounceTimeout ) {
					clearTimeout( pageDebounceTimeout );
				}
				pageDebounceTimeout = setTimeout( () => giftCardManagerClaimedLogPaginate( $(), 0 ), 500 );
			}
		);
		$( "#expired .txtPage" ).on(
			"input",
			function () {
				const input = $( "#expired .txtPage" ).val();
				if ( input && isFinite( input ) && input.match( /^\d+$/ ) ) {
					expiredCurrentPage = parseInt( input );
				}
				if ( pageDebounceTimeout ) {
					clearTimeout( pageDebounceTimeout );
				}
				pageDebounceTimeout = setTimeout( () => giftCardManagerExpiredLogPaginate( $(), 0 ), 500 );
			}
		);
		$( "#unclaimed .txtPage" ).on(
			"input",
			function () {
				const input = $( "#unclaimed .txtPage" ).val();
				if ( input && isFinite( input ) && input.match( /^\d+$/ ) ) {
					unclaimedCurrentPage = parseInt( input );
				}
				if ( pageDebounceTimeout ) {
					clearTimeout( pageDebounceTimeout );
				}
				pageDebounceTimeout = setTimeout( () => giftCardManagerUnclaimedLogPaginate( $(), 0 ), 500 );
			}
		);
		$( "#claimed .btnNext" ).click( event => giftCardManagerClaimedLogPaginate( $( event.target ), 1 ) );
		$( "#claimed .btnPrev" ).click( event => giftCardManagerClaimedLogPaginate( $( event.target ), -1 ) );
		$( "#expired .btnDeleteSelected" ).click( event => giftCardManagerDeleteSelectedExpired( $( event.target ), -1 ) );
		$( "#expired .btnNext" ).click( event => giftCardManagerExpiredLogPaginate( $( event.target ), 1 ) );
		$( "#expired .btnPrev" ).click( event => giftCardManagerExpiredLogPaginate( $( event.target ), -1 ) );
		$( "#unclaimed .btnDeleteSelected" ).click( event => giftCardManagerDeleteSelectedUnclaimed( $( event.target ), -1 ) );
		$( "#unclaimed .btnNext" ).click( event => giftCardManagerUnclaimedLogPaginate( $( event.target ), 1 ) );
		$( "#unclaimed .btnPrev" ).click( event => giftCardManagerUnclaimedLogPaginate( $( event.target ), -1 ) );
		$( "#btnGenerate" ).click( () => giftCardManagerGenerate( $( "#btnGenerate" ) ) );
		$( "#btnPreview" ).click( () => giftCardManagerPreview( $( "#btnPreview" ) ) );
		$( "#btnReportEmail" ).click( () => giftCardManagerReportEmail( $( "#btnReportEmail" ) ) );
		$( "#btnSaveSettings" ).click( () => giftCardManagerSaveSettings( $( "#btnSaveSettings" ) ) );
		$( window.document ).click(
			function () {
				const classList = Array.from( event.target.classList );
				if ( classList.includes( "btnDeleteExpired" ) ) {
					giftCardManagerDeleteExpired( $( event.target ) );
				}
				if ( classList.includes( "btnDeleteUnclaimed" ) ) {
					giftCardManagerDeleteUnclaimed( $( event.target ) );
				}
			}
		);
	}

	function giftCardManagerSaveSettings( jq ) {
		const options     = giftCardManagerGetOptions();
		let validateError = ! ! giftCardManagerValidateAll();

		if ( validateError ) {
			$( ".saveStatus" ).html( "Validation error(s), see highlighted fields." );
			return;
		}
		$( ".saveStatus" ).html( "Saving..." );

		const enableButton = giftCardManagerDebounceButton( jq );

		$.post(
			gift_card_manager_php_admin_settings.ajax_url,
			{
				action: "gift_card_manager_save_settings_ajax",
				_ajax_nonce: gift_card_manager_php_admin_settings.ajax_nonce,
				options: JSON.stringify( options )
			}
		)
		.done( () => $( ".saveStatus" ).html( "Settings saved." ) )
		.fail( () => $( ".saveStatus" ).html( "Error while saving settings." ) )
		.always( enableButton );
	}

	function giftCardManagerReportEmail( jq ) {
		$( "#reportEmailStatus" ).html( "Sending report email..." );

		const enableButton = giftCardManagerDebounceButton( jq );

		$.post(
			gift_card_manager_php_admin_settings.ajax_url,
			{
				action: "gift_card_manager_report_email_ajax",
				_ajax_nonce: gift_card_manager_php_admin_settings.ajax_nonce,
				ip: jq.attr( "data-ip" )
			}
		)
		.done( data => $( "#reportEmailStatus" ).html( data ) )
		.fail( () => $( "#reportEmailStatus" ).html( "Could not send email." ) )
		.always( enableButton );
	}

	let unclaimedCurrentPage       = 1;
	let unclaimedPaginateRequestId = null;
	function giftCardManagerUnclaimedLogPaginate( jq, pageIncDecAmount ) {
		const enableButton = giftCardManagerDebounceButton( jq );

		$( "#unclaimed .feedback" ).css( "display", "" );
		unclaimedPaginateRequestId = window.crypto.randomUUID();
		const search               = $( "#unclaimed .txtSearch" ).val().trim();
		const rpp                  = $( "#unclaimed .ddRpp" ).val() == "all" ? Number.MAX_SAFE_INTEGER : $( "#unclaimed .ddRpp" ).val();
		const page                 = Math.max( 1, unclaimedCurrentPage + pageIncDecAmount );
		$.post(
			gift_card_manager_php_admin_settings.ajax_url,
			{
				action: "gift_card_manager_unclaimed_log_ajax",
				_ajax_nonce: gift_card_manager_php_admin_settings.ajax_nonce,
				request_id: unclaimedPaginateRequestId,
				search,
				rpp,
				page
			}
		)
		.done(
			function ( data ) {
				if ( data.request_id === unclaimedPaginateRequestId ) {
					unclaimedCurrentPage = data.page;
					$( "#unclaimed .feedback" ).css( "display", "none" );
					$( "#unclaimed table.log tbody" ).html( data.data );
					$( "#unclaimed .page" ).html( `Page ${unclaimedCurrentPage} of ${data.total_pages}` );
				}
			}
		)
		.always( enableButton );
	}

	let groupExistsDebounceTimeout = null

	window.giftCardManagerValidate = function ( option ) {
		let validateError = false;
		const error       = function ( selector ) {
			$( `.${selector.replace( /[A-Z]/g, match => "-" + match[0].toLowerCase() ).replace( /^-/, "" )}` ).css( "border", "solid red 1px" );
			validateError = true;
		};
		const unerror     = selector => $( `.${selector.replace( /[A-Z]/g, match => "-" + match[0].toLowerCase() ).replace( /^-/, "" )}` ).css( "border", "" );
		const options     = giftCardManagerGetOptions(); // all options.

		const validators = {
			"accountLimit": () => ! options.accountLimit || ( isFinite( options.accountLimit ) && ! options.accountLimit.toString().match( /\D/ ) && options.accountLimit > 0 ),
			"amount": () => options.amount && ( isFinite( options.amount ) && ! options.amount.toString().match( /\D/ ) && options.amount > 0 ),
			"currency": () => options.currency,
			"expire": () => ! options.expire || ( options.expire.match( /^\d{4}-\d{2}-\d{2}$/ ) ),
			"format": () => ! options.format || ( options.format.match( /^[alnA-Z0-9\-]{8,32}$/ ) && ! options.format.match( /--/ ) && options.format.replace( /[^\-]/g, "" ).length <= 3 ),
			"group": () => ! options.group || ( options.group.length <= 128 ),
			"monitorEmail": () => ! options.monitorEmail || ( options.monitorEmail.match( /^.+@.+\..+$/ ) ),
			"value": () => options.value && ( isFinite( options.value ) && options.value > 0 )
		};
		if ( validators[option] && ! validators[option]() ) {
			error( option );
		} else {
			unerror( option );
		}

		$( "#groupExists" ).css( "display", "none" );
		if ( ! validateError && option === "group" ) {
			clearTimeout( groupExistsDebounceTimeout );
			groupExistsDebounceTimeout = setTimeout(
				function () {
					if ( ! options.group ) {
						return;
					}
					$.post(
						gift_card_manager_php_admin_settings.ajax_url,
						{
							action: "gift_card_manager_group_name_exists_ajax",
							_ajax_nonce: gift_card_manager_php_admin_settings.ajax_nonce,
							group: options.group
						}
					)
					.done( data => data && $( "#groupExists" ).css( "display", "" ).text( `Note: Group "${data}" already exists` ) );
				},
				250
			);
		}

		return validateError;
	}

	function giftCardManagerValidateAll( ) {
		let validateError = false;

		validateError = giftCardManagerValidate( "monitorEmail" ) ? true : validateError;

		return validateError;
	}

	$( document ).ready( giftCardManagerInit );
} )( jQuery );
