<?php
/**
 * Gift Card Manager for WooCommerce
 *
 * @package   Gift_Card_Manager
 * @author    Fulfillment Global Corporation <wordpress@fulfillment.global>
 * @copyright 2025-2026 Fulfillment Global Corporation
 * @license   GPL-2.0-or-later
 *
 * @wordpress-plugin
 * Plugin Name:       Gift Card Manager for WooCommerce
 * Description:       Generate and manage Gift Cards for WooCommerce sites.
 * Version:           2.1
 * Requires at least: 5.2
 * Requires PHP:      7.2
 * Requires Plugins:  woocommerce, woo-wallet
 * Author:            Fulfillment Global Corporation
 * Author URI:        https://fulfillment.global/
 * Text Domain:       gift-card-manager
 * License:           GPL v2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 */

namespace GIFT_CARD_MANAGER;

/**
 * Handler for Ajax action - Claim Gift Card
 */
function gift_card_manager_claim_ajax() {
	check_ajax_referer( 'ajax' );
	gift_card_manager_validate_ajax( false );

	header( 'Content-Type: application/json' );

	global $wpdb;
	$rollback = function ( $error, $reason = null ) {
		gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( 'ROLLBACK' ) ); // phpcs:ignore
		die( wp_json_encode( array_merge( array( 'error' => $error ), $reason ? array( 'reason' => $reason ) : array() ) ) );
	};
	$user     = wp_get_current_user();
	$user_id  = $user->ID ?? null;
	if ( ! $user_id ) {
		die( wp_json_encode( array( 'error' => 'login' ) ) );
	}
	$code     = sanitize_text_field( wp_unslash( $_POST['code'] ?? '' ) ); // phpcs:ignore
	$test     = sanitize_text_field( wp_unslash( $_POST['test'] ?? '' ) ); // phpcs:ignore
	gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( 'START TRANSACTION' ) ); // phpcs:ignore
	gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( "LOCK TABLES {$wpdb->prefix}fg_gift_card_manager_gift_cards WRITE, {$wpdb->prefix}fg_gift_card_manager_gift_card_groups READ, {$wpdb->prefix}woo_wallet_transactions WRITE, {$wpdb->prefix}users READ" ) ); // phpcs:ignore
	if ( $wpdb->last_error ) {
		$rollback( 'server-error-1', 1 );
	}
	$rows = gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( $wpdb->prepare( "SELECT *, {$wpdb->prefix}fg_gift_card_manager_gift_card_groups.id `group_id`, {$wpdb->prefix}fg_gift_card_manager_gift_card_groups.name `group`, {$wpdb->prefix}fg_gift_card_manager_gift_card_groups.account_limit FROM {$wpdb->prefix}fg_gift_card_manager_gift_cards LEFT JOIN {$wpdb->prefix}fg_gift_card_manager_gift_card_groups ON {$wpdb->prefix}fg_gift_card_manager_gift_cards.group_id = {$wpdb->prefix}fg_gift_card_manager_gift_card_groups.id WHERE code = %s AND delete_time IS NULL", $code ) ) ); // phpcs:ignore
	if ( $wpdb->last_error ) {
		$rollback( 'server-error-2', 2 );
	}
	if ( ! count( $rows ) ) {
		$rollback( 'invalid' );
	}
	$gift_card = $rows[0];
	if ( $gift_card->claim_time ) {
		$rollback( 'claimed' );
	}
	if ( $gift_card->expire_time && gift_card_manager_time() >= $gift_card->expire_time ) {
		$rollback( 'expired' );
	}
	$time     = gift_card_manager_time();
	$currency = $gift_card->currency;
	$value    = round( $gift_card->value, 2 );
	if ( $value <= 0 ) {
		$rollback( 'value' );
	}
	$group         = $gift_card->group ?? 'N/A';
	$account_limit = $gift_card->account_limit;
	if ( $account_limit ) {
		if ( gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( $wpdb->prepare( "SELECT COUNT(*) `count` FROM {$wpdb->prefix}fg_gift_card_manager_gift_cards WHERE claim_user_id = %d AND group_id = %d AND claim_time IS NOT NULL AND delete_time IS NULL", $user_id, $gift_card->group_id ) ) )[0]->count >= $account_limit ) { // phpcs:ignore
			$rollback( 'account-limit' );
		}
	}
	$admin_user = gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( "SELECT * FROM {$wpdb->prefix}users ORDER BY ID LIMIT 1" ) )[0]; // phpcs:ignore
	if ( $wpdb->last_error ) {
		$rollback( 'server-error-3', 3 );
	}
	if ( ! $test ) {
		$current_balance = round( ( gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( $wpdb->prepare( "SELECT SUM(amount) total FROM {$wpdb->prefix}woo_wallet_transactions WHERE type = 'credit' AND user_id = %d AND currency = %s ORDER BY date DESC LIMIT 1", $user_id, $currency ) ) )[0]->total ?? 0 ) - ( gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( $wpdb->prepare( "SELECT SUM(amount) total FROM {$wpdb->prefix}woo_wallet_transactions WHERE type = 'debit' AND user_id = %d AND currency = %s ORDER BY date DESC LIMIT 1", $user_id, $currency ) ) )[0]->total ?? 0 ), 2 ); // phpcs:ignore
		if ( $wpdb->last_error ) {
			$rollback( 'server-error-4', 4 );
		}
		gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( $wpdb->prepare( "INSERT INTO {$wpdb->prefix}woo_wallet_transactions SET blog_id = 1, user_id = %d, type = 'credit', amount = %f, balance = %f, currency = %s, details = %s, created_by = %d, deleted = 0, date = %s", $user_id, $value, $current_balance + $value, $currency, "Gift Card Manager Claim ($code)", $admin_user->ID, wp_date( 'Y-m-d H:i:s', $time ) ) ) ); // phpcs:ignore
		if ( $wpdb->last_error ) {
			$rollback( 'server-error-5', 5 );
		}
		gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( $wpdb->prepare( "UPDATE {$wpdb->prefix}fg_gift_card_manager_gift_cards SET claim_user_id = %d, claim_time = %d WHERE code = %s", $user_id, $time, $code ) ) ); // phpcs:ignore
		if ( $wpdb->last_error ) {
			$rollback( 'server-error-6', 6 );
		}
	}
	gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( 'COMMIT' ) ); // phpcs:ignore
	gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( 'UNLOCK TABLES' ) ); // phpcs:ignore
	wp_mail( strtolower( $admin_user->user_email ), gift_card_manager_concat( 'Gift Card Manager for WooCommerce - Gift Card Claimed on ', wp_date( 'Y-m-d' ), $test ? ' (Test)' : '' ), gift_card_manager_concat( '<!DOCTYPE html><html><body><style>table.aaa,table.aaa td{border:solid 1px #000}</style><table cellpadding="10" cellspacing="0" class="aaa"><tr><td>Code:</td><td>', $code, '</td></tr><tr><td>Gift Card Value:</td><td>', htmlentities( $currency ), ' ', $value, '</td></tr><tr><td>Group:</td><td>', htmlentities( $group ), '</td></tr><tr><td>User Name:</td><td>', htmlentities( $user->display_name ), '</td></tr><tr><td>Email:</td><td>', htmlentities( strtolower( $user->user_email ) ), '</td></tr><tr><td>Claim Time:</td><td>', wp_date( 'Y-m-d H:i:s', $time ), '</td></tr></table></body></html>' ), array( 'Content-Type: text/html;charset=utf-8' ) );
	die(
		wp_json_encode(
			array(
				'ok'    => 1,
				'value' => gift_card_manager_concat( $currency, ' ', number_format( $value ) ),
			)
		)
	);
}

/**
 * Main shortcode handler function
 *
 * @param array $atts Shortcode attributes.
 */
function gift_card_manager_shortcode( $atts ) {
	$atts = shortcode_atts(
		array(
			'test' => false,
		),
		$atts,
		'gift_card_manager'
	);

	$ajax_div_id = gift_card_manager_concat( 'gift_card_manager_', bin2hex( random_bytes( 5 ) ) );
	return gift_card_manager_concat(
		'<div class="fg-gcm-woogiftcard-claim">',
		'<input placeholder="Enter Gift Card Code" size="30" class="fg-gcm-input">&nbsp;',
		'<button class="fg-gcm-button fg-gcm__button" onclick="if (window.giftCardManagerClaim) window.giftCardManagerClaim( this, ' .
			esc_attr( (int) (bool) $atts['test'] ),
		' ); else jQuery( this ).closest( `div` ).find( `.woogiftcard-site-error` ).css( `display`, ``)">Claim</button>&nbsp;',
		'<span class="woogiftcard-status woogiftcard-account-limit" style="display: none">Account limit reached for this Gift Card.</span>',
		'<span class="woogiftcard-status woogiftcard-busy" style="display: none">Claim in progress...</span>',
		'<span class="woogiftcard-status woogiftcard-claimed" style="display: none">Gift Card already claimed.</span>',
		'<span class="woogiftcard-status woogiftcard-expired" style="display: none">Gift Card expired.</span>',
		'<span class="woogiftcard-status woogiftcard-server-error-1" style="display: none">Server error 1, please try again.</span>',
		'<span class="woogiftcard-status woogiftcard-server-error-2" style="display: none">Server error 2, please try again.</span>',
		'<span class="woogiftcard-status woogiftcard-server-error-3" style="display: none">Server error 3, please try again.</span>',
		'<span class="woogiftcard-status woogiftcard-server-error-4" style="display: none">Server error 4, please try again.</span>',
		'<span class="woogiftcard-status woogiftcard-server-error-5" style="display: none">Server error 5, please try again.</span>',
		'<span class="woogiftcard-status woogiftcard-server-error-6" style="display: none">Server error 6, please try again.</span>',
		'<span class="woogiftcard-status woogiftcard-failed" style="display: none">Empty or invalid server response, please try again.</span>',
		'<span class="woogiftcard-status woogiftcard-invalid" style="display: none">Invalid Gift Card code.</span>',
		'<span class="woogiftcard-status woogiftcard-login" style="display: none">Please login to claim Gift Card.</span>',
		'<span class="woogiftcard-status woogiftcard-site-error" style="display: none">Claim cannot be processed right now. Please try again in a few moments, or consult site contact details.</span>',
		'<span class="woogiftcard-status woogiftcard-success" style="display: none">Claim successful, an amount of <span class="woogiftcard-amount">0.00</span> has been added to your wallet.</span>',
		'<span class="woogiftcard-status woogiftcard-value" style="display: none">Gift Card has zero value, cannot claim.</span>',
		'</div>'
	);
}

/**
 * Handler for Ajax action - Render [gcm-claim test=1] shortcode
 */
function gift_card_manager_shortcode_test_ajax() {
	check_ajax_referer( 'ajax' );
	gift_card_manager_validate_ajax( false );

	echo do_shortcode( '[gcm-claim test=1]' );
	wp_die();
}

/**
/**
 * Handler for plugin uninstall hook
 */
function gift_card_manager_uninstall() {
	gift_card_manager_cron_job_unschedule();
	/* // phpcs:ignore
	gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( "DROP TABLE IF EXISTS {$wpdb->prefix}fg_gift_card_manager_config" ) ); // phpcs:ignore
	gift_card_manager_query( fn( &$wpdb ) => $wpdb->get_results( "DROP TABLE IF EXISTS {$wpdb->prefix}fg_gift_card_manager_gift_cards" ) ); // phpcs:ignore
	*/
}

require_once __DIR__ . '/lib.php';
require_once __DIR__ . '/settings.php';

// Initialize.
gift_card_manager_init();
gift_card_manager_cron_job_schedule();

// Actions.
add_action( 'gift_card_manager_cron_job_report_email', gift_card_manager_concat( __NAMESPACE__, '\\gift_card_manager_cron_job_report_email' ), 10, 2 );
add_action( 'wp_ajax_nopriv_gift_card_manager_claim_ajax', gift_card_manager_concat( __NAMESPACE__, '\\gift_card_manager_claim_ajax' ) );
add_action( 'wp_ajax_gift_card_manager_claim_ajax', gift_card_manager_concat( __NAMESPACE__, '\\gift_card_manager_claim_ajax' ) );
add_action( 'wp_ajax_nopriv_gift_card_manager_shortcode_test_ajax', gift_card_manager_concat( __NAMESPACE__, '\\gift_card_manager_shortcode_test_ajax' ) );
add_action( 'wp_ajax_gift_card_manager_shortcode_test_ajax', gift_card_manager_concat( __NAMESPACE__, '\\gift_card_manager_shortcode_test_ajax' ) );
add_action( 'wp_enqueue_scripts', gift_card_manager_concat( __NAMESPACE__, '\\gift_card_manager_enqueue_scripts' ) );

// Shortcodes.
add_shortcode( 'gcm-claim', gift_card_manager_concat( __NAMESPACE__, '\\gift_card_manager_shortcode' ) );

// Activation/deactivation/uninstall.
register_deactivation_hook( __FILE__, gift_card_manager_concat( __NAMESPACE__, '\\gift_card_manager_deactivate' ) );
register_uninstall_hook( __FILE__, gift_card_manager_concat( __NAMESPACE__, '\\gift_card_manager_uninstall' ) );
