<?php
/**
 * Uninstall routine for the GhostGate plugin.
 *
 * Executed when the plugin is deleted from WordPress admin. Removes
 * plugin options, transients, user meta and cleans .htaccess entries.
 *
 * @package   GhostGate
 * @subpackage Uninstall
 * @since     1.0.0
 * @license   GPL-2.0-or-later
 */



// REVIEW: セキュリティ — 直アクセス防止（WPCS準拠のスペースとブロック構文）.
if ( ! defined( 'WP_UNINSTALL_PLUGIN' ) ) {
	exit;
}

global $wpdb;

// REVIEW: Doc/Prefix — オプション削除リストをファイルスコープ変数として定義（prefix必須）.
// REVIEW: Style  — WPCSの推奨に合わせて長配列記法 array() を使用（短配列 [] を避ける）.
$ghostgate_option_keys = array(
	'ghostgate_login_slug',
	'ghostgate_code_length',
	'ghostgate_enable_code_length',
	'ghostgate_code_mode',
	'ghostgate_enable_code_mode',
	'ghostgate_custom_code',
	'ghostgate_enable_2fa',
	'ghostgate_ajax_limit',
	'ghostgate_enable_ajax_limit',
	'ghostgate_ajax_cooldown',
	'ghostgate_ajax_notify',
	'ghostgate_log_file_path',
	'ghostgate_enable_log_file',
	'ghostgate_log_retention_days',
	'ghostgate_notify_on_block',
	'ghostgate_block_author_enum',
	'ghostgate_disable_xmlrpc',
	'ghostgate_enable_login_limit',
	'ghostgate_login_attempt_limit',
	'ghostgate_enable_login_slug',
	'ghostgate_block_unused_rest',
	'ghostgate_enable_rest_limit',
	'ghostgate_rest_limit_count',
	'ghostgate_rest_limit_cooldown',
	'ghostgate_enable_session_control',
	'ghostgate_session_duration',
	'ghostgate_disable_remember_me',
	'ghostgate_session_timeout_redirect',
	'ghostgate_idle_mode',
	'ghostgate_remember_cookie_lifetime',
	'ghostgate_hide_json_endpoints',
	'ghostgate_json_allowed_routes',
	'ghostgate_json_allowed_prefixes',
	'ghostgate_session_timeout',
	'ghostgate_enable_session_control',
	'ghostgate_session_duration',
	'ghostgate_idle_mode',
	'ghostgate_session_timeout_redirect',
	'ghostgate_disable_remember_me',
	'ghostgate_remember_cookie_lifetime',
	'ghostgate_enable_session_control',
	'ghostgate_session_timeout',
	'ghostgate_login_block_cooldown',
	'ghostgate_block_index',
	'ghostgate_block_preview'
);

foreach ( $ghostgate_option_keys as $ghostgate_key ) {
	delete_option( $ghostgate_key );
	delete_site_option( $ghostgate_key );
}

// REVIEW: Uninstall routine runs once; direct query is acceptable here.
/* phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching */
$ghostgate_like       = $wpdb->esc_like( '_transient_ghostgate_' ) . '%';
$ghostgate_transients = $wpdb->get_col(
	$wpdb->prepare(
		"SELECT option_name FROM {$wpdb->options} WHERE option_name LIKE %s",
		$ghostgate_like
	)
);
/* phpcs:enable */

foreach ( $ghostgate_transients as $ghostgate_t ) {
	$ghostgate_t_key = str_replace( '_transient_', '', $ghostgate_t );
	delete_transient( $ghostgate_t_key );
}

// REVIEW: Uninstall routine runs once; direct query is acceptable here.
/* phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching */
$ghostgate_like            = $wpdb->esc_like( '_transient_ghostgate_rest_' ) . '%';
$ghostgate_rest_transients = $wpdb->get_col(
	$wpdb->prepare(
		"SELECT option_name FROM {$wpdb->options} WHERE option_name LIKE %s",
		$ghostgate_like
	)
);
/* phpcs:enable */

foreach ( $ghostgate_rest_transients as $ghostgate_t ) {
	$ghostgate_t_key = str_replace( '_transient_', '', $ghostgate_t );
	delete_transient( $ghostgate_t_key );
}


// REVIEW: Uninstall routine runs once; direct query is acceptable here.
/* phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching */
$ghostgate_like            = $wpdb->esc_like( '_transient_ghostgate_ajax_' ) . '%';
$ghostgate_ajax_transients = $wpdb->get_col(
	$wpdb->prepare(
		"SELECT option_name FROM {$wpdb->options} WHERE option_name LIKE %s",
		$ghostgate_like
	)
);
/* phpcs:enable */

foreach ( $ghostgate_ajax_transients as $ghostgate_t ) {
	$ghostgate_t_key = str_replace( '_transient_', '', $ghostgate_t );
	delete_transient( $ghostgate_t_key );
}


// REVIEW: Uninstall routine runs once; direct query is acceptable here.
/* phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching */
$ghostgate_like               = $wpdb->esc_like( '_transient_ghostgate_attempts_' ) . '%';
$ghostgate_attempt_transients = $wpdb->get_col(
	$wpdb->prepare(
		"SELECT option_name FROM {$wpdb->options} WHERE option_name LIKE %s",
		$ghostgate_like
	)
);
/* phpcs:enable */

foreach ( $ghostgate_attempt_transients as $ghostgate_t ) {
	$ghostgate_t_key = str_replace( '_transient_', '', $ghostgate_t );
	delete_transient( $ghostgate_t_key );
}


// REVIEW: Uninstall routine runs once; direct query is acceptable here.
/* phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching */
$ghostgate_like             = $wpdb->esc_like( '_transient_ghostgate_2fa_' ) . '%';
$ghostgate_twofa_transients = $wpdb->get_col(
	$wpdb->prepare(
		"SELECT option_name FROM {$wpdb->options} WHERE option_name LIKE %s",
		$ghostgate_like
	)
);
/* phpcs:enable */

foreach ( $ghostgate_twofa_transients as $ghostgate_t ) {
	$ghostgate_t_key = str_replace( '_transient_', '', $ghostgate_t );
	delete_transient( $ghostgate_t_key );
}


// REVIEW: Prefix — ユーザーメタ削除のループ変数も prefix 付与.
$ghostgate_users = get_users();

foreach ( $ghostgate_users as $ghostgate_user ) {
	delete_user_meta( $ghostgate_user->ID, 'ghostgate_enable_2fa' );
	delete_user_meta( $ghostgate_user->ID, 'ghostgate_2fa_verified' );
	delete_user_meta( $ghostgate_user->ID, 'ghostgate_2fa_enabled' );
}

// REVIEW: Prefix — ログファイル削除関連の変数に prefix 付与.
$ghostgate_log_path = get_option( 'ghostgate_log_file_path' );

if ( ! $ghostgate_log_path || ! file_exists( $ghostgate_log_path ) ) {
	$ghostgate_log_path = WP_CONTENT_DIR . '/ghostgate_logs.log';
}

if ( file_exists( $ghostgate_log_path ) ) {
	wp_delete_file( $ghostgate_log_path );
}

// REVIEW: Prefix — ログディレクトリ/ファイル走査の変数にも prefix 付与.
$ghostgate_log_dir   = WP_CONTENT_DIR . '/logs';
$ghostgate_log_files = glob( trailingslashit( $ghostgate_log_dir ) . 'ghostgate_logs_*.log' );

if ( $ghostgate_log_files ) {
	foreach ( $ghostgate_log_files as $ghostgate_file ) {
		wp_delete_file( $ghostgate_file );
	}
}

// REVIEW: Prefix — レガシーログのパス変数にも prefix 付与.
$ghostgate_legacy_log = WP_CONTENT_DIR . '/sbp2-login.log';

if ( file_exists( $ghostgate_legacy_log ) ) {
	wp_delete_file( $ghostgate_legacy_log );
}

// REVIEW: Prefix — .htaccess 関連のファイル・内容・正規表現変数に prefix 付与.
$ghostgate_htaccess_path = ABSPATH . '.htaccess';

global $wp_filesystem;

if ( empty( $wp_filesystem ) ) {
	require_once ABSPATH . '/wp-admin/includes/file.php';
	WP_Filesystem();
}

if ( $wp_filesystem->exists( $ghostgate_htaccess_path ) && $wp_filesystem->is_writable( $ghostgate_htaccess_path ) ) {
	$ghostgate_content = $wp_filesystem->get_contents( $ghostgate_htaccess_path );

	if ( false !== $ghostgate_content ) {
		$ghostgate_start_tag = '# BEGIN GhostGate XML-RPC Block';
		$ghostgate_end_tag   = '# END GhostGate XML-RPC Block';

		$ghostgate_pattern = '/' . preg_quote( $ghostgate_start_tag, '/' ) . '.*?' . preg_quote( $ghostgate_end_tag, '/' ) . "(\r?\n)?/s";

		$ghostgate_cleaned = preg_replace( $ghostgate_pattern, '', $ghostgate_content );

		if ( $ghostgate_cleaned !== $ghostgate_content ) {
			$wp_filesystem->put_contents( $ghostgate_htaccess_path, $ghostgate_cleaned, FS_CHMOD_FILE );
		}
	}
}
