<?php
// inc/two-factor-auth.php

if (!defined('ABSPATH')) exit;


// =======================
// 2FA：ユーザープロファイルUI
// =======================
function ghostgate_user_2fa_field($user) {
    if (!current_user_can('edit_users')) return;

    // 🔒 プラグイン設定で2FAが有効でなければ非表示
    if (!get_option('ghostgate_enable_2fa')) {
        return;
    }

    $enabled = get_user_meta($user->ID, 'ghostgate_2fa_enabled', true);
    ?>
    <h2><?php esc_html_e('GhostGate 2FA', 'ghostgate'); ?></h2>
    <table class="form-table">
        <tr>
            <th><label for="ghostgate_2fa_enabled"><?php esc_html_e('2FAを有効にする', 'ghostgate'); ?></label></th>
            <td>
                <input type="checkbox" name="ghostgate_2fa_enabled" id="ghostgate_2fa_enabled" value="1" <?php checked($enabled, '1'); ?>>
                <span class="description"><?php esc_html_e('このユーザーに対してログイン後にメール認証を要求します。', 'ghostgate'); ?></span>
            </td>
        </tr>
    </table>
    <?php wp_nonce_field('ghostgate_2fa_save', 'ghostgate_2fa_nonce'); ?>
    <?php
}


add_action('show_user_profile', 'ghostgate_user_2fa_field');
add_action('edit_user_profile', 'ghostgate_user_2fa_field');

// =======================
// 2FA：保存処理
// =======================
function ghostgate_save_user_2fa_field( $user_id ) {
	if ( ! current_user_can( 'edit_user', $user_id ) ) {
		return;
	}

	// 先にノンスを安全に取得→検証（$_POST 直参照しない）
	$nonce = sanitize_text_field( (string) ( filter_input( INPUT_POST, 'ghostgate_2fa_nonce', FILTER_UNSAFE_RAW ) ?? '' ) );
	if ( ! wp_verify_nonce( $nonce, 'ghostgate_2fa_save' ) ) {
		return;
	}

	// ノンスOK後にチェックボックスを読む（存在＝ON）
	$enabled_raw = filter_input( INPUT_POST, 'ghostgate_2fa_enabled', FILTER_UNSAFE_RAW );
	if ( null !== $enabled_raw ) {
		update_user_meta( $user_id, 'ghostgate_2fa_enabled', '1' );
	} else {
		delete_user_meta( $user_id, 'ghostgate_2fa_enabled' );
	}
}





add_action('personal_options_update', 'ghostgate_save_user_2fa_field');
add_action('edit_user_profile_update', 'ghostgate_save_user_2fa_field');

// 🔐 ログイン時：2FAを割り込み
add_filter('wp_authenticate_user', 'ghostgate_intercept_2fa', 20, 2);
function ghostgate_intercept_2fa( $user, $password ) {
    if ( is_wp_error( $user ) || ! ( $user instanceof WP_User ) ) {
        return $user;
    }
    if ( ! get_option( 'ghostgate_enable_2fa' ) ) return $user;
    if ( ! get_user_meta( $user->ID, 'ghostgate_2fa_enabled', true ) ) return $user;

    // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Core login form has no nonce; read-only check of our flag
    $verified = (string) ( filter_input( INPUT_POST, 'ghostgate_2fa_verified', FILTER_UNSAFE_RAW ) ?? '' );
    if ( $verified === '1' ) { return $user; }

    // 6桁コードを発行（5分）
    $code = str_pad( strval( wp_rand( 0, 999999 ) ), 6, '0', STR_PAD_LEFT );
    set_transient( "ghostgate_2fa_code_{$user->ID}", $code, 300 );

    // ✅ パスワードは保存しない。ペンディング情報のみ保持
    // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Core login form has no nonce; reading 'rememberme' only to persist intended session length
    $remember_raw = filter_input( INPUT_POST, 'rememberme', FILTER_UNSAFE_RAW );
    $remember     = is_string( $remember_raw ) && in_array( $remember_raw, array( '1', 'on', 'true', 'forever' ), true );

    set_transient( "ghostgate_2fa_pending_{$user->ID}", array(
        'user_id'  => $user->ID,
        'remember' => (bool) $remember,
    ), 300 );


    ghostgate_log( sprintf(
        // translators: %1$s: user login, %2$s: user email
        __( '2FA code generated → User: %1$s, Email: %2$s', 'ghostgate' ),
        $user->user_login,
        $user->user_email
    ) );

    wp_mail(
        $user->user_email,
        __( '【GhostGate】ログイン認証コード自動送信', 'ghostgate' ),
        // translators: %s is the 2FA code string.
        sprintf( __( 'ユーザーの認証コードは次の通りです : %s', 'ghostgate' ), $code ) . "\n" . __( '(有効期限5分)', 'ghostgate' ),
        [ 'Content-Type: text/plain; charset=UTF-8' ]
    );

ghostgate_log(
    sprintf(
        /* translators: %s is the WordPress username of the user who received the 2FA email. */
        __( '2FA email sent → User: %s', 'ghostgate' ),
        $user->user_login
    )
);


    // /ghost-2fa?user=... に移動（add_query_arg には未エンコードで渡す）
    $url = add_query_arg( 'user', $user->user_login, home_url( '/ghost-2fa' ) );
    wp_safe_redirect( esc_url_raw( $url ) );
    exit;
}


// ✅ 2FA画面処理
// ✅ フロントエンド 2FA ページ用スタイルを安全に enqueue
function ghostgate_enqueue_2fa_assets() {
    // REQUEST_URI → 安全にパスへ正規化
    $path = '';
    $path = ghostgate_get_request_path();

    // GET 'user' は存在チェックでもサニタイズして読む（読み取りのみ）
// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- read-only check for enqueue
$user_raw = filter_input( INPUT_GET, 'user', FILTER_UNSAFE_RAW );
$user     = $user_raw !== null ? sanitize_user( (string) $user_raw, true ) : '';


    if ( $user !== '' && strpos( $path, 'ghost-2fa' ) !== false ) {
        wp_enqueue_style(
            'ghostgate-2fa',
            plugins_url( '../assets/css/ghost-style-2fa.css', __FILE__ ),
            array(),
            defined( 'GHOSTGATE_VERSION' ) ? GHOSTGATE_VERSION : '1.3.2'
        );
    }
}


add_action('wp_enqueue_scripts', 'ghostgate_enqueue_2fa_assets');


// ✅ メイン処理
function ghostgate_handle_2fa_page() {
    // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- read-only GET param
    $user_login = sanitize_user( (string) ( filter_input( INPUT_GET, 'user', FILTER_UNSAFE_RAW ) ?? '' ), true );
    if ( $user_login === '' ) {
        return;
    }

    $user_id  = username_exists( $user_login );
    if ( ! $user_id ) {
        return;
    }

    // 👇 旧: ghostgate_2fa_user_* は使わない。保留情報の有無でゲートする
    $pending = get_transient( "ghostgate_2fa_pending_{$user_id}" );
    if ( ! $pending ) {
        return;
    }

    // $_SERVER を直参照せず取得
    $method_raw = (string) ( filter_input( INPUT_SERVER, 'REQUEST_METHOD', FILTER_UNSAFE_RAW ) ?? '' );
    $method     = strtoupper( sanitize_text_field( $method_raw ) );

    if ( 'GET' === $method ) {
        // translators: %s is the username of the person viewing the 2FA screen.
        ghostgate_log( sprintf( __( '2FA screen displayed → User: %s', 'ghostgate' ), $user_login ) );
        ?>
        <!DOCTYPE html>
        <html <?php language_attributes(); ?>>
        <head>
            <meta charset="<?php bloginfo( 'charset' ); ?>">
            <meta name="viewport" content="width=device-width, initial-scale=1">
            <title><?php esc_html_e( '2FA 認証', 'ghostgate' ); ?></title>
            <?php wp_head(); ?>
        </head>
        <body <?php body_class( 'ghostgate-2fa-page' ); ?>>
        <?php wp_body_open(); ?>
        <div class="ghostgate-2fa-wrapper">
            <form method="post" action="<?php echo esc_url( add_query_arg( 'user', $user_login, home_url( '/ghost-2fa' ) ) ); ?>">
                <h2><?php esc_html_e( '認証コードを入力してお入りください。', 'ghostgate' ); ?></h2>
                <?php wp_nonce_field( 'ghostgate_2fa_verify', 'ghostgate_2fa_nonce' ); ?>
                <input type="text" name="ghostgate_2fa_code" maxlength="6" inputmode="numeric" pattern="[0-9]{6}" required>
                <input type="submit" value="<?php esc_attr_e( 'Login', 'ghostgate' ); ?>">
            </form>
        </div>
        <?php wp_footer(); ?>
        </body>
        </html>
        <?php
        exit;
    }

    // POST: ノンス検証と入力サニタイズ
    $nonce = sanitize_text_field( (string) ( filter_input( INPUT_POST, 'ghostgate_2fa_nonce', FILTER_UNSAFE_RAW ) ?? '' ) );
    if ( ! wp_verify_nonce( $nonce, 'ghostgate_2fa_verify' ) ) {
        wp_die(
            esc_html__( 'Security check failed.', 'ghostgate' ),
            esc_html__( '2FA Verification Error', 'ghostgate' )
        );
    }

    // 入力コード
    $raw_code = (string) ( filter_input( INPUT_POST, 'ghostgate_2fa_code', FILTER_UNSAFE_RAW ) ?? '' );
    $provided = sanitize_text_field( $raw_code );

    // フォーマット軽検証（任意だが堅牢化）
    if ( strlen( $provided ) !== 6 || ! ctype_digit( $provided ) ) {
        wp_die(
            esc_html__( 'The verification code is invalid.', 'ghostgate' ),
            esc_html__( '2FA Failed', 'ghostgate' )
        );
    }

    // 期待コード
    $expected = get_transient( "ghostgate_2fa_code_{$user_id}" );
    if ( ! $expected || $provided !== $expected ) {
        // コード本体はログに残さない
        ghostgate_log(
            sprintf(
                /* translators: %1$s: Username */
                __( '❌ 2FA code mismatch → User: %1$s', 'ghostgate' ),
                $user_login
            ),
            'warning'
        );
        wp_die(
            esc_html__( 'The verification code is incorrect or expired.', 'ghostgate' ),
            esc_html__( '2FA Failed', 'ghostgate' )
        );
    }

    // remember フラグ
    $remember = is_array( $pending ) && ! empty( $pending['remember'] );

    // 片付け
    delete_transient( "ghostgate_2fa_code_{$user_id}" );
    delete_transient( "ghostgate_2fa_pending_{$user_id}" );

    // 認証確定（パスワード不要）
    wp_set_auth_cookie( $user_id, $remember, is_ssl() );
    wp_set_current_user( $user_id ); // 同一リクエスト内で current_user を即反映
    $user_obj = get_user_by( 'id', $user_id );
    do_action( 'wp_login', $user_login, $user_obj );

    // リダイレクト
    $redirect = get_transient( "ghostgate_2fa_redirect_{$user_login}" );
    if ( $redirect ) {
        delete_transient( "ghostgate_2fa_redirect_{$user_login}" );
    } else {
        $redirect = admin_url();
    }

    // translators: %s is the WordPress username of the user who passed 2FA.
    ghostgate_log( sprintf( __( '✅ 2FA success → User: %s', 'ghostgate' ), $user_login ) );


    $target = wp_validate_redirect( esc_url_raw( $redirect ), admin_url() );
    wp_safe_redirect( $target );
    exit;
}


add_action('init', 'ghostgate_handle_2fa_page');
