<?php
/**
 * Session manager for the GhostGate plugin.
 *
 * Provides front-end session tracking and heartbeat utilities.
 *
 * @package   GhostGate
 * @since     1.0.0
 * @license   GPL-2.0-or-later
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// ============================
// セッション開始（最小限）
// ============================
// 機能ON && ログイン中のみ、cron以外で早い段階で開始（フロント/管理共通）
add_action( 'init', function () {
	if ( get_option( 'ghostgate_enable_session_control' )
	  && is_user_logged_in()
	  && session_status() === PHP_SESSION_NONE
	  && ! wp_doing_cron()
	) {
		@session_start();
	}
}, 5 );

// ============================
// ログイン直後の初期化（再ログイン即落ち防止）
// ============================
add_action( 'wp_login', function( $user_login, $user ) {
	if ( ! get_option( 'ghostgate_enable_session_control' ) ) {
		return;
	}
	if ( session_status() === PHP_SESSION_NONE ) {
		@session_start();
	}
	$now = function_exists( 'ghostgate_now' ) ? (int) ghostgate_now() : time();

	// ここで必ず現在時刻をセットし直す
	$_SESSION['ghostgate_login_time']  = $now;
	$_SESSION['ghostgate_last_active'] = $now;
	$_SESSION['ghostgate_user_id']     = (int) $user->ID;

	if ( session_status() === PHP_SESSION_ACTIVE ) {
		@session_write_close();
	}
}, 10, 2 );

// ============================
// ログイン画面で当プラグインのキーだけ掃除
// ============================
// 不要なセッション開始を避けるため、セッションクッキーがある場合のみ開始してクリア
add_action( 'login_init', function () {
	if ( ! get_option( 'ghostgate_enable_session_control' ) ) {
		return;
	}

	$cookie_name  = session_name();
	$cookie_value = (string) ( filter_input( INPUT_COOKIE, $cookie_name, FILTER_UNSAFE_RAW ) ?? '' );

	// 既存セッションがあるときだけ触る
	if ( $cookie_value !== '' ) {
		if ( session_status() === PHP_SESSION_NONE ) {
			@session_start();
		}
		if ( session_status() === PHP_SESSION_ACTIVE ) {
			unset(
				$_SESSION['ghostgate_login_time'],
				$_SESSION['ghostgate_last_active'],
				$_SESSION['ghostgate_user_id']
			);
			@session_write_close();
		}
	}
}, 1 );

// ============================
// JS 読み込み（フロント/管理）
// ============================
add_action( 'wp_enqueue_scripts',    'ghostgate_enqueue_session_tracker_js' );
add_action( 'admin_enqueue_scripts', 'ghostgate_enqueue_session_tracker_js' );

function ghostgate_enqueue_session_tracker_js() {
	if ( ! is_user_logged_in() ) {
		return;
	}
	if ( ! get_option( 'ghostgate_enable_session_control' ) ) {
		return;
	}

	$handle = 'ghostgate-session-tracker';
	$src    = GHOSTGATE_URL . 'assets/js/ghost-session-tracker.js';
	$ver    = defined( 'GHOSTGATE_VERSION' ) ? GHOSTGATE_VERSION : '1.3.2';

	// UIは秒、JSにはmsで渡す。未設定時は5秒。
	$interval_ms = absint( get_option( 'ghostgate_session_interval', 5 ) ) * 1000;
	$nonce       = wp_create_nonce( 'ghostgate_touch_session' );
	$timeout_sec = absint( get_option( 'ghostgate_session_timeout', 1800 ) );
	$login_url   = add_query_arg( 'reauth', '1', wp_login_url() );

	wp_register_script( $handle, $src, array(), $ver, true );
	wp_localize_script(
		$handle,
		'ghostgateSession',
		array(
			// 既存仕様維持：action をクエリに含める
			'ajax_url'    => admin_url( 'admin-ajax.php?action=ghostgate_touch_session' ),
			'intervalMs'  => $interval_ms,
			'nonce'       => $nonce,
			// 任意：UI/拡張用
			'timeoutSec'  => $timeout_sec,
			'loginUrl'    => $login_url,
		)
	);
	wp_enqueue_script( $handle );
}

// ============================
// セッション延長（AJAX）
// ============================
// ログイン済みのみ
add_action( 'wp_ajax_ghostgate_touch_session', 'ghostgate_touch_session' );

function ghostgate_touch_session() {
	if ( ! is_user_logged_in() ) {
		wp_send_json_error( array( 'status' => 'unauthenticated' ), 401 );
	}

	// 機能OFF時は何もしない（200で返す）
	if ( ! get_option( 'ghostgate_enable_session_control' ) ) {
		wp_send_json_success( array( 'status' => __( 'session control disabled', 'ghostgate' ) ) );
	}

	// Nonce チェック
	check_ajax_referer( 'ghostgate_touch_session', 'nonce' );

	// セッション準備
	if ( session_status() === PHP_SESSION_NONE ) {
		@session_start();
	}

	$now = function_exists( 'ghostgate_now' ) ? (int) ghostgate_now() : time();
	$_SESSION['ghostgate_last_active'] = $now;

	wp_send_json_success( array( 'status' => __( 'session touched', 'ghostgate' ) ) );
}
