<?php
if (!defined('ABSPATH')) exit;

/**
 * GhostGate ログを記録し、指定日数より前のファイルを削除
 *
 * @param string $message ログに書き込むメッセージ
 * @param string $level ログレベル（info, warning, error）
 */
function ghostgate_log( $message, $level = 'info' ) {
    if ( ! get_option( 'ghostgate_enable_log_file', 1 ) ) {
        return;
    }

    $level_key = sanitize_key( (string) $level );
    $allowed   = array( 'debug', 'info', 'warning', 'error' );
    if ( ! in_array( $level_key, $allowed, true ) ) {
        $level_key = 'info';
    }

    $upload_dir  = wp_upload_dir();
    $uploads     = isset( $upload_dir['basedir'] ) ? $upload_dir['basedir'] : WP_CONTENT_DIR . '/uploads';
    $uploads     = wp_normalize_path( $uploads );
    $default_dir = trailingslashit( $uploads ) . 'ghostgate/logs';

    $base_path = (string) get_option( 'ghostgate_log_file_path' );
    $prefix    = 'ghostgate_logs';
    $log_dir   = $default_dir;

    if ( $base_path !== '' ) {
        $base_path = wp_normalize_path( $base_path );
        $info      = pathinfo( $base_path );

        $dir_try = isset( $info['dirname'] ) && $info['dirname'] !== '' ? $info['dirname'] : $default_dir;
        if ( strpos( $dir_try, $uploads ) === 0 ) {
            $log_dir = $dir_try;
        }

        if ( ! empty( $info['filename'] ) ) {
            $prefix = sanitize_file_name( $info['filename'] );
            if ( $prefix === '' ) {
                $prefix = 'ghostgate_logs';
            }
        }
    }

    if ( ! wp_mkdir_p( $log_dir ) ) {
        $log_dir = $default_dir;
        if ( ! wp_mkdir_p( $log_dir ) ) {
            return;
        }
    }

    $today     = date_i18n( 'Ymd' );
    $filename  = sanitize_file_name( "{$prefix}_{$today}.log" );
    if ( $filename === '' ) {
        $filename = 'ghostgate_logs_' . $today . '.log';
    }
    $file_path = trailingslashit( $log_dir ) . $filename;

    $timestamp = date_i18n( 'Y-m-d H:i:s' );
    $log_line  = sprintf( "[%s] [%s] %s%s", $timestamp, strtoupper( $level_key ), (string) $message, PHP_EOL );

    // --- WP_Filesystem ---
    require_once ABSPATH . 'wp-admin/includes/file.php';
    global $wp_filesystem;

    if ( ! function_exists( 'WP_Filesystem' ) ) {
        return;
    }
    if ( ! WP_Filesystem() ) {
        return;
    }
    if ( ! ( $wp_filesystem instanceof WP_Filesystem_Base ) ) {
        return;
    }

    // 書き込み
    $existing = '';
    if ( $wp_filesystem->exists( $file_path ) ) {
        $buf = $wp_filesystem->get_contents( $file_path );
        if ( is_string( $buf ) ) {
            $existing = $buf;
        }
    }

    $chmod = defined( 'FS_CHMOD_FILE' ) ? FS_CHMOD_FILE : 0644;
    if ( ! $wp_filesystem->put_contents( $file_path, $existing . $log_line, $chmod ) ) {
        return;
    }

    $retention_days = absint( get_option( 'ghostgate_log_retention_days', 60 ) );
    ghostgate_cleanup_old_logs( $log_dir, $prefix, $retention_days );
}




/**
 * 指定日数より古いログファイルを削除する
 */
function ghostgate_cleanup_old_logs($dir, $prefix, $keep_days = 60) {
    if (!file_exists($dir) || !is_dir($dir)) {
        return;
    }

    $files = glob(trailingslashit($dir) . $prefix . '_*.log');
    if (!$files) {
        return;
    }

    $threshold = strtotime('-' . intval($keep_days) . ' days');
    $deleted = 0;

    foreach ($files as $file) {
        if (preg_match('/_(\d{8})\.log$/', $file, $matches)) {
            $file_date = DateTime::createFromFormat('Ymd', $matches[1]);
            if ($file_date && $file_date->getTimestamp() < $threshold) {
                if (wp_delete_file($file)) {
                    $deleted++;
                } else {
                    // ...
                }
            }
        }
    }

}
