<?php
/**
 * Login slug feature for the GhostGate plugin.
 *
 * Rewrites/guards the login URL and applies rate limits & notifications.
 *
 * @package   GhostGate
 * @since     1.0.0
 * @license   GPL-2.0-or-later
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

add_action( 'init', 'ghostgate_apply_login_slug', 1 );

/**
 * Apply custom login slug rules and rewrite hooks.
 *
 * @return void
 */
function ghostgate_apply_login_slug() {

	if ( ! get_option( 'ghostgate_enable_login_slug' ) ) {
		return;
	}

	$custom_slug = trim( get_option( 'ghostgate_login_slug', 'ghost-login' ) );
	if ( $custom_slug === '' ) {
		return;
	}

	$expected_slug = '/' . ltrim( trim( $custom_slug ), '/' ); // → 例: /ghost-login
	$parsed_path = '';

	// REQUEST_URI を安全にパスへ正規化
	$parsed_path = ghostgate_get_request_path();

	$normalized_path = '/' . ltrim( $parsed_path, '/' );


	// ✅ カスタムスラッグでアクセスされた場合
//	if ( $normalized_path === $expected_slug ) {
	if ( untrailingslashit($normalized_path) === untrailingslashit($expected_slug) ) {

		// 🔒 認証コード未入力 → フォームを表示
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- read-only
		$code_param = (string) ( filter_input( INPUT_GET, 'code', FILTER_UNSAFE_RAW ) ?? '' );
		if ( $code_param === '' ) {


			$ip = ghostgate_get_user_ip();

			ghostgate_log( "認証コード未入力でログインスラッグにアクセス: IP={$ip}", 'warning' );

			if ( function_exists( 'ghostgate_render_code_form' ) ) {
				ghostgate_render_code_form(
					array(
						'max_length'  => absint( get_option( 'ghostgate_code_length', 4 ) ),
						'redirect_to' => esc_url( home_url( '/' . ltrim( $custom_slug, '/' ) ) ),
					)
				);
				exit;
			} else {
				wp_die(
					esc_html__( 'コード入力テンプレートが読み込まれていません。', 'ghostgate' ),
					esc_html__( 'GhostGate Error', 'ghostgate' )
				);
			}
		}

		// 🚫 Check if the IP is currently blocked (even correct requests will be denied)
		if ( ghostgate_is_ip_blocked() ) {

			$ip = ghostgate_get_user_ip();

			ghostgate_log( "Blocked IP attempted access: IP={$ip}", 'error' );

			wp_die(
				esc_html__( 'このIPアドレスは一時的にアクセスを制限されています。しばらくしてから再試行してください。', 'ghostgate' ),
				esc_html__( 'アクセス制限中', 'ghostgate' ),
				array( 'response' => 403 )
			);
		}

		// 🔒 コードが不正 → 試行カウント増加＋エラー表示
		if ( ! ghostgate_is_correct_auth_code() ) {
			ghostgate_increment_login_attempt();

			$ip = ghostgate_get_user_ip();

			// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- read-only
			$code_raw = (string) ( filter_input( INPUT_GET, 'code', FILTER_UNSAFE_RAW ) ?? '' );
			$code     = sanitize_text_field( $code_raw );

			// 任意（ログの見やすさ向上用）：桁数に合わない場合は印を付ける
			$len = absint( get_option( 'ghostgate_code_length', 4 ) );
			if ( $len < 1 || $len > 10 ) { $len = 4; }
			if ( $code !== '' && ! preg_match( '/^\d{' . $len . '}$/', $code ) ) {
				$code = '[invalid-format]';
			}

			ghostgate_log( "認証コードエラー: IP={$ip}, 入力={$code}", 'warning' );

			wp_die(
				esc_html__( '無効なコードです。再度入力してください。', 'ghostgate' ),
				esc_html__( '認証失敗', 'ghostgate' ),
				array( 'response' => 403 )
			);
		}

		$ip = ghostgate_get_user_ip();

		ghostgate_log( "認証成功: IP={$ip}", 'info' );

// ▼▼ これを追加（wp-login.php が期待するグローバルを用意）▼▼
global $user_login, $error, $errors, $interim_login, $login_link_separator, $pagenow;
if (!isset($user_login)) $user_login = '';
if (!isset($error))      $error      = '';
if (!isset($errors) || !($errors instanceof WP_Error)) $errors = new WP_Error();
if (!isset($interim_login))        $interim_login = false;
if (!isset($login_link_separator)) $login_link_separator = ' | ';
$pagenow = 'wp-login.php'; // 一部コードが参照するため念のため
// ▲▲ ここまで ▲▲


		require ABSPATH . 'wp-login.php';
		exit;
	}

	// ✅ wp-login.php に対する直接アクセス（GET）を検知してブロック
	$rm_raw         = (string) ( filter_input( INPUT_SERVER, 'REQUEST_METHOD', FILTER_UNSAFE_RAW ) ?? '' );
	$request_method = strtoupper( sanitize_text_field( $rm_raw ) );


	$ip = ghostgate_get_user_ip();


	if (
		$normalized_path === '/wp-login.php' &&
		$request_method === 'GET'
	) {
		/**
		 * Block direct access to wp-login.php when accessed via GET.
		 *
		 * - REQUEST_METHOD is a server-defined variable and trusted, but unslash/sanitize to meet PHPCS standards.
		 * - REMOTE_ADDR is also server-defined, not user input, but treated similarly to satisfy PHPCS.
		 */
		ghostgate_log( "wp-login.php への直接アクセスを拒否: IP={$ip}", 'warning' );

		wp_die(
			esc_html__( 'このページにはアクセスできません。', 'ghostgate' ),
			esc_html__( 'アクセス拒否', 'ghostgate' ),
			array( 'response' => 404 )
		);
	}
}

// 🔐 認証コードチェック
function ghostgate_is_correct_auth_code() {
	$mode           = get_option( 'ghostgate_code_mode', 'date' );
	$length         = intval( get_option( 'ghostgate_code_length', 4 ) );
	$enabled_mode   = get_option( 'ghostgate_enable_code_mode' );
	$enabled_length = get_option( 'ghostgate_enable_code_length' );

	if ( ! $enabled_mode && ! $enabled_length ) {
		return true;
	}

	$expected = ghostgate_generate_auth_code( $mode, $length );
	// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- read-only
	$provided = sanitize_text_field( (string) ( filter_input( INPUT_GET, 'code', FILTER_UNSAFE_RAW ) ?? '' ) );

	return hash_equals( $expected, $provided );
}

// 🔐 認証コード生成
function ghostgate_generate_auth_code( $mode, $length = 4 ) {
	$now = ghostgate_get_wp_timestamp_safe();

	switch ( $mode ) {
		case 'date':
			$code = date_i18n( 'md', $now );
			break;
		case 'reverse_date':
			$code = strrev( date_i18n( 'md', $now ) );
			break;
		case 'time':
			$code = date_i18n( 'Hi', $now );
			break;
		case 'reverse_time':
			$code = strrev( date_i18n( 'Hi', $now ) );
			break;
		case 'custom':
			$code = get_option( 'ghostgate_custom_code', '1234' );
			break;
		default:
			$code = substr( md5( date_i18n( 'YmdHi', $now ) ), 0, max( 3, min( 8, $length ) ) );
			break;
	}

	return $code;
}

/**
 * Return a timestamp suitable for comparisons/TTL.
 *
 * @return int Unix epoch.
 */
function ghostgate_get_wp_timestamp_safe() {
	if ( function_exists( 'wp_timestamp' ) ) {
		return wp_timestamp();
	}
	return time() + ( get_option( 'gmt_offset', 0 ) * HOUR_IN_SECONDS );
}

// 🚫 IPがブロックされているか判定
function ghostgate_is_ip_blocked( $type = 'login', $ip = null ) {
    if ( ! $ip ) {
        $ip = ghostgate_get_user_ip();
    }
    if ( $ip === 'unknown' ) {
        return false;
    }

    $block_key = "ghostgate_{$type}_block_{$ip}";
    $until     = get_transient( $block_key );
    if ( ! $until ) {
        return false;
    }

    $now = current_time( 'timestamp' );
    if ( (int) $until <= $now ) {
        // 期限切れ → 自動解除（掃除）
        delete_transient( $block_key );
        if ( function_exists( 'ghostgate_block_index_remove' ) ) {
            ghostgate_block_index_remove( $ip, $type );
        }
        return false;
    }
    return true;
}




// 🚫 試行カウントを増加.
function ghostgate_increment_login_attempt() {
    if ( ! get_option( 'ghostgate_enable_login_limit' ) ) {
        return;
    }

    $ip = ghostgate_get_user_ip();
    if ( $ip === 'unknown' ) {
        return;
    }

    $limit    = absint( get_option( 'ghostgate_login_attempt_limit', 3 ) );
    $window   = 30 * MINUTE_IN_SECONDS;
    $cooldown = absint( get_option( 'ghostgate_login_block_cooldown', 15 * MINUTE_IN_SECONDS ) );

    // ★ sanitize_key を使わない（他所と同じ「生のキー」に統一）
    $count_key = "ghostgate_attempts_$ip";
    $attempts  = (int) get_transient( $count_key );
    $attempts++;
    set_transient( $count_key, $attempts, $window );

    // 既にブロック中なら何もしない
    if ( function_exists( 'ghostgate_is_ip_blocked' ) && ghostgate_is_ip_blocked( 'login', $ip ) ) {
        return;
    }

    if ( $attempts >= $limit ) {
        if ( function_exists( 'ghostgate_block_ip' ) ) {
            ghostgate_block_ip( $ip, 'login' );
        } else {
            $until     = current_time( 'timestamp' ) + $cooldown;
            // ★ ここも sanitize しない
            $block_key = "ghostgate_login_block_$ip";
            set_transient( $block_key, $until, $cooldown );
            if ( function_exists( 'ghostgate_block_index_add' ) ) {
                ghostgate_block_index_add( $ip, 'login', $until );
            }
        }

        ghostgate_log( sprintf( '[LOGIN] BLOCK TRIGGERED → ip=%s, attempts=%d, limit=%d', $ip, $attempts, $limit ), 'warning' );

        if ( get_option( 'ghostgate_notify_on_block' ) && function_exists( 'ghostgate_notify_admin_on_block' ) ) {
            ghostgate_notify_admin_on_block( $ip, $attempts );
        }
    }
}




// 📩 管理者通知（ブロック時）.
function ghostgate_notify_admin_on_block( $ip, $attempts ) {
	$admin_email = get_option( 'admin_email' );

	$subject = __( '🚫 [GhostGate] Login attempt limit exceeded', 'ghostgate' );

	// translators: %1$s: IP address, %2$d: attempt count, %3$d: attempt limit
	$message = sprintf(
		__( "IP Address: %1\$s\nAttempts: %2\$d\nLimit: %3\$d\n\nThis IP has been blocked due to too many failed login attempts.", 'ghostgate' ),
		$ip,
		$attempts,
		get_option( 'ghostgate_login_attempt_limit' )
	);

	wp_mail( $admin_email, $subject, $message );

	// ログ記録（翻訳しない：内部処理用）.
	ghostgate_log(
		sprintf(
			'通知送信：ログイン試行回数超過 → IP=%s, 試行回数=%d',
			$ip,
			$attempts
		),
		'error'
	);
}



// Remember MEの無効化.
function ghostgate_enqueue_login_scripts() {
	if ( ! get_option( 'ghostgate_enable_session_control' ) ) {
		return;
	}

	$handle = 'ghostgate-disable-remember';
	$src    = GHOSTGATE_URL . 'assets/js/ghost-disable-remember.js';
	$ver    = defined( 'GHOSTGATE_VERSION' ) ? GHOSTGATE_VERSION : '1.3.2';

	// セッションが残っていた場合のクリア処理（ログアウト後の安全対策）
	if ( session_id() && defined( 'DOING_AJAX' ) && ! DOING_AJAX ) {
		session_destroy();
		$_SESSION = array();
	}

	wp_enqueue_script( $handle, $src, array(), $ver, true );
}
add_action( 'login_enqueue_scripts', 'ghostgate_enqueue_login_scripts' );
