<?php
if (!defined('ABSPATH')) exit;

// ========== Core utilities for GhostGate ==========
if ( ! function_exists( 'ghostgate_block_key' ) ) {
    /**
     * Transientキーを統一生成。
     * - "rest_count" 等はそのまま通す
     * - "rest"/"ajax"/"login"/"xmlrpc" は *_block を付与
     * - 未指定/不明な type は login_block にフォールバックして警告ログ
     */
    function ghostgate_block_key( $ip, $type = null ) {
        $ip   = preg_replace( '/[^0-9a-fA-F:\.\-]/', '', (string) $ip );
        //$type = is_string( $type ) ? strtolower( preg_replace( '/[^a-z0-9_]/i', '', $type ) ) : ''; 20260121修正
        $type = strtolower( preg_replace( '/[^a-z0-9_]/i', '', (string) $type ) );
        $no_block_suffixes = array( 'block', 'count', 'notify', 'attempts', 'cooldown' );
        $channels          = array( 'login', 'xmlrpc', 'ajax', 'rest' );

        // 1) 未指定/空 → 安全側(login_block)に倒して警告
        if ( $type === '' ) {
            if ( function_exists( 'ghostgate_log' ) ) {
                ghostgate_log( 'WARN ghostgate_block_key(): $type is empty; fallback to login_block', 'warning' );
            }
            return "ghostgate_login_block_{$ip}";
        }

        // 2) "rest_count" など末尾が既知の種別なら、そのまま通す
        if ( strpos( $type, '_' ) !== false ) {
            $suffix = substr( $type, strrpos( $type, '_' ) + 1 );
            if ( in_array( $suffix, $no_block_suffixes, true ) ) {
                return "ghostgate_{$type}_{$ip}";
            }
            // 既知以外のサフィックスは *_block に寄せる
            return "ghostgate_{$type}_block_{$ip}";
        }

        // 3) 単語: チャンネル名なら *_block、count/attempts系はそのまま
        if ( in_array( $type, $channels, true ) ) {
            return "ghostgate_{$type}_block_{$ip}";
        }
        if ( in_array( $type, $no_block_suffixes, true ) ) {
            return "ghostgate_{$type}_{$ip}";
        }

        // 4) 不明値 → login_block にフォールバックして警告
        if ( function_exists( 'ghostgate_log' ) ) {
            ghostgate_log( "WARN ghostgate_block_key(): unknown type '{$type}'; fallback to login_block", 'warning' );
        }
        return "ghostgate_login_block_{$ip}";
    }
}




if ( ! function_exists( 'ghostgate_update_block_index' ) ) {
	function ghostgate_update_block_index( $ip, $type, $ttl ) {
		$ip = sanitize_text_field( (string) $ip );

		// 'rest_block' や 'rest_count' → 'rest' に正規化
		$channel = preg_replace( '/_(block|count|notify|attempts|cooldown)$/', '', (string) $type );

		$idx = get_option( 'ghostgate_block_index', array() );
		if ( ! is_array( $idx ) ) { $idx = array(); }

		$now   = current_time( 'timestamp' );
		$until = ( $ttl && is_numeric( $ttl ) && (int)$ttl > 0 ) ? ( $now + (int)$ttl ) : 0;

		if ( $until > 0 ) {
			if ( ! isset( $idx[ $ip ] ) || ! is_array( $idx[ $ip ] ) ) { $idx[ $ip ] = array(); }
			$idx[ $ip ][ $channel ] = (int) $until;        // ← 新形式：$idx['1.2.3.4']['rest'] = 1234567890
		} else {
			// ttl=0/false のときは削除扱い
			if ( isset( $idx[ $ip ][ $channel ] ) ) { unset( $idx[ $ip ][ $channel ] ); }
			if ( isset( $idx[ $ip ] ) && empty( $idx[ $ip ] ) ) { unset( $idx[ $ip ] ); }
		}

		update_option( 'ghostgate_block_index', $idx, false );
	}
}


if ( ! function_exists( 'ghostgate_prune_block_index' ) ) {
	function ghostgate_prune_block_index( $unused = null ) {
		$idx = get_option( 'ghostgate_block_index', array() );
		if ( ! is_array( $idx ) ) { $idx = array(); }

		$now = current_time( 'timestamp' );

		foreach ( $idx as $ip => &$types ) {
			// 旧式（types/updated_at）で入っていた場合の互換：新式に丸める
			if ( isset( $types['types'] ) && is_array( $types['types'] ) ) {
				$normalized = array();
				foreach ( $types['types'] as $typeKey => $meta ) {
					$until = isset( $meta['expires_at'] ) ? (int) $meta['expires_at'] : 0;
					if ( $until > $now ) {
						$ch = preg_replace( '/_(block|count|notify|attempts|cooldown)$/', '', (string) $typeKey );
						$normalized[ $ch ] = $until;
					}
				}
				$types = $normalized; // 新式に差し替え
			}

			if ( ! is_array( $types ) ) { unset( $idx[ $ip ] ); continue; }

			foreach ( $types as $ch => $until ) {
				$until = is_numeric( $until ) ? (int) $until : 0;
				if ( $until <= $now ) {
					unset( $types[ $ch ] );             // 期限切れは削除
					continue;
				}
				// 実体のトランジェントが無ければ復元（TTL不整合対策）
				$tkey = ghostgate_block_key( $ip, $ch ); // -> *_block
				if ( false === get_transient( $tkey ) ) {
					set_transient( $tkey, (int) $until, max( 1, (int) $until - $now ) );
				}
			}

			if ( empty( $types ) ) { unset( $idx[ $ip ] ); }
		}
		unset( $types );

		update_option( 'ghostgate_block_index', $idx, false );
		return $idx; // 形式： [ '1.2.3.4' => ['rest'=>until, 'ajax'=>until, 'login'=>until], ... ]
	}
}


if ( ! function_exists( 'ghostgate_get_blocked_ips' ) ) {
function ghostgate_get_blocked_ips() {
    $raw = get_option( 'ghostgate_block_index', array() );
    $now = current_time( 'timestamp' );
    $idx = array();

    if ( ! is_array( $raw ) ) $raw = array();

    foreach ( $raw as $ip => $entry ) {
        // 旧式: ['types' => ['rest_block'=>['expires_at'=>...], ...], 'updated_at'=>...]
        if ( is_array( $entry ) && isset( $entry['types'] ) && is_array( $entry['types'] ) ) {
            foreach ( $entry['types'] as $typeKey => $meta ) {
                $until = isset( $meta['expires_at'] ) && is_numeric( $meta['expires_at'] ) ? (int) $meta['expires_at'] : 0;
                if ( $until > $now ) {
                    $ch = preg_replace( '/_(block|count|notify|attempts|cooldown)$/', '', (string) $typeKey );
                    $idx[ $ip ][ $ch ] = $until;
                }
            }
        }
        // 新式/混在: ['rest'=>until, 'rest_block'=>until, ...]
        elseif ( is_array( $entry ) ) {
            foreach ( $entry as $typeKey => $until ) {
                $until = is_numeric( $until ) ? (int) $until : 0;
                if ( $until > $now ) {
                    $ch = preg_replace( '/_(block|count|notify|attempts|cooldown)$/', '', (string) $typeKey );
                    $idx[ $ip ][ $ch ] = $until;
                }
            }
        }
    }

    // 実体トランジェントと同期（無ければ復元）
    foreach ( $idx as $ip => $types ) {
        foreach ( $types as $ch => $until ) {
            $tkey = ghostgate_block_key( $ip, $ch ); // -> *_block
            if ( false === get_transient( $tkey ) ) {
                set_transient( $tkey, (int) $until, max( 1, (int) $until - $now ) );
            }
        }
    }

    // 空のIPを除去し保存を新式で更新
    foreach ( $idx as $ip => $types ) {
        if ( empty( $types ) ) unset( $idx[ $ip ] );
    }
    update_option( 'ghostgate_block_index', $idx, false );

    // UIが期待する形式：
    // [ '1.2.3.4' => ['rest'=>until, 'ajax'=>until, 'login'=>until], ... ]
    return $idx;
}
}


function ghostgate_apply_rest_api_restrictions() {
    if (!is_admin() && get_option('ghostgate_block_unused_rest')) {

        // 1. REST API 全体の有効性は維持（全無効にはしない）
        // → apply_filters('rest_enabled') はコア側が持つので変更しない

        // 2. Application Passwords 無効
        add_filter('wp_is_application_passwords_available', '__return_false', 100);

        // 3. Gutenberg ブロックエディタを完全に無効化
        add_filter('use_block_editor_for_post_type', '__return_false', 100);

        // 4. REST API フィルタで個別ルートを除外
        add_filter('rest_endpoints', function ($endpoints) {

            // 未使用 or 脆弱性のあるREST APIルートを個別に無効化
            $block_routes = [
                '/wp/v2/media',
                '/wp/v2/plugins',
                '/wp/v2/themes',
                '/wp/v2/settings',
                '/jwt-auth/v1/token',
                '/wp/v2/users',
            ];

            foreach ($block_routes as $route) {
                if (isset($endpoints[$route])) {
                    unset($endpoints[$route]);
                }
            }

            return $endpoints;
        }, 100);

        // 5. Jetpack REST API 拡張を強制除去
        remove_all_actions('rest_api_init', 10); // Jetpack REST登録を防ぐ

        // 6. WooCommerce REST APIの基本クラスもチェックして必要なら除去（ただし任意）
        add_filter('rest_endpoints', function ($endpoints) {
            foreach ($endpoints as $route => $handlers) {
                if (str_contains($route, '/wc/v') || str_contains($route, '/wc-')) {
                    unset($endpoints[$route]);
                }
            }
            return $endpoints;
        }, 101);

        // 7. REST API によるユーザー列挙防止（再念押し）
        add_filter('rest_authentication_errors', function ($result) {
            if (!current_user_can('edit_posts')) {
                return new WP_Error(
                    'rest_cannot_access',
                    __('REST API access is restricted.', 'ghostgate'),
                    ['status' => rest_authorization_required_code()]
                );
            }
            return $result;
        });

        // 8. WP-CLI 判定は遮断不要（検出のみ）

        // 9. RESTに公開されてる投稿タイプ・タクソノミーは残す（ここで遮断はしない）
        // → 要件があれば `register_post_type()` や `register_taxonomy()` の `$show_in_rest` を操作
    }
}

add_action('init', 'ghostgate_apply_rest_api_restrictions', 20);


add_action('update_option_ghostgate_block_unused_rest', 'ghostgate_update_block_unused_rest', 10, 2);

function ghostgate_update_block_unused_rest($old_value, $new_value) {
    // ログ出力（翻訳対応）
    ghostgate_log(sprintf(
        '[GhostGate] %s: %s → %s',
        __('REST API制限の設定が変更されました', 'ghostgate'),
        $old_value,
        $new_value
    ));

    // フック発火時点では反映済みなので、init側のフィルタが次回リロードで有効になる

    // 即時反映したいなら、一時的にフック実行（あくまで実行対象は次回リクエストで効く）
    if ((int) $new_value === 1) {
        do_action('ghostgate_rest_restriction_enabled');
    } else {
        do_action('ghostgate_rest_restriction_disabled');
    }
}



add_filter('rest_pre_dispatch', 'ghostgate_maybe_rate_limit_rest_api', 10, 3);

function ghostgate_maybe_rate_limit_rest_api($result, $server, $request) {
    if (!get_option('ghostgate_enable_rest_limit')) {
        return $result;
    }

    // 管理者・ログイン済み除外（必要なら外す）
    if (is_user_logged_in()) {
        return $result;
    }

    $ip        = ghostgate_get_user_ip();
    $limit     = absint(get_option('ghostgate_rest_limit_count', 20));
    $cooldown  = absint(get_option('ghostgate_rest_limit_cooldown', 300)); // 秒

    $count_key  = ghostgate_block_key( $ip, 'rest_count' ); // 例: ghostgate_rest_count_127.0.0.1
    $block_key  = ghostgate_block_key( $ip, 'rest' );       // 例: ghostgate_rest_block_127.0.0.1
    $notify_key = ghostgate_block_key( $ip, 'rest_notify' );

    // ✅ ブロック中
    $until = get_transient( $block_key ); // 値は「解除予定UNIX秒」
    if ( false !== $until ) {
        $now = current_time('timestamp');
        if ( is_numeric($until) && (int)$until > $now ) {
            ghostgate_log( sprintf('[REST] Blocked IP (cooldown active): %s -> until %d', $ip, $until), 'warning' );
            return new WP_Error( 'rest_blocked', esc_html__('Too Many Requests', 'ghostgate'), ['status' => 429] );
        }
        // 期限切れなら掃除（TTL不整合対策）
        delete_transient( $block_key );
        ghostgate_block_index_remove( $ip, 'rest' ); 
    }

    // ✅ カウント更新
    $count = (int) get_transient($count_key);
    $count++;
    set_transient($count_key, $count, 60); // 1分カウント

    // ✅ 制限超過 → 初回のみブロック処理
    if ($count > $limit) {
        if (!get_transient($block_key)) {
            $until = current_time('timestamp') + $cooldown;
            set_transient($block_key, $until, $cooldown);
            ghostgate_block_index_add( $ip, 'rest', $until );

            ghostgate_log(sprintf(
                '[REST] BLOCK TRIGGERED → ip=%s, count=%d, limit=%d',
                $ip,
                $count,
                $limit
            ), 'warning');

            if (!get_transient($notify_key)) {
                set_transient($notify_key, 1, $cooldown);

                $subject = __('[GhostGate] REST API Rate Limit Triggered', 'ghostgate');
                $message = sprintf(
                    __("IP Address: %1\$s\nCount: %2\$d\nLimit: %3\$d\nCooldown: %4\$d seconds", 'ghostgate'),
                    $ip,
                    $count,
                    $limit,
                    $cooldown
                );
                wp_mail(get_option('admin_email'), $subject, $message);

                ghostgate_log(sprintf(
                    '[REST] Notification sent → ip=%s, count=%d, limit=%d, cooldown=%ds',
                    $ip,
                    $count,
                    $limit,
                    $cooldown
                ), 'warning');
            }
        }

        return new WP_Error(
            'rest_rate_limited',
            esc_html__('Rate limit exceeded', 'ghostgate'),
            ['status' => 429]
        );
    }

    return $result;
}

function ghostgate_block_index_add( $ip, $type, $until ) {
    $idx = get_option( 'ghostgate_block_index', array() );
    if ( ! is_array( $idx ) ) { $idx = array(); }
    if ( ! isset( $idx[ $ip ] ) ) { $idx[ $ip ] = array(); }
    $idx[ $ip ][ $type ] = (int) $until;
    update_option( 'ghostgate_block_index', $idx, false );
}

function ghostgate_block_index_remove( $ip, $type = null ) {
    $idx = get_option( 'ghostgate_block_index', array() );
    if ( ! is_array( $idx ) || ! isset( $idx[ $ip ] ) ) {
        return;
    }
    if ( $type ) {
        unset( $idx[ $ip ][ $type ] );
        if ( empty( $idx[ $ip ] ) ) {
            unset( $idx[ $ip ] );
        }
    } else {
        unset( $idx[ $ip ] );
    }
    update_option( 'ghostgate_block_index', $idx, false );
}


if ( ! function_exists( 'ghostgate_delete_any_transient' ) ) {
    function ghostgate_delete_any_transient( $name ) {
        delete_transient( $name );
        if ( function_exists( 'delete_site_transient' ) ) {
            delete_site_transient( $name );
        }
    }
}

function ghostgate_unblock_ip( $ip ) {
    if ( ! $ip || $ip === 'unknown' || ! filter_var( $ip, FILTER_VALIDATE_IP ) ) {
        return false;
    }

    // チャネル×サフィックスを一括掃除（cooldown を必ず含める）
    $channels = array( 'login', 'ajax', 'rest' );
    $suffixes = array( 'block', 'count', 'notify', 'attempts', 'cooldown' );

    foreach ( $channels as $ch ) {
        foreach ( $suffixes as $suf ) {
            // ghostgate_block_key は先ほどの修正版（*_count/notify/attempts/cooldown には _block を付けない）
            $key = ghostgate_block_key( $ip, "{$ch}_{$suf}" );
            ghostgate_delete_any_transient( $key );
        }
    }

    // 互換・誤実装クリーンアップ（過去に作られた可能性のあるキー）
    ghostgate_delete_any_transient( ghostgate_block_key( $ip, 'attempts' ) );        // ghostgate_attempts_{ip}
    ghostgate_delete_any_transient( ghostgate_block_key( $ip, 'attempts_block' ) );  // 誤キーの掃除
    // 旧 sanitize_key() 名の掃除（必要なら）
    $maybe_legacy = array(
        sanitize_key( "ghostgate_login_block_{$ip}" ),
        sanitize_key( "ghostgate_attempts_{$ip}" ),
        sanitize_key( "ghostgate_rest_cooldown_{$ip}" ),
        sanitize_key( "ghostgate_ajax_cooldown_{$ip}" ),
        sanitize_key( "ghostgate_rest_block_{$ip}" ),
        sanitize_key( "ghostgate_ajax_block_{$ip}" ),
        sanitize_key( "ghostgate_ajax_count_{$ip}" ),
        sanitize_key( "ghostgate_ajax_notify_{$ip}" ),
    );
    foreach ( $maybe_legacy as $k ) {
        ghostgate_delete_any_transient( $k );
    }

    // 索引を使っているなら完全削除
    if ( function_exists( 'ghostgate_block_index_remove' ) ) {
        ghostgate_block_index_remove( $ip, null );
    }

    return true;
}


// 置換推奨：core.php の関数をこれに差し替え（他でも使い回し可）
function ghostgate_get_user_ip() {
    // できるだけ REMOTE_ADDR を優先（サーバ由来）
    $candidates = array();

    $ra = (string) ( filter_input( INPUT_SERVER, 'REMOTE_ADDR', FILTER_UNSAFE_RAW ) ?? '' );
    if ( $ra !== '' ) { $candidates[] = $ra; }


    // 代理ヘッダは任意（必要な場合のみ使用）。先頭要素だけ使用。
    $xff = (string) ( filter_input( INPUT_SERVER, 'HTTP_X_FORWARDED_FOR', FILTER_UNSAFE_RAW ) ?? '' );
    if ( $xff !== '' ) {

        $parts = array_map( 'trim', explode( ',', $xff ) );
        if ( ! empty( $parts[0] ) ) {
            $candidates[] = $parts[0];
        }
    }

    $hci = (string) ( filter_input( INPUT_SERVER, 'HTTP_CLIENT_IP', FILTER_UNSAFE_RAW ) ?? '' );
    if ( $hci !== '' ) {
        $candidates[] = $hci;
    }


    foreach ( $candidates as $ip ) {
        $ip = sanitize_text_field( $ip );
        if ( filter_var( $ip, FILTER_VALIDATE_IP ) ) {
            return $ip;
        }
    }
    return 'unknown';
}



//json秘匿化処理
add_filter( 'rest_endpoints', 'ghostgate_hide_json_endpoints', 102 );
function ghostgate_hide_json_endpoints( $endpoints ) {
    // REST サーバ不在なら何もしない（早期リターン）
    if ( ! function_exists( 'rest_get_server' ) ) return $endpoints;

    global $ghostgate_bypass_json_filter;

    // ✅ 秘匿 OFF or UI 構築中（バイパス中）は何もしない
    $enabled = get_option( 'ghostgate_hide_json_endpoints', '0' );
    if ( $enabled !== '1' || ! empty( $ghostgate_bypass_json_filter ) ) {
        return $endpoints;
    }

    // ✅ 許可ルートの取得（配列のみ許可）— 正規表現を壊さない
    $allowed_routes = get_option( 'ghostgate_json_allowed_routes', array() );
    if ( ! is_array( $allowed_routes ) ) $allowed_routes = array();
    // 文字列化 & UTF-8 妥当性のみ（sanitize_key などは絶対に使わない）
    $allowed_routes = array_map( 'strval', $allowed_routes );
    $allowed_routes = array_map( 'wp_check_invalid_utf8', $allowed_routes );

    // ✅ `/` は常に許可（ルート一覧のルート）
    if ( ! in_array( '/', $allowed_routes, true ) ) {
        $allowed_routes[] = '/';
    }

    // ✅ 自作プレフィックス（カンマ区切り → 正規化）
    $prefixes_raw = (string) get_option( 'ghostgate_json_allowed_prefixes', '' );
    $prefixes = array_filter( array_map( static function( $p ) {
        $p = trim( (string) $p );
        if ( $p === '' ) return '';
        // 先頭に `/` を付与、連続スラッシュも畳む
        $p = ( strpos( $p, '/' ) === 0 ) ? $p : "/{$p}";
        $p = preg_replace( '#/{2,}#', '/', $p );
        return $p;
    }, explode( ',', $prefixes_raw ) ) );

    // ✅ 高速化のため、許可ルートを set 化
    $allow_set = array_fill_keys( $allowed_routes, true );

    // ✅ フィルタリング
    foreach ( $endpoints as $route => $handler ) {
        // 完全一致で許可
        if ( isset( $allow_set[ $route ] ) ) {
            continue;
        }
        // 接頭辞で許可
        $allowed_by_prefix = false;
        foreach ( $prefixes as $px ) {
            if ( $px !== '' && strpos( $route, $px ) === 0 ) {
                $allowed_by_prefix = true;
                break;
            }
        }
        if ( $allowed_by_prefix ) {
            continue;
        }

        // ここまで来たら非表示
        unset( $endpoints[ $route ] );
    }

    return $endpoints;
}



//名前空間の削除
add_filter('rest_pre_serve_request', 'ghostgate_clean_json_namespaces', 20, 4);
function ghostgate_clean_json_namespaces($served, $result, $request, $server) {
    // wp-jsonのトップ（ルート）へのアクセス以外は無視
    if ($request->get_route() !== '/') return $served;

    // 秘匿機能がOFFなら何もしない
    if (!get_option('ghostgate_hide_json_endpoints')) return $served;

    $data = $result->get_data();
    if (!is_array($data) || !isset($data['namespaces'])) return $served;

    // 実際に残っているルート一覧から有効な名前空間を抽出
    $routes = $server->get_routes();
    $valid_namespaces = [];

    foreach (array_keys($routes) as $route) {
        if (preg_match('#^/([^/]+)/#', $route, $m)) {
            $valid_namespaces[] = $m[1];
        }
    }

    $valid_namespaces = array_values(array_unique($valid_namespaces));
    $data['namespaces'] = $valid_namespaces;

    // データ上書き
    $result->set_data($data);

    return false; // ← 上書き応答として出力
}



// 現在時刻（サイトのタイムゾーン基準のUNIXタイム）
if ( ! function_exists('ghostgate_now') ) {
    function ghostgate_now() : int {
        if ( function_exists('current_datetime') ) {
            return (int) current_datetime()->getTimestamp(); // WP 5.3+
        }
        return (int) current_time('timestamp'); // フォールバック
    }
}

// 日付フォーマット（サイトTZで表示）
if ( ! function_exists('ghostgate_wp_date') ) {
    function ghostgate_wp_date( string $format, int $ts ) : string {
        if ( function_exists('wp_date') ) {
            return wp_date($format, $ts);       // WP 5.3+
        }
        return date_i18n($format, $ts);         // フォールバック（古いWP）
    }
}


// セッション開始と記録
function ghostgate_start_session_timer() {
    if ( ! get_option('ghostgate_enable_session_control') ) return;
    if ( ! is_user_logged_in() ) return;

    if ( session_status() === PHP_SESSION_NONE ) {
        session_start();
    }

    if ( wp_doing_ajax() || ( defined('REST_REQUEST') && REST_REQUEST ) ) return;

    global $pagenow;
    $method = filter_input( INPUT_SERVER, 'REQUEST_METHOD' );
    if ( is_admin() && 'POST' === $method && isset( $pagenow ) && 'options.php' === $pagenow ) {
        return; // 設定保存中は誤爆させない
    }

    $timeout = absint( get_option('ghostgate_session_timeout', 1800) );
    $now     = ghostgate_now();

    // ★グレース（ON/変更直後の数秒）は判定しない【ここに追加】
    $grace_until = (int) get_option('ghostgate_session_grace_until', 0);
    if ( $grace_until && $now < $grace_until ) {
        // グレース中でも初期値は入れておく
        if ( empty($_SESSION['ghostgate_login_time']) )  $_SESSION['ghostgate_login_time']  = $now;
        if ( empty($_SESSION['ghostgate_last_active']) ) $_SESSION['ghostgate_last_active'] = $now;
        if ( empty($_SESSION['ghostgate_user_id']) )     $_SESSION['ghostgate_user_id']     = (int) get_current_user_id();
        if ( session_status() === PHP_SESSION_ACTIVE ) { session_write_close(); }
        return;
    }
    // グレースが終わっていれば掃除（任意）
    if ( $grace_until && $now >= $grace_until ) {
        delete_option('ghostgate_session_grace_until');
    }

    // 初期化（ログイン時刻・最終操作時刻・ユーザーID）
    if ( empty($_SESSION['ghostgate_login_time']) )  $_SESSION['ghostgate_login_time']  = $now;
    if ( empty($_SESSION['ghostgate_last_active']) ) $_SESSION['ghostgate_last_active'] = $now;
    if ( empty($_SESSION['ghostgate_user_id']) )     $_SESSION['ghostgate_user_id']     = (int) get_current_user_id();

    // アイドル時間判定
    $elapsed = max( 0, $now - (int) $_SESSION['ghostgate_last_active'] );
    if ( $elapsed >= $timeout ) {
        if ( session_status() === PHP_SESSION_ACTIVE ) { session_write_close(); }
        wp_logout();
        wp_safe_redirect( wp_login_url( add_query_arg( 'reauth', '1' ) ) );
        exit;
    }
}
add_action('init', 'ghostgate_start_session_timer', 20);


// ON/OFF 保存時に、セッション値をクリア/初期化し、ON直後は短いグレースを付与
add_action('update_option_ghostgate_enable_session_control', function ($old, $new) {
    // 管理画面の保存＝ログイン中のはずだが、念のためチェック
    if ( ! is_user_logged_in() ) {
        return;
    }
    if ( session_status() === PHP_SESSION_NONE ) {
        session_start();
    }

    if ( (int) $new === 0 ) {
        // OFF → 自プラグインのキーだけ削除（セッション全体は壊さない）
        unset(
            $_SESSION['ghostgate_last_active'],
            $_SESSION['ghostgate_login_time'],
            $_SESSION['ghostgate_user_id']
        );
        delete_option('ghostgate_session_grace_until');
    } else {
        // ON → 現在時刻で初期化
        $now = ghostgate_now();
        $_SESSION['ghostgate_last_active'] = $now;
        $_SESSION['ghostgate_login_time']  = $now;
        $_SESSION['ghostgate_user_id']     = (int) get_current_user_id();

        // ON直後の“即失効”を避けるため短いグレース（10秒）
        update_option('ghostgate_session_grace_until', $now + 10);
    }

    if ( session_status() === PHP_SESSION_ACTIVE ) {
        session_write_close();
    }
}, 10, 2);


/**
 * GhostGate内で使用するHTMLタグホワイトリストを返す。
 */
function ghostgate_get_allowed_html_tags() {
    return [
        'form' => [
            'action' => true, 'method' => true, 'id' => true, 'class' => true,
        ],
        'input' => [
            'type' => true, 'name' => true, 'value' => true, 'checked' => true,
            'id' => true, 'class' => true, 'style' => true, 'pattern' => true,
            'maxlength' => true, 'placeholder' => true, 'disabled' => true,
            'autocomplete' => true, 'onclick' => true, 'onchange' => true,
            'data-mode' => true, 'data-toggle' => true, 'data-target' => true,
            'data-action' => true, 'data-label' => true, 'data-*' => true,
        ],
        'label' => ['for' => true, 'class' => true, 'style' => true],
        'select' => ['name' => true, 'id' => true, 'class' => true, 'style' => true, 'disabled' => true],
        'option' => ['value' => true, 'selected' => true, 'disabled' => true],
        'textarea' => [
            'name' => true, 'id' => true, 'class' => true, 'rows' => true,
            'cols' => true, 'style' => true, 'placeholder' => true,
        ],
        'button' => [
            'type' => true, 'id' => true, 'class' => true, 'style' => true,
            'data-action' => true, 'data-toggle' => true, 'data-target' => true,
            'data-label' => true, 'data-*' => true,
        ],
        'table' => ['class' => true, 'style' => true],
        'tr' => ['class' => true, 'id' => true, 'style' => true],
        'td' => ['class' => true, 'colspan' => true, 'rowspan' => true, 'style' => true],
        'th' => ['class' => true, 'style' => true],
        'fieldset' => ['class' => true, 'id' => true, 'style' => true],
        'legend' => ['style' => true],
        'div' => ['class' => true, 'id' => true, 'style' => true, 'data-toggle' => true, 'data-target' => true, 'data-mode' => true, 'data-*' => true],
        'span' => [
            'class' => true, 'style' => true, 'id' => true,
            'data-elapsed' => true, 'data-remaining' => true,
            'data-state' => true, 'data-step' => true,
            'data-toggle' => true, 'data-target' => true,
            'data-*' => true,
        ],
        'p' => ['class' => true, 'style' => true],
        'br' => [],
        'strong' => [],
        'em' => [],
        'ul' => ['class' => true, 'style' => true],
        'li' => ['class' => true, 'style' => true],
        'code' => ['class' => true],
        'pre' => ['class' => true],
        'hr' => ['class' => true, 'id' => true, 'style' => true],
    ];
}


// REQUEST_URI を安全に「パス」に正規化して返す（直参照回避）
if ( ! function_exists( 'ghostgate_get_request_path' ) ) {
	function ghostgate_get_request_path(): string {
		// $_SERVER を直接参照せず取得
		$raw = filter_input( INPUT_SERVER, 'REQUEST_URI', FILTER_UNSAFE_RAW );
		if ( ! is_string( $raw ) ) {
			return '';
		}

		$raw  = esc_url_raw( $raw );                         // 入力側サニタイズ
		$path = (string) wp_parse_url( $raw, PHP_URL_PATH ); // パスのみ抽出
		$path = wp_normalize_path( $path );                  // / に正規化

		return $path;
	}
}


/**
 * WP_Filesystem を安全に初期化してハンドルを返す。
 * 失敗時は false。
 */
if ( ! function_exists( 'ghostgate_fs' ) ) {
	function ghostgate_fs() {
		static $ok = null;
		global $wp_filesystem;

		if ( $ok === true && $wp_filesystem instanceof WP_Filesystem_Base ) {
			return $wp_filesystem;
		}

		require_once ABSPATH . 'wp-admin/includes/file.php';

		if ( ! WP_Filesystem() ) {
			return false;
		}
		if ( ! ( $wp_filesystem instanceof WP_Filesystem_Base ) ) {
			return false;
		}
		$ok = true;
		return $wp_filesystem;
	}
}

/** ファイル存在 */
if ( ! function_exists( 'ghostgate_fs_exists' ) ) {
	function ghostgate_fs_exists( $path ) {
		$fs = ghostgate_fs();
		if ( ! $fs ) return false;
		return $fs->exists( $path );
	}
}

/** 読み込み（失敗で空文字） */
if ( ! function_exists( 'ghostgate_fs_get' ) ) {
	function ghostgate_fs_get( $path ) {
		$fs = ghostgate_fs();
		if ( ! $fs ) return '';
		$buf = $fs->get_contents( $path );
		return is_string( $buf ) ? $buf : '';
	}
}

/** 書き込み（bool） */
if ( ! function_exists( 'ghostgate_fs_put' ) ) {
	function ghostgate_fs_put( $path, $contents, $chmod = null ) {
		$fs = ghostgate_fs();
		if ( ! $fs ) return false;
		if ( $chmod === null && defined( 'FS_CHMOD_FILE' ) ) {
			$chmod = FS_CHMOD_FILE;
		}
		return (bool) $fs->put_contents( $path, $contents, $chmod );
	}
}


