<?php
// inc/block-admin-ajax.php

if (!defined('ABSPATH')) exit;

add_action('init', 'ghostgate_throttle_admin_ajax');

function ghostgate_throttle_admin_ajax() {
    if (!get_option('ghostgate_enable_ajax_limit')) return;
    if (!defined('DOING_AJAX') || !DOING_AJAX) return;

    $ip = ghostgate_get_ip();
    $limit = intval(get_option('ghostgate_ajax_limit', 30));
    $cooldown = intval(get_option('ghostgate_ajax_cooldown', 60)); // 秒

    $count_key  = ghostgate_block_key( $ip, 'ajax_count' ); // ghostgate_ajax_count_{ip}
    $block_key  = ghostgate_block_key( $ip, 'ajax' );       // ghostgate_ajax_block_{ip}
    $notify_key = ghostgate_block_key( $ip, 'ajax_notify' );

    // ✅ すでにブロック中の場合 → 429即返
    $until = get_transient( $block_key ); // 値は「解除予定UNIX秒」
    if ( false !== $until ) {
        $now = current_time('timestamp');

        if ( is_numeric($until) && (int)$until > $now ) {
            $retry = max( 1, (int)$until - $now );
            ghostgate_log( sprintf('[AJAX] Blocked IP: %s (cooldown active, %ds left)', $ip, $retry), 'warning' );
            if ( ! headers_sent() ) {
                header( 'Retry-After: ' . $retry );
                status_header(429);
            }
            echo esc_html__('Too Many Requests (GhostGate)', 'ghostgate');
            exit;
        }

        // 期限切れ → 掃除（TTL不整合対策）
        delete_transient( $block_key );
        if ( function_exists('ghostgate_block_index_remove') ) {
            ghostgate_block_index_remove( $ip, 'ajax' );
        }
    }

    // ✅ カウント更新
    $count = (int) get_transient($count_key);
    $count++;
    set_transient($count_key, $count, 60);

    // ✅ 制限超過 → ブロック & 通知（初回のみ）
    if ($count > $limit) {
        if ( ! get_transient( $block_key ) ) {

            // ▼ 追加：UA/Referer を安全に取得
            $ua_raw = (string) ( filter_input( INPUT_SERVER, 'HTTP_USER_AGENT', FILTER_UNSAFE_RAW ) ?? '' );
            $ua     = $ua_raw !== '' ? sanitize_text_field( $ua_raw ) : 'unknown';
            if ( strlen( $ua ) > 512 ) { $ua = substr( $ua, 0, 512 ); } // 長すぎ対策

            $ref = wp_get_referer();
            if ( $ref ) {
                $ref = esc_url_raw( $ref );
            } else {
                $ref_raw = (string) ( filter_input( INPUT_SERVER, 'HTTP_REFERER', FILTER_UNSAFE_RAW ) ?? '' );
                $ref     = $ref_raw !== '' ? esc_url_raw( $ref_raw ) : 'unknown';
            }


            $until = current_time('timestamp') + $cooldown;     // ← 解除予定UNIX秒を算出
            set_transient( $block_key, $until, $cooldown );     // ← 値＝解除予定時刻で保存
            if ( function_exists('ghostgate_block_index_add') ) {
                ghostgate_block_index_add( $ip, 'ajax', $until ); // ← UI用の索引にも登録
            }

            ghostgate_log( sprintf(
                '429 Too Many AJAX Requests | IP=%s | Count=%d | UA=%s | REF=%s',
                $ip,
                $count,
                $ua,
                $ref
            ), 'warning' );

            // ✅ 通知は初回のみ
            if ( get_option('ghostgate_ajax_notify') && ! get_transient($notify_key) ) {
                set_transient( $notify_key, 1, $cooldown );

                ghostgate_log( sprintf('🔔 Admin notification triggered: %s', $ip), 'info' );

                $subject = __('[GhostGate] AJAX Request Limit Exceeded', 'ghostgate');
                $message = sprintf(
                    __("IP: %1\$s\nCount: %2\$d times\nLimit: %3\$d times\nCooldown: %4\$d seconds", 'ghostgate'),
                    $ip, $count, $limit, $cooldown
                );
                wp_mail( get_option('admin_email'), $subject, $message );
            }
        }


        if (!headers_sent()) {
            header('Retry-After: ' . $cooldown);
            status_header(429);
        }

        echo esc_html__('Too Many Requests (GhostGate)', 'ghostgate');
        exit;
    }
}



//REST APIによるユーザー参照のブロック
add_action('template_redirect', function () {
    if (is_admin()) return;

    if (get_option('ghostgate_block_author_enum', '1')) {
        // 1. ?author=1 のようなGET直接指定（直参照しない）
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- read-only
        $author_raw = filter_input( INPUT_GET, 'author', FILTER_SANITIZE_NUMBER_INT );
        $author     = absint( (string) ( $author_raw ?? '' ) );
        if ( $author > 0 ) {
            ghostgate_block_author_access();
        }


        // 2. is_author() 判定（rewriteによる author アクセス時も検出）
        if (is_author()) {
            ghostgate_block_author_access();
        }
    }
});

function ghostgate_block_author_access() {
    if (function_exists('ghostgate_log_event')) {
        $ip = ghostgate_get_ip();
        ghostgate_log_event(sprintf(
            // translators: %s is the IP address that attempted author enumeration.
            __('🔒 Blocked author enumeration attempt (IP: %s)', 'ghostgate'),
            $ip
        ));
    }

    wp_die(
        esc_html__('Author enumeration is blocked for security reasons.', 'ghostgate'),
        esc_html__('Access Denied', 'ghostgate'),
        ['response' => 403]
    );
}




//IP取得関数
function ghostgate_get_ip() {
    return ghostgate_get_user_ip(); // core.php の共通関数に統一
}


// XML-RPCの遮断：.htaccessに追記
function ghostgate_apply_xmlrpc_blocking( $enabled ) {
	global $wp_filesystem; // ← 必須

	$htaccess_path = wp_normalize_path( ABSPATH . '.htaccess' );

	require_once ABSPATH . 'wp-admin/includes/file.php';
	if ( empty( $wp_filesystem ) ) {
		WP_Filesystem();
	}

	// WP_Filesystem 利用可能 & .htaccess が存在して書き込み可でなければ終了
	if (
		! ( isset( $wp_filesystem ) && $wp_filesystem instanceof WP_Filesystem_Base ) ||
		! $wp_filesystem->exists( $htaccess_path ) ||
		! $wp_filesystem->is_writable( $htaccess_path )
	) {
		return;
	}

	$start_tag = '# BEGIN GhostGate XML-RPC Block';
	$end_tag   = '# END GhostGate XML-RPC Block';

	// .htaccess 挿入内容（翻訳対象外）
	$block_rules = $start_tag . "\n" .
		"# このブロックは GhostGate プラグインにより自動追加されました。\n" .
		"# https://wordpress.org/plugins/ghostgate/\n\n" .
		"<Files \"xmlrpc.php\">\n" .
		"    Require all denied\n" .
		"</Files>\n" .
		$end_tag;

	// 読み取り：file_get_contents → WP_Filesystem に変更
	$content = $wp_filesystem->get_contents( $htaccess_path );
	if ( $content === false ) {
		return;
	}

	// 既存ブロックを除去
	$pattern     = '/' . preg_quote( $start_tag, '/' ) . '.*?' . preg_quote( $end_tag, '/' ) . '(\r?\n)?/s';
	$new_content = preg_replace( $pattern, '', $content );

	if ( $new_content !== $content ) {
		ghostgate_log( __( '🔧 Existing XML-RPC block removed from .htaccess.', 'ghostgate' ) );
	} else {
		ghostgate_log( __( 'ℹ️ No existing XML-RPC block found in .htaccess.', 'ghostgate' ) );
	}

	// 有効化なら追記
	if ( $enabled ) {
		$new_content = rtrim( $new_content ) . "\n\n" . $block_rules . "\n";
		ghostgate_log( __( '🛡️ XML-RPC block will be added to .htaccess.', 'ghostgate' ) );
	}

	// 書き込み：file_put_contents → WP_Filesystem に変更
	$ok = $wp_filesystem->put_contents( $htaccess_path, $new_content, FS_CHMOD_FILE );
	if ( ! $ok ) {
		ghostgate_log( __( '❌ Failed to write to .htaccess file.', 'ghostgate' ), 'error' );
		return;
	}

	ghostgate_log(
		sprintf(
			// translators: %d is the number of bytes written to the .htaccess file.
			__( '✅ Successfully wrote to .htaccess. Bytes written: %d', 'ghostgate' ),
			strlen( $new_content )
		)
	);
}


// オプション登録・保存時にブロック処理発火
register_setting('ghostgate_options', 'ghostgate_disable_xmlrpc', [
    'sanitize_callback' => function ($value) {
        $enabled = $value === '1';
        ghostgate_apply_xmlrpc_blocking($enabled);
        return absint($value);
    }
]);


// プレビュー直叩き遮断（URLに ?preview が載る通常のWPプレビューをブロック）
add_action('template_redirect', function () {
    // オプションOFFなら何もしない
    if (!get_option('ghostgate_block_preview', 0)) {
        return;
    }

    // 管理画面は対象外（/wp-admin 内での挙動を壊さないため）
    if (is_admin()) {
        return;
    }

    // 既定のバイパス権限（必要なら filter で変更可）
    $bypass_cap = apply_filters('ghostgate_preview_bypass_cap', 'manage_options');

    // is_preview は投稿プレビューで true
    $has_preview_flag = is_preview();

    // 念のためクエリでも判定
    if (!$has_preview_flag) {
        $has_preview_flag = (isset($_GET['preview']) || isset($_GET['preview_id']));
    }

    // さらに「ブロックするか」を外から制御したいときのフック
    $should_block = apply_filters('ghostgate_preview_should_block', $has_preview_flag);

    if ($should_block) {
        // 例外: 一部ロールには許可したい場合（任意）
        if ($bypass_cap && current_user_can($bypass_cap)) {
            return;
        }

        // 403 を返して終了（WPのやり方で）
        status_header(403);
        wp_die(
            esc_html__('プレビューは無効化されています。', 'ghostgate'),
            esc_html__('Forbidden', 'ghostgate'),
            ['response' => 403]
        );
    }
}, 1);
