<?php
// セキュリティ: 直アクセス防止
if (!defined('ABSPATH')) {
    exit;
}

// inc/admin-ui.php（新規）
add_action('admin_menu', function () {
    add_menu_page(
        __('GhostGate', 'ghostgate'),
        'GhostGate',
        'manage_options',
        'ghostgate-settings',
        'ghostgate_render_settings_page',
        'dashicons-shield-alt',
        80
    );
});


if ( ! function_exists('ghostgate_sanitize_allowed_routes') ) {
function ghostgate_sanitize_allowed_routes( $value ) {
    if ( empty( $value ) || ! is_array( $value ) ) return array();

    global $ghostgate_bypass_json_filter;
    $old_bypass = $ghostgate_bypass_json_filter ?? false;
    $ghostgate_bypass_json_filter = true;

    // rest_api_init がまだなら起動（UI保存時でも安全に）
    if ( ! did_action( 'rest_api_init' ) ) {
        do_action( 'rest_api_init' );
    }

    $server = function_exists('rest_get_server') ? rest_get_server() : null;
    $all_routes = $server ? array_keys( $server->get_routes() ) : array();

    // ← 必ず復帰
    $ghostgate_bypass_json_filter = $old_bypass;

    $value = array_map( 'strval', $value );
    $value = array_map( 'wp_check_invalid_utf8', $value );

    $whitelist = array_values( array_unique( array_intersect( $all_routes, $value ) ) );
    if ( ! in_array( '/', $whitelist, true ) ) {
      $whitelist[] = '/';
    }
    return $whitelist;
}
}


if ( ! function_exists('ghostgate_sanitize_allowed_prefixes') ) {
  function ghostgate_sanitize_allowed_prefixes( $value ) {
    $value = is_string( $value ) ? wp_unslash( $value ) : '';
    $parts = array_map( 'trim', explode( ',', $value ) );

    $norms = array();
    foreach ( $parts as $p ) {
      if ( $p === '' ) continue;

      // 許可する文字は「英数/_- とスラッシュ・%・.」くらいまで緩める（REST名前空間の実情に合わせる）
      // 余計な空白などは削除
      $p = preg_replace( '#[^A-Za-z0-9/_\.\-%]#', '', $p );
      if ( $p === '' ) continue;

      // 先頭スラッシュ付与、連続スラッシュを1本に
      if ( strpos( $p, '/' ) !== 0 ) $p = '/' . $p;
      $p = preg_replace( '#/{2,}#', '/', $p );

      $norms[] = $p;
    }

    // 重複除去して、保存/表示はカンマ区切り
    $norms = array_values( array_unique( $norms ) );
    return implode( ', ', $norms );
  }
}



function ghostgate_render_settings_page() {


    // ✅ admin-post からの戻りメッセージ
// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- read-only GET
$status = sanitize_key( (string) ( filter_input( INPUT_GET, 'ghostgate_status', FILTER_UNSAFE_RAW ) ?? '' ) );
if ( $status === 'unblocked' ) {
    $ips = sanitize_text_field( (string) ( filter_input( INPUT_GET, 'ip', FILTER_UNSAFE_RAW ) ?? '' ) );
add_settings_error(
    'ghostgate',
    'ghostgate_unblocked',
    sprintf(
        /* translators: %s is the IP address (or comma-separated list of IPs) that was unblocked. */
        esc_html__( 'Unblocked: %s', 'ghostgate' ),
        esc_html( is_array( $ips ) ? implode( ', ', $ips ) : (string) $ips )
    ),
    'updated'
);


} elseif ( $status === 'no_ip' ) {
    add_settings_error(
        'ghostgate',
        'ghostgate_unblocked_none',
        esc_html__( 'No IP selected.', 'ghostgate' ),
        'error'
    );
}

// 通知表示（既存のまま）
settings_errors( 'ghostgate' );



    ?>
    <div class="wrap">
        <h1><?php esc_html_e('GhostGate 設定画面', 'ghostgate'); ?></h1>
        <p><?php esc_html_e('WordPress ログインをかつてないほど安全に保護します。', 'ghostgate'); ?></p>

        <!-- タブ -->
        <div class="ghostgate-tabs">
            <div class="ghostgate-tab active" data-target="ghostgate-tab-settings">
                <?php esc_html_e('⚙️ 設定・ステータス', 'ghostgate'); ?>
            </div>
            <div class="ghostgate-tab" data-target="ghostgate-tab-about">
                <?php esc_html_e('🧾 機能と説明', 'ghostgate'); ?>
            </div>
        </div>

        <!-- タブ2: 設定・ステータス -->
        <div class="ghostgate-tab-content active" id="ghostgate-tab-settings">
            <div class="ghostgate-flex-wrap">

                <div class="ghostgate-settings-box">
                    <form method="post" action="options.php">
                        <?php
                        settings_fields('ghostgate_options');

                        ob_start();
                        do_settings_fields('ghostgate-settings', 'ghostgate_section_main');
                        $fields_html = ob_get_clean();

                        // Wrap each <tr> row from settings fields with a custom <div> container for layout purposes.
                        // The original content is generated by WordPress (do_settings_sections) and may include form elements.
                        // We sanitize the HTML via wp_kses with a controlled allowlist to preserve functionality and meet security guidelines.

                        $sanitized_html = preg_replace_callback('/(<tr.*?>.*?<\\/tr>)/s', function ($matches) {
                            return '<div class="ghostgate-setting-card">' . wp_kses($matches[1], ghostgate_get_allowed_html_tags()) . '</div>';
                        }, $fields_html);

                        // ✅ タグバランス
                        $final_output = force_balance_tags($sanitized_html);

                        // ✅ エスケープ済として出力
                        /* phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Output is sanitized with wp_kses allowlist */
                        echo $final_output;

                        submit_button();
                        ?>
                    </form>

<h2><?php esc_html_e('🚫 現在ブロック中のIP', 'ghostgate'); ?></h2>

<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
  <input type="hidden" name="action" value="ghostgate_unblock_ip">
  <?php wp_nonce_field( 'ghostgate_unblock_ip_action' ); ?>

  <?php
  // ← ここにあった redirect_to hidden は削除
  $blocked_ips = ghostgate_get_blocked_ips();
  if ( ! empty( $blocked_ips ) ) {
      echo '<ul class="ghostgate-ip-list">';

      echo '<li style="margin-bottom:8px;">';
      echo '<label><input type="checkbox" id="gg-check-all"> ' . esc_html__( 'すべて選択', 'ghostgate' ) . '</label>';
      echo '</li>';

      foreach ( $blocked_ips as $ip => $types ) {
          $labels = [];
          if ( ! empty( $types['login'] ) ) $labels[] = __( 'ログイン', 'ghostgate' );
          if ( ! empty( $types['ajax'] ) )  $labels[] = __( 'AJAX', 'ghostgate' );
          if ( ! empty( $types['rest'] ) )  $labels[] = __( 'REST API', 'ghostgate' );

          echo '<li>';
          echo '<label style="margin-right:8px;"><input class="gg-ip-item" type="checkbox" name="ip[]" value="' . esc_attr( $ip ) . '"> ' . esc_html( $ip ) . '</label>';
          echo '<span class="ghostgate-block-type">' . esc_html( implode( ', ', $labels ) ) . '</span> ';
          echo '<button type="submit" name="ip" value="' . esc_attr( $ip ) . '" class="button" style="margin-left:8px;">' .
               esc_html__( 'このIPだけ解除', 'ghostgate' ) .
               '</button>';
          echo '</li>';
      }

      echo '</ul>';
      echo '<p><button type="submit" class="button button-primary">' . esc_html__( '選択したIPを解除', 'ghostgate' ) . '</button></p>';

      echo '<script>(function(){var a=document.getElementById("gg-check-all");if(!a)return;var b=[].slice.call(document.querySelectorAll(".gg-ip-item"));a.addEventListener("change",function(){b.forEach(function(i){i.checked=a.checked;});});})();</script>';
  } else {
      echo '<p>' . esc_html__( '現在ブロックされているIPはありません。', 'ghostgate' ) . '</p>';
  }
  ?>
</form>


                </div>

                <div class="ghostgate-status-box">
                    <h2><?php esc_html_e('🛡 現在のセキュリティ機能一覧', 'ghostgate'); ?></h2>
                    <table class="widefat striped">
                        <tbody>
                            <?php ghostgate_render_security_status_rows(); ?>
                        </tbody>
                    </table>
                </div>

            </div>
        </div>


        <!-- タブ1: 機能と説明 -->
<div class="ghostgate-tab-content" id="ghostgate-tab-about">
    <div class="ghostgate-about-intro">
        <p>
        <?php echo esc_html__('GhostGate（ゴーストゲート）は、WordPressの最も脆弱な部分「ログイン画面」そのものを不可視化し、攻撃対象を消すことで、従来の防御とは異なる', 'ghostgate'); ?>
        <strong><?php echo esc_html__('"入り口を消す"', 'ghostgate'); ?></strong>
        <?php echo esc_html__('という新しい発想から生まれたセキュリティプラグインです。', 'ghostgate'); ?>
        </p>

        <p>
        <?php echo esc_html__('軽量・高速でありながら、URL秘匿・2FA・IP制限・通信遮断・列挙防止・診断分析などを包括的に備え、セキュリティ強化と運用のしやすさを両立。', 'ghostgate'); ?>
        </p>

        <p>
        <?php echo esc_html__('名前の通り、「Ghost」のように気配を消し、「Gate」のように正規ユーザーだけを通す、透明で強固な守りを実現します。', 'ghostgate'); ?>
        </p>

    </div>


    <hr>

    <div class="ghostgate-feature-toc" style="margin-bottom: 20px;">
        <strong><?php esc_html_e('🔍 機能一覧:', 'ghostgate'); ?></strong>
        <a href="#feature-login-url"><?php esc_html_e('🔐 ログインURL秘匿', 'ghostgate'); ?></a> |
        <a href="#feature-2fa"><?php esc_html_e('📧 2FA認証', 'ghostgate'); ?></a> |
        <a href="#feature-ip-limit"><?php esc_html_e('🛑 IP制限', 'ghostgate'); ?></a> |
        <a href="#feature-xmlrpc"><?php esc_html_e('📁 通信遮断', 'ghostgate'); ?></a> |
        <a href="#feature-session-control"><?php esc_html_e('⏱️ ログインセッション管理', 'ghostgate'); ?></a> |
        <a href="#feature-logging"><?php esc_html_e('📜 ログ記録', 'ghostgate'); ?></a> |
        <a href="#feature-author"><?php esc_html_e('👤 author列挙防止', 'ghostgate'); ?></a> |
        <a href="#feature-wp-json-control"><?php esc_html_e('📡 wp-json制御', 'ghostgate'); ?></a> |
        <a href="#feature-analysis"><?php esc_html_e('📊 ステータス分析', 'ghostgate'); ?></a> |
        <a href="#feature-exist"><?php esc_html_e('🌐 GhostGateの存在意義', 'ghostgate'); ?></a>
    </div>

        <div class="ghostgate-feature-grid">

        <div class="ghostgate-feature-item" id="feature-login-url" style="border-left-color: #0073aa;">
            <div class="ghostgate-feature-title">
                <?php esc_html_e('🔐 ログインURL秘匿＋コード入力', 'ghostgate'); ?>
            </div>
            <div class="ghostgate-feature-desc">
                <p class="lead"><?php echo esc_html__('ログインページそのものを隠し、侵入の入口を完全に遮断します。', 'ghostgate'); ?></p>

                <p><?php echo wp_kses_post(__('WordPressのログインURLは非常に単純で、たとえば', 'ghostgate')); ?>
                <code>https://yourdomain.com/wp-login.php</code>
                <?php echo wp_kses_post(__('のように、誰でも容易にアクセス可能です。', 'ghostgate')); ?><br>
                <?php echo wp_kses_post(__('つまり、ドメイン名さえ分かれば攻撃対象になってしまいます。', 'ghostgate')); ?></p>

                <p><?php echo wp_kses_post(__('しかし、ログインURLが毎回動的に変わり、特定の条件を知らなければ辿り着けないとしたらどうでしょうか？', 'ghostgate')); ?></p>

                <p><?php echo wp_kses_post(__('GhostGateは、', 'ghostgate')); ?>
                <strong><?php echo esc_html__('ログインURLそのものを秘匿し、アクセスには任意のスラッグと時間コードを必要とするゲート機構', 'ghostgate'); ?></strong>
                <?php echo wp_kses_post(__('を実装します。', 'ghostgate')); ?></p>

                <p><strong><?php echo esc_html__('そもそも攻撃が成立しない', 'ghostgate'); ?></strong>
                <?php echo wp_kses_post(__('という革新的なセキュリティ構造を実現します。', 'ghostgate')); ?></p>

            </div>
        </div>


        <div class="ghostgate-feature-item" id="feature-2fa" style="border-left-color: #0073aa;">
            <div class="ghostgate-feature-title">
                <?php esc_html_e('📧 メール2段階認証（2FA）', 'ghostgate'); ?>
            </div>
            <div class="ghostgate-feature-desc">
                <p class="lead">
                <?php echo esc_html__('パスワードが漏れても、本人確認で不正ログインを防ぎます。', 'ghostgate'); ?>
                </p>

                <p>
                <?php echo esc_html__('仮にログイン画面を突破されても、それだけで管理画面に入らせてしまっては意味がありません。', 'ghostgate'); ?><br>
                <?php echo esc_html__('GhostGateでは、', 'ghostgate'); ?>
                <strong><?php echo esc_html__('ログイン後すぐに本人確認用のメールを送信', 'ghostgate'); ?></strong>
                <?php echo esc_html__('し、その中に記載された認証コードを正しく入力しない限り、本ログインを完了させません。', 'ghostgate'); ?>
                </p>

                <p>
                <?php echo esc_html__('この「もう一段階の確認」により、', 'ghostgate'); ?>
                <strong><?php echo esc_html__('パスワードが漏れてもアカウントを守れる仕組み', 'ghostgate'); ?></strong>
                <?php echo esc_html__('になっています。', 'ghostgate'); ?><br>
                <?php echo esc_html__('コードは毎回ランダムに発行され、一定時間を過ぎると無効になります。', 'ghostgate'); ?>
                </p>

                <p>
                <?php echo esc_html__('また、ユーザーごとにこの2段階認証のON/OFFを切り替えることができ、必要なアカウントのみに強化保護を施すことができます。', 'ghostgate'); ?>
                </p>

            </div>
        </div>


        <div class="ghostgate-feature-item" id="feature-ip-limit" style="border-left-color: #cc0000;">
            <div class="ghostgate-feature-title">
                <?php esc_html_e('🛑 IP制限（ログイン試行制限）', 'ghostgate'); ?>
            </div>
            <div class="ghostgate-feature-desc">
                <p class="lead">
                <?php echo esc_html__('総当たり攻撃をIP単位で即時ブロック。', 'ghostgate'); ?>
                </p>

                <p>
                <?php echo esc_html__('不正ログインの多くは、Botが無数のパスワードを機械的に試す「総当たり攻撃（ブルートフォース）」によって行われます。', 'ghostgate'); ?><br>
                <?php echo esc_html__('GhostGateは、このような攻撃を防ぐために、', 'ghostgate'); ?>
                <strong><?php echo esc_html__('短時間に複数回ログインに失敗したIPアドレスを自動的にブロック', 'ghostgate'); ?></strong>
                <?php echo esc_html__('します。', 'ghostgate'); ?>
                </p>

                <p>
                <?php echo esc_html__('例えば、「30分間に3回まで」など、', 'ghostgate'); ?>
                <strong><?php echo esc_html__('細かい制限ルールを設定可能', 'ghostgate'); ?></strong>
                <?php echo esc_html__('で、人間の操作には支障なく、Botや攻撃スクリプトだけを排除できます。', 'ghostgate'); ?><br>
                <?php echo esc_html__('ブロックされたIPは管理画面から確認・解除できるため、安全性と管理のしやすさを両立しています。', 'ghostgate'); ?>
                </p>

                <p>
                <?php echo esc_html__('この制限により、', 'ghostgate'); ?>
                <strong><?php echo esc_html__('突破される前にアクセスそのものを遮断する', 'ghostgate'); ?></strong>
                <?php echo esc_html__('という、実用的かつ堅実な防御が可能になります。', 'ghostgate'); ?>
                </p>

            </div>
        </div>


        <div class="ghostgate-feature-item" id="feature-xmlrpc" style="border-left-color: #cc0000;">
            <div class="ghostgate-feature-title">
                <?php esc_html_e('📁 XML-RPC / admin-ajax 制御', 'ghostgate'); ?>
            </div>
            <div class="ghostgate-feature-desc">
                <p class="lead">
                <?php echo esc_html__('使っていない通信機能は徹底的に遮断して守りを固めます。', 'ghostgate'); ?>
                </p>

                <p>
                <?php echo esc_html__('WordPressには、外部アプリやテーマ・プラグインとの通信を行うための機能として、「XML-RPC」や「admin-ajax.php」などのエンドポイントが用意されています。', 'ghostgate'); ?><br>
                <?php echo esc_html__('しかし、これらは現在では一部の特殊な用途を除き、', 'ghostgate'); ?>
                <strong><?php echo esc_html__('ほとんどのサイトで使われていないにもかかわらず有効なまま', 'ghostgate'); ?></strong>
                <?php echo esc_html__('になっているケースが多く、', 'ghostgate'); ?><br>
                <?php echo esc_html__('実際には、', 'ghostgate'); ?>
                <strong><?php echo esc_html__('攻撃者にとって“狙いやすい裏口”として悪用されるリスクが高い', 'ghostgate'); ?></strong>
                <?php echo esc_html__('のです。', 'ghostgate'); ?>
                </p>

                <p>
                <?php echo esc_html__('特に「XML-RPC」は、WordPressの初期にリモート投稿やアプリ連携に使われていた仕組みですが、現在ではほとんどのサイトにとって不要な存在となっています。', 'ghostgate'); ?>
                </p>

                <p>
                <?php echo esc_html__('GhostGateは、こうした', 'ghostgate'); ?>
                <strong><?php echo esc_html__('不要な通信機能を確実に無効化', 'ghostgate'); ?></strong>
                <?php echo esc_html__('することで、Botによるスパム投稿・不正ログイン・ユーザー列挙といった攻撃の経路を断ちます。', 'ghostgate'); ?><br>
                <code>.htaccess</code>
                <?php echo esc_html__('による物理的な遮断にも対応しており、', 'ghostgate'); ?>
                <strong><?php echo esc_html__('WordPressの設定だけでなく、サーバーレベルでもセキュリティを強化', 'ghostgate'); ?></strong>
                <?php echo esc_html__('できます。', 'ghostgate'); ?>
                </p>

                <p>
                <?php echo esc_html__('利用しない機能を閉じておくことは、', 'ghostgate'); ?>
                <strong><?php echo esc_html__('「攻撃される前に入り口を消す」GhostGateの中核思想', 'ghostgate'); ?></strong>
                <?php echo esc_html__('そのものです。', 'ghostgate'); ?>
                </p>

            </div>
        </div>

        <div class="ghostgate-feature-item" id="feature-session-control" style="border-left-color: #0073aa;">
            <div class="ghostgate-feature-title">
                <?php esc_html_e('🔒 ログインセッション管理', 'ghostgate'); ?>
            </div>
            <div class="ghostgate-feature-desc">
                <p class="lead">
                    <?php echo esc_html__('セッションの寿命や並行ログインを制御し、乗っ取りリスクの土台を断ちます。', 'ghostgate'); ?>
                </p>

                <p>
                    <?php echo esc_html__('一定時間の操作がない場合に自動ログアウトする「アイドルタイムアウト」と、最大継続時間を超えたら必ず再ログインを求める「絶対タイムアウト」を設定できます。', 'ghostgate'); ?>
                    <?php echo esc_html__('長時間放置や共有PCでのログイン放置によるリスクを抑制します。', 'ghostgate'); ?>
                </p>

                <p>
                    <?php echo esc_html__('さらに、', 'ghostgate'); ?>
                    <strong><?php echo esc_html__('同一ユーザーの同時ログイン数を制限', 'ghostgate'); ?></strong>
                    <?php echo esc_html__('することで、第三者による使い回しを防止。新しいログインが発生した場合は既存セッションを無効化できます。', 'ghostgate'); ?>
                    <?php echo esc_html__('管理画面から現在のセッションを一覧し、対象ユーザーを強制ログアウトすることも可能です。', 'ghostgate'); ?>
                </p>

                <p>
                    <?php echo esc_html__('オプションで、', 'ghostgate'); ?>
                    <strong><?php echo esc_html__('IPアドレスやユーザーエージェントの継続チェック', 'ghostgate'); ?></strong>
                    <?php echo esc_html__('を有効にでき、環境が大きく変わった場合は再認証を要求。Remember Me 選択時でも過度な長期化を抑える細かなルール設定に対応します。', 'ghostgate'); ?>
                </p>

                <p>
                    <?php echo esc_html__('ログイン直後や権限昇格時にはセッションIDを自動再生成（セッション固定化対策）し、2FAと組み合わせることで、', 'ghostgate'); ?>
                    <strong><?php echo esc_html__('「侵入させない」「侵入しても居座らせない」', 'ghostgate'); ?></strong>
                    <?php echo esc_html__('二段構えの防御を実現します。', 'ghostgate'); ?>
                </p>
            </div>
        </div>


        <div class="ghostgate-feature-item" id="feature-logging" style="border-left-color: #228b22;">
            <div class="ghostgate-feature-title">
                <?php esc_html_e('📜 ログ記録＋ローテート', 'ghostgate'); ?>
            </div>
            <div class="ghostgate-feature-desc">
                <p class="lead">
                <?php echo esc_html__('すべての操作を記録・保存し、証拠と備えを残します。', 'ghostgate'); ?>
                </p>

                <p>
                <?php echo esc_html__('セキュリティ対策でもう一つ重要なのが、「何が起きたのかを正確に記録しておくこと」です。', 'ghostgate'); ?><br>
                <?php echo esc_html__('GhostGateは、ログインの成功・失敗、2段階認証の状態、IPブロックの発動など、', 'ghostgate'); ?>
                <strong><?php echo esc_html__('あらゆる重要な操作をログとして記録', 'ghostgate'); ?></strong>
                <?php echo esc_html__('します。', 'ghostgate'); ?>
                </p>

                <p>
                <?php echo esc_html__('ログは、指定したファイルパスに保存され、', 'ghostgate'); ?>
                <strong><?php echo esc_html__('保存日数を設定することで、一定期間経過後に自動削除（ローテート）される', 'ghostgate'); ?></strong>
                <?php echo esc_html__('ため、サーバーの容量も圧迫しません。', 'ghostgate'); ?><br>
                <?php echo esc_html__('また、ログ出力はチェックボックスで簡単にON/OFFでき、保存先パスや保存期間もUIから自由に設定できます。', 'ghostgate'); ?>
                </p>

                <p>
                <?php echo esc_html__('こうした記録は、', 'ghostgate'); ?>
                <strong><?php echo esc_html__('「いつ・誰が・何をしたか」を後から確認できる唯一の証拠', 'ghostgate'); ?></strong>
                <?php echo esc_html__('となるため、万が一のトラブル時にも迅速な対応が可能です。', 'ghostgate'); ?><br>
                <strong><?php echo esc_html__('GhostGateは、“守る”だけでなく“記録して備える”という観点でも、実用的なセキュリティを提供します。', 'ghostgate'); ?></strong>
                </p>

            </div>
        </div>


        <div class="ghostgate-feature-item" id="feature-author" style="border-left-color: #cc0000;">
            <div class="ghostgate-feature-title">
                <?php esc_html_e('👤 author列挙防止', 'ghostgate'); ?>
            </div>
            <div class="ghostgate-feature-desc">
                <p class="lead">
                <?php echo esc_html__('ユーザー名の漏洩を防ぎ、攻撃の足掛かりを与えません。', 'ghostgate'); ?>
                </p>

                <p>
                <?php echo esc_html__('WordPressでは、特定のURL（例：', 'ghostgate'); ?>
                <code>?author=1</code>
                <?php echo esc_html__('）やREST APIを利用することで、', 'ghostgate'); ?>
                <strong><?php echo esc_html__('サイトに登録されているユーザー名を外部から簡単に取得できてしまう', 'ghostgate'); ?></strong>
                <?php echo esc_html__('という問題があります。', 'ghostgate'); ?><br>

                <?php echo esc_html__('攻撃者はこの情報をもとにパスワードを割り出し、不正ログインを試みるため、', 'ghostgate'); ?>
                <strong><?php echo esc_html__('ユーザー名の漏洩は重大なセキュリティリスク', 'ghostgate'); ?></strong>
                <?php echo esc_html__('となります。', 'ghostgate'); ?>
                </p>

                <p>
                <?php echo esc_html__('GhostGateは、こうした', 'ghostgate'); ?>
                <strong><?php echo esc_html__('author列挙のあらゆる手口をブロック', 'ghostgate'); ?></strong>
                <?php echo esc_html__('します。REST API、RSS、クエリパラメータなど、', 'ghostgate'); ?>
                <strong><?php echo esc_html__('複数の経路を横断的に検出・遮断', 'ghostgate'); ?></strong>
                <?php echo esc_html__('し、どこからもユーザー名が漏れないよう徹底的に防御します。', 'ghostgate'); ?>
                </p>

                <p>
                <?php echo esc_html__('また、他のプラグインやテーマによって無効化が機能していない場合も検知できるため、', 'ghostgate'); ?>
                <strong><?php echo esc_html__('現在の防御状態を明確に可視化', 'ghostgate'); ?></strong>
                <?php echo esc_html__('することができます。', 'ghostgate'); ?><br>

                <strong><?php echo esc_html__('GhostGateは、「知られなければ、狙われることもない」', 'ghostgate'); ?></strong>
                <?php echo esc_html__('という根本的な防御思想を、この機能でも体現しています。', 'ghostgate'); ?>
                </p>

            </div>
        </div>

        <div class="ghostgate-feature-item" id="feature-wp-json-control" style="border-left-color: #5e5ce6;">
            <div class="ghostgate-feature-title">
                <?php esc_html_e('🛡️ /wp-json の表示制御（REST APIの露出最小化）', 'ghostgate'); ?>
            </div>
            <div class="ghostgate-feature-desc">
                <p class="lead">
                    <?php echo esc_html__('RESTのルート一覧の露出を最小限にし、偵察・探索型攻撃の足がかりを断ちます。', 'ghostgate'); ?>
                </p>

                <p>
                    <?php echo esc_html__('/wp-json には、サイトで利用可能な名前空間やルートの一覧が表示されます。攻撃者はこれを手掛かりに指紋採取や脆弱性探索を行います。', 'ghostgate'); ?><br>
                    <?php echo esc_html__('GhostGate は、この一覧表示を', 'ghostgate'); ?>
                    <strong><?php echo esc_html__('ホワイトリスト方式で制御', 'ghostgate'); ?></strong>
                    <?php echo esc_html__('し、不要な情報の露出を抑制します。', 'ghostgate'); ?>
                    <strong><?php echo esc_html__('実際のAPI自体の動作は維持', 'ghostgate'); ?></strong>
                    <?php echo esc_html__('されるため、正規の連携や機能には影響しません。', 'ghostgate'); ?>
                </p>

                <p>
                    <?php echo esc_html__('管理画面では、検出済みのエンドポイント一覧から表示許可を選択でき、さらに', 'ghostgate'); ?>
                    <strong><?php echo esc_html__('自作エンドポイントの接頭辞（例: sbp2, ghostgate など）をプレフィックス指定', 'ghostgate'); ?></strong>
                    <?php echo esc_html__('して許可することも可能です。', 'ghostgate'); ?><br>
                    <?php echo esc_html__('必要なルートだけを明示的に見せることで、互換性と秘匿性のバランスを実現します。', 'ghostgate'); ?>
                </p>

                <p>
                    <?php echo esc_html__('この制御により、Bot による自動スキャンや不要なルート列挙のノイズを低減し、', 'ghostgate'); ?>
                    <strong><?php echo esc_html__('攻撃面（アタックサーフェス）の縮小', 'ghostgate'); ?></strong>
                    <?php echo esc_html__('に貢献します。', 'ghostgate'); ?>
                    <?php echo esc_html__('Discovery に依存するプラグインをお使いの場合は、必要なルートのみ許可してください。', 'ghostgate'); ?>
                </p>

                <p>
                    <?php echo esc_html__('さらに、オプションで', 'ghostgate'); ?>
                    <strong><?php echo esc_html__('特定の名前空間項目を一覧から除外', 'ghostgate'); ?></strong>
                    <?php echo esc_html__('するなど、表示粒度を細かく調整できます。', 'ghostgate'); ?>
                    <?php echo esc_html__('ログや一時的な無効化にも対応し、運用時の確認・切り戻しが容易です。', 'ghostgate'); ?>
                </p>
            </div>
        </div>


        <div class="ghostgate-feature-item" id="feature-analysis" style="border-left-color: #9932cc;">
            <div class="ghostgate-feature-title">
                <?php esc_html_e('📊 セキュリティステータス分析', 'ghostgate'); ?>
            </div>
            <div class="ghostgate-feature-desc">
                <p class="lead">
                <?php echo esc_html__('現在の防御状態を視覚化し、設定ミスや競合も即検出。', 'ghostgate'); ?>
                </p>

                <p>
                <?php echo esc_html__('WordPressのセキュリティ対策は、複数のプラグインや設定が絡み合うことで、', 'ghostgate'); ?>
                <strong><?php echo esc_html__('「効いていると思っていたけど実は無効」', 'ghostgate'); ?></strong>
                <?php echo esc_html__('や、', 'ghostgate'); ?>
                <strong><?php echo esc_html__('「機能が競合して逆効果」', 'ghostgate'); ?></strong>
                <?php echo esc_html__('といった状況に陥ることがあります。', 'ghostgate'); ?>
                </p>

                <p>
                <?php echo esc_html__('GhostGateでは、', 'ghostgate'); ?>
                <strong><?php echo esc_html__('自分自身の設定に加え、他のセキュリティプラグインや .htaccess などの干渉状況を自動的に診断', 'ghostgate'); ?></strong>
                <?php echo esc_html__('し、現在の防御状態を', 'ghostgate'); ?>
                <strong><?php echo esc_html__('視覚的に一覧表示', 'ghostgate'); ?></strong>
                <?php echo esc_html__('します。', 'ghostgate'); ?>
                </p>

                <p>
                <?php echo esc_html__('有効・無効・部分的な保護・競合の兆候などを、', 'ghostgate'); ?>
                <strong><?php echo esc_html__('色分けバッジと警告付きで明確に表示', 'ghostgate'); ?></strong>
                <?php echo esc_html__('するため、管理者は迷わず判断ができます。', 'ghostgate'); ?>
                </p>

                <p>
                <?php echo esc_html__('この機能により、「何が有効で、何が衝突しているか」が一目でわかるため、', 'ghostgate'); ?>
                <strong><?php echo esc_html__('“設定のつもり”を“確実な防御”に変える', 'ghostgate'); ?></strong>
                <?php echo esc_html__('ことができます。', 'ghostgate'); ?><br>

                <?php echo esc_html__('GhostGateは、セキュリティを', 'ghostgate'); ?>
                <strong><?php echo esc_html__('“見えない壁”ではなく、“見て理解できる防御”', 'ghostgate'); ?></strong>
                <?php echo esc_html__('へと進化させます。', 'ghostgate'); ?>
                </p>

            </div>
        </div>


        <div class="ghostgate-feature-item" id="feature-exist" style="border-left-color: #333;">
            <div class="ghostgate-feature-title">
                <?php esc_html_e('🌐 GhostGateの存在意義', 'ghostgate'); ?>
            </div>
            <div class="ghostgate-feature-desc">
                <p class="lead">
                <?php echo esc_html__('ただ守るだけでなく、透明な守りへと進化する。', 'ghostgate'); ?>
                </p>

                <p>
                <?php echo esc_html__('多くのセキュリティツールが「既に起きた攻撃への対処」を重視する中で、GhostGateは根本から異なる哲学を持っています。', 'ghostgate'); ?><br>
                <?php echo esc_html__('攻撃の発生を前提にするのではなく、', 'ghostgate'); ?>
                <strong><?php echo esc_html__('「攻撃されない状態」そのものを作り出す', 'ghostgate'); ?></strong>
                <?php echo esc_html__('──それがGhostGateの本質です。', 'ghostgate'); ?>
                </p>

                <p>
                <?php echo esc_html__('ログインURLを不可視化し、機能を絞り込み、通信を遮断し、ユーザー名すら漏らさない。これらの機能は単体でも強力ですが、GhostGateは', 'ghostgate'); ?>
                <strong><?php echo esc_html__('それらを軽量かつ相互干渉せずに共存させる設計', 'ghostgate'); ?></strong>
                <?php echo esc_html__('にこだわっています。', 'ghostgate'); ?>
                </p>

                <p>
                <?php echo esc_html__('さらに、設定の衝突や無効化を「見える化」するステータス分析機能により、管理者は現在のセキュリティ状態を確実に把握できます。', 'ghostgate'); ?><br>
                <?php echo esc_html__('これは、セキュリティをブラックボックス化せず、', 'ghostgate'); ?>
                <strong><?php echo esc_html__('「透明な守り」へと進化させる思想', 'ghostgate'); ?></strong>
                <?php echo esc_html__('そのものです。', 'ghostgate'); ?>
                </p>

                <p>
                <?php echo esc_html__('GhostGateは、ただのセキュリティ対策ではありません。', 'ghostgate'); ?>
                <strong><?php echo esc_html__('WordPressのあらゆる環境と共存しながら、“見えない壁”となってサイトを守る、新しい形のゲート', 'ghostgate'); ?></strong>
                <?php echo esc_html__('です。', 'ghostgate'); ?>
                </p>

            </div>
        </div>


        </div>


        </div>


    </div>
    <?php
}


// ✅ ここに追加
add_action('admin_init', 'ghostgate_register_settings');


function ghostgate_register_settings() {

    $settings = [
        'ghostgate_login_slug'              => 'sanitize_title',
        'ghostgate_code_length'             => 'absint',
        'ghostgate_enable_code_length'      => 'absint',
        'ghostgate_code_mode'               => 'sanitize_text_field',
        'ghostgate_enable_code_mode'        => 'absint',
        'ghostgate_custom_code'             => 'sanitize_text_field',
        'ghostgate_enable_2fa'              => 'absint',
        'ghostgate_ajax_limit'              => 'absint',
        'ghostgate_enable_ajax_limit'       => 'absint',
        'ghostgate_ajax_cooldown'           => 'absint',
        'ghostgate_ajax_notify'             => 'absint',
        'ghostgate_log_file_path'           => 'sanitize_text_field',
        'ghostgate_enable_log_file'         => 'absint',
        'ghostgate_log_retention_days'      => 'absint',
        'ghostgate_notify_on_block'         => 'absint',
        'ghostgate_block_author_enum'       => 'absint',
        'ghostgate_disable_xmlrpc'          => 'absint',
        'ghostgate_enable_login_limit'      => 'absint',
        'ghostgate_login_attempt_limit'     => 'absint',
        'ghostgate_enable_login_slug'       => 'absint',
        'ghostgate_block_unused_rest'       => 'absint',
        'ghostgate_enable_rest_limit'     => 'absint',
        'ghostgate_rest_limit_count'      => 'absint',
        'ghostgate_rest_limit_cooldown'   => 'absint',
        //'ghostgate_hide_json_endpoints' => 'absint',
        //'ghostgate_json_allowed_routes'   => 'ghostgate_sanitize_array',
        //'ghostgate_json_allowed_prefixes' => 'sanitize_text_field',
        'ghostgate_enable_session_control' => 'absint',
        'ghostgate_session_timeout'        => 'absint',
        'ghostgate_session_interval' => 'absint',
        'ghostgate_login_block_cooldown' => 'absint',
        'ghostgate_block_preview' => 'absint',
    ];

    // 一括登録（セキュリティ審査対応）
    foreach ($settings as $option_name => $sanitize_callback) {
        register_setting('ghostgate_options', $option_name, [
            'sanitize_callback' => $sanitize_callback
        ]);
    }

    // 1) 許可ルート（配列）— 正規表現ルートを壊さず、存在ルートとの積集合のみ通す
    register_setting(
        'ghostgate_options',                   // ← 既存の settings group に合わせる
        'ghostgate_json_allowed_routes',
        array(
            'type'              => 'array',
            'sanitize_callback' => 'ghostgate_sanitize_allowed_routes',
            'default'           => array(),
            'show_in_rest'      => false,
        )
    );

    // 2) 接頭辞（文字列だが正規化は独自）— "/gbrl, /line/v1" のように整える
    register_setting(
        'ghostgate_options',
        'ghostgate_json_allowed_prefixes',
        array(
            'type'              => 'string',
            'sanitize_callback' => 'ghostgate_sanitize_allowed_prefixes',
            'default'           => '',
            'show_in_rest'      => false,
        )
    );

    // 3) トグル（ON/OFF）
    register_setting(
        'ghostgate_options',
        'ghostgate_hide_json_endpoints',
        array(
            'type'              => 'string',
            'sanitize_callback' => static function( $v ) { return ( $v === '1' ) ? '1' : '0'; },
            'default'           => '0',
            'show_in_rest'      => false,
        )
    );

    // セクション（1つでOK）
    add_settings_section('ghostgate_section_main', __('基本設定', 'ghostgate'), '__return_null', 'ghostgate-settings');

    // 設定フィールドの登録
    add_settings_field('ghostgate_login_slug', __('GhostGateのURLスラッグ', 'ghostgate'), 'ghostgate_field_login_slug', 'ghostgate-settings', 'ghostgate_section_main');
    add_settings_field('ghostgate_code_length', __('認証コードの桁数', 'ghostgate'), 'ghostgate_field_code_length', 'ghostgate-settings', 'ghostgate_section_main');
    add_settings_field('ghostgate_code_mode', __('認証コード形式', 'ghostgate'), 'ghostgate_field_code_mode', 'ghostgate-settings', 'ghostgate_section_main');
    add_settings_field('ghostgate_enable_2fa', __('2FAを有効にする', 'ghostgate'), 'ghostgate_field_enable_2fa', 'ghostgate-settings', 'ghostgate_section_main');
    add_settings_field('ghostgate_ajax_limit', __('admin-ajax制限回数（1分）', 'ghostgate'), 'ghostgate_field_ajax_limit', 'ghostgate-settings', 'ghostgate_section_main');
    add_settings_field('ghostgate_block_author_enum', __('ユーザー列挙のブロック', 'ghostgate'), 'ghostgate_field_block_author_enum', 'ghostgate-settings', 'ghostgate_section_main');
    add_settings_field('ghostgate_block_unused_rest', __('未使用REST APIの遮断', 'ghostgate'), 'ghostgate_field_block_unused_rest', 'ghostgate-settings', 'ghostgate_section_main');
    add_settings_field('ghostgate_rest_limit_group', __('REST APIアクセス制限', 'ghostgate'), 'ghostgate_field_rest_limit_group', 'ghostgate-settings', 'ghostgate_section_main');
    add_settings_field('ghostgate_disable_xmlrpc', __('XML-RPC 通信の無効化', 'ghostgate'), 'ghostgate_field_disable_xmlrpc', 'ghostgate-settings', 'ghostgate_section_main');
    add_settings_field('ghostgate_hide_json_endpoints', __('wp-json構造の秘匿化', 'ghostgate'), 'ghostgate_field_hide_json_endpoints', 'ghostgate-settings', 'ghostgate_section_main');
    add_settings_field('ghostgate_login_limit_settings', __('ログイン試行制限の設定', 'ghostgate'), 'ghostgate_field_login_limit_settings', 'ghostgate-settings', 'ghostgate_section_main');
    add_settings_field('ghostgate_enable_session_control', __('ログインセッション管理', 'ghostgate'), 'ghostgate_field_session_control', 'ghostgate-settings', 'ghostgate_section_main');
    add_settings_field('ghostgate_log_file_path', __('ログファイルパス', 'ghostgate'), 'ghostgate_field_log_file_path', 'ghostgate-settings', 'ghostgate_section_main');
    add_settings_field('ghostgate_block_preview', __('プレビュー表示のブロック', 'ghostgate'), 'ghostgate_field_block_preview', 'ghostgate-settings', 'ghostgate_section_main');
}




// ログインURLスラッグ
function ghostgate_field_login_slug() {
    $slug_value = esc_attr(get_option('ghostgate_login_slug', 'my-login'));
    $enabled = get_option('ghostgate_enable_login_slug', 0);


    // ✅ スイッチUI
    echo '<br>';
    echo '<label class="ghostgate-switch">';
    echo '<input type="checkbox" id="ghostgate_enable_login_slug" name="ghostgate_enable_login_slug" value="1"' . checked($enabled, 1, false) . '>';
    echo '<span class="ghostgate-slider"></span>';
    echo '</label>';

    // ✅ ラベル（スイッチの横）
    echo '<span>' . esc_html__('使用する', 'ghostgate') . '</span><br>';

    // ✅ スラッグ入力欄
    echo '<input type="text" name="ghostgate_login_slug" value="' . esc_attr($slug_value) . '" class="regular-text">';
    echo '<p class="description">' . esc_html__('例: "my-login" にすると /my-login でログインします。', 'ghostgate') . '</p>';
}


// 認証コードの桁数
function ghostgate_field_code_length() {
    $value = intval(get_option('ghostgate_code_length', 4));
    $enabled = get_option('ghostgate_enable_code_length', 1);

    // ✅ スイッチUI＋ラベル（共通CSS使用前提）
    echo '<br>';
    echo '<label class="ghostgate-switch">';
    echo '<input type="checkbox" id="ghostgate_enable_code_length" name="ghostgate_enable_code_length" value="1" ' . checked($enabled, 1, false) . '>';
    echo '<span class="ghostgate-slider"></span>';
    echo '</label>';

    echo '<span>' . esc_html__('使用する', 'ghostgate') . '</span><br>';

    // ✅ 数値入力欄
    echo '<input type="number" name="ghostgate_code_length" value="' . esc_attr($value) . '" min="3" max="8">';
    echo '<p class="description">' . esc_html__('日付・時間コード以外に切り替える場合の桁数（数字のみ）', 'ghostgate') . '</p>';
}


// 認証コード形式（モード）
function ghostgate_field_code_mode() {
    $mode    = esc_attr(get_option('ghostgate_code_mode', 'date'));
    $enabled = get_option('ghostgate_enable_code_mode', 1);
    $custom  = esc_attr(get_option('ghostgate_custom_code', ''));

    echo '<br>';
    echo '<label class="ghostgate-switch">';
    echo '<input type="checkbox" id="ghostgate_enable_code_mode" name="ghostgate_enable_code_mode" value="1" ' . checked($enabled, 1, false) . '>';
    echo '<span class="ghostgate-slider"></span>';
    echo '</label>';
    echo '<span>' . esc_html__('使用する', 'ghostgate') . '</span><br>';

    echo '<select name="ghostgate_code_mode" id="ghostgate_code_mode">';
    echo '<option value="date"' . selected($mode, 'date', false) . '>' . esc_html__('日付コード（例: 0722）', 'ghostgate') . '</option>';
    echo '<option value="reverse_date"' . selected($mode, 'reverse_date', false) . '>' . esc_html__('逆日付コード（例: 2270）', 'ghostgate') . '</option>';
    echo '<option value="time"' . selected($mode, 'time', false) . '>' . esc_html__('時間コード（例: 1307）', 'ghostgate') . '</option>';
    echo '<option value="reverse_time"' . selected($mode, 'reverse_time', false) . '>' . esc_html__('逆時間コード（例: 7310）', 'ghostgate') . '</option>';
    echo '<option value="custom"' . selected($mode, 'custom', false) . '>' . esc_html__('カスタムコード（任意指定）', 'ghostgate') . '</option>';
    echo '</select>';

$wrap_class = ($mode === 'custom') ? 'visible' : '';

echo '<div id="ghostgate_custom_code_wrap" class="' . esc_attr($wrap_class) . '">';
echo '<input type="text" name="ghostgate_custom_code" value="' . esc_attr($custom) . '" pattern="^[0-9]{3,8}$" maxlength="8">';
echo '<p class="description">' . esc_html__('3〜8桁の任意コードを半角数字で入力してください。', 'ghostgate') . '</p>';
echo '</div>';


    echo '<p class="description">' . esc_html__('ログイン時に要求する認証コードの方式を選択してください。', 'ghostgate') . '</p>';

}


// 2FAの有効化
function ghostgate_field_enable_2fa() {
    $value = get_option('ghostgate_enable_2fa', '1');

    echo '<br>';
    echo '<label class="ghostgate-switch">';
    echo '<input type="checkbox" name="ghostgate_enable_2fa" value="1"' . checked($value, '1', false) . '>';
    echo '<span class="ghostgate-slider"></span>';
    echo '</label>';
    echo '<span>' . esc_html__('有効にする', 'ghostgate') . '</span>';

    echo '<p class="description">' . esc_html__('ユーザー設定に基づいて、2段階認証を有効にします。', 'ghostgate') . '</p>';
    echo '<p class="description">' . esc_html__('ワードプレスのユーザー毎に2FAのチェックを入れてください。', 'ghostgate') . '</p>';
}




// admin-ajaxのリクエスト制限数
function ghostgate_field_ajax_limit() {
    $value    = absint( get_option( 'ghostgate_ajax_limit', 30 ) );
    $cooldown = absint( get_option( 'ghostgate_ajax_cooldown', 60 ) );
    $enabled  = get_option( 'ghostgate_enable_ajax_limit', 1 );
    $notify   = get_option( 'ghostgate_ajax_notify', 0 );

    echo '<br>';
    echo '<label class="ghostgate-switch">';
    echo '<input type="checkbox" name="ghostgate_enable_ajax_limit" value="1" ' . checked( (int) $enabled, 1, false ) . '>';
    echo '<span class="ghostgate-slider"></span>';
    echo '</label> ';
    echo '<span>' . esc_html__( '使用する', 'ghostgate' ) . '</span><br>';

    echo '<label>';
    echo esc_html__( 'リクエスト数:', 'ghostgate' ) . ' ';
    echo '<input type="number" name="ghostgate_ajax_limit" value="' . esc_attr( $value ) . '" min="1" step="1" inputmode="numeric" pattern="\d*"> ';
    echo esc_html__( '回/1分', 'ghostgate' ); // ← 「期間」を明記
    echo '</label><br>';

    echo '<label>';
    echo esc_html__( 'クールダウン:', 'ghostgate' ) . ' ';
    echo '<input type="number" name="ghostgate_ajax_cooldown" value="' . esc_attr( $cooldown ) . '" min="10" step="1" inputmode="numeric" pattern="\d*"> ';
    echo esc_html__( '秒', 'ghostgate' );
    echo '</label><br>';

    echo '<label class="ghostgate-switch">';
    echo '<input type="checkbox" name="ghostgate_ajax_notify" value="1" ' . checked( (int) $notify, 1, false ) . '>';
    echo '<span class="ghostgate-slider"></span>';
    echo '</label> ';
    echo '<span>' . esc_html__( '超過時に管理者へ通知', 'ghostgate' ) . '</span>';

    echo '<p class="description">' . esc_html__( 'admin-ajax.php へのアクセスを 1 分単位で制限します。指定回数を超えると 429 応答と通知が行われます。', 'ghostgate' ) . '</p>';
}


// ログファイルの保存パス
function ghostgate_field_log_file_path() {
    // WP_CONTENT_DIR は WordPress コアが定義する定数です。
    // ログファイル保存先の初期値として使用します。
    // WP_CONTENT_DIR is a WordPress core constant.
    // Used here as the default path for log file storage.
    $path_value = esc_attr(get_option('ghostgate_log_file_path', WP_CONTENT_DIR . '/ghostgate_logs.log'));
    $enabled    = get_option('ghostgate_enable_log_file', 1);
    $days_value = esc_attr(get_option('ghostgate_log_retention_days', '60')); // デフォルト60日

    // ✅ スイッチUI：ログ記録ON/OFF
    echo '<br>';
    echo '<label class="ghostgate-switch">';
    echo '<input type="checkbox" name="ghostgate_enable_log_file" value="1"' . checked($enabled, 1, false) . '>';
    echo '<span class="ghostgate-slider"></span>';
    echo '</label>';
    echo '<span>' . esc_html__('使用する', 'ghostgate') . '</span><br>';

    // ✅ ログファイルパス入力欄
    echo '<input type="text" name="ghostgate_log_file_path" value="' . esc_attr($path_value) . '" class="regular-text">';
    echo '<p class="description">' .
        esc_html__('ログファイルの保存先を指定します（デフォルト: wp-content/uploads/ghostgate/logs/ghostgate_logs_YYYYMMDD.log）', 'ghostgate') .
        '</p>';

    // ✅ ログ保存日数
    echo '<label for="ghostgate_log_retention_days">';
    echo esc_html__('保存日数（空欄で60日）', 'ghostgate');
    echo '</label><br>';

    echo '<input type="number" name="ghostgate_log_retention_days" value="' . esc_attr($days_value) . '" min="1" max="365" style="width: 80px;"> ';
    echo '<p class="description">' .
        esc_html__('ログファイルを自動削除するまでの日数。例：60', 'ghostgate') .
        '</p>';
}


//ログイン制限
function ghostgate_field_login_limit_settings() {
    $enable_limit  = get_option( 'ghostgate_enable_login_limit', '1' );
    $attempt_limit = intval( get_option( 'ghostgate_login_attempt_limit', 3 ) );
    $notify        = get_option( 'ghostgate_notify_on_block', '1' );

    // 追加：現在のクールダウン秒（未保存なら 15 分）
    $cooldown = absint( get_option( 'ghostgate_login_block_cooldown', 15 * MINUTE_IN_SECONDS ) );

    echo '<br>';
    echo '<label class="ghostgate-switch">';
    echo '<input type="checkbox" name="ghostgate_enable_login_limit" value="1" ' . checked( $enable_limit, '1', false ) . '>';
    echo '<span class="ghostgate-slider"></span>';
    echo '</label> ';
    echo '<span>' . esc_html__( 'ログイン制限を有効にする', 'ghostgate' ) . '</span><br><br>';

    echo '<label>';
    echo esc_html__( 'ログイン試行回数制限: ', 'ghostgate' ) . ' ';
    echo '<input type="number" name="ghostgate_login_attempt_limit" value="' . esc_attr( $attempt_limit ) . '" min="1" style="width:80px;"> ';
    echo '</label>';
    echo '<span class="description">' . esc_html__( '30分以内に許可する最大ログイン試行回数。例：3回まで', 'ghostgate' ) . '</span><br><br>';

    // ★ ここが今回の追加：ブロック維持時間（秒）
    echo '<label>';
    echo esc_html__( 'ブロックのクールダウン: ', 'ghostgate' ) . ' ';
    echo '<input type="number" name="ghostgate_login_block_cooldown" value="' . esc_attr( $cooldown ) . '" min="60" step="10" inputmode="numeric" pattern="\d*" style="width:100px;"> ';
    echo esc_html__( '秒', 'ghostgate' );
    echo '</label>';
    echo '<span class="description">' . esc_html__( 'ブロック発動後、この秒数が経過すると自動解除されます。', 'ghostgate' ) . '</span><br><br>';

    echo '<label class="ghostgate-switch">';
    echo '<input type="checkbox" name="ghostgate_notify_on_block" value="1"' . checked( $notify, '1', false ) . '>';
    echo '<span class="ghostgate-slider"></span>';
    echo '</label> ';
    echo '<span>' . esc_html__( '管理者にメールで通知', 'ghostgate' ) . '</span>';
}




// author列挙ブロックの有効化
function ghostgate_field_block_author_enum() {
    $value = get_option('ghostgate_block_author_enum', '1');

    echo '<br>';
    echo '<label class="ghostgate-switch">';
    echo '<input type="checkbox" name="ghostgate_block_author_enum" value="1"' . checked($value, '1', false) . '>';
    echo '<span class="ghostgate-slider"></span>';
    echo '</label> ';
    echo '<span>' . esc_html__('有効にする', 'ghostgate') . '</span>';

    echo '<p class="description">';
    echo esc_html__('?author=1 のようなリクエストによるユーザー名列挙を遮断します。', 'ghostgate');
    echo '</p>';
}


//xmlrpc制御
function ghostgate_field_disable_xmlrpc() {
    $value = get_option('ghostgate_disable_xmlrpc', '1');

    echo '<br>';
    echo '<label class="ghostgate-switch">';
    echo '<input type="checkbox" name="ghostgate_disable_xmlrpc" value="1"' . checked($value, '1', false) . '>';
    echo '<span class="ghostgate-slider"></span>';
    echo '</label> ';
    echo '<span>' . esc_html__('有効にする', 'ghostgate') . '</span>';

    echo '<p class="description">';
    echo esc_html__('セキュリティ向上のため、XML-RPCアクセスを遮断します。', 'ghostgate');
    echo '</p>';
}


//未使用REST APIの遮断
function ghostgate_field_block_unused_rest() {
    $enabled = get_option('ghostgate_block_unused_rest', 0);

    echo '<br>';
    echo '<label class="ghostgate-switch">';
    echo '<input type="checkbox" name="ghostgate_block_unused_rest" value="1"' . checked($enabled, 1, false) . '>';
    echo '<span class="ghostgate-slider"></span>';
    echo '</label> ';
    echo '<span>' . esc_html__('未使用の標準REST APIを外部から遮断する', 'ghostgate') . '</span>';

    echo '<p class="description">';
    echo esc_html__('Gutenberg、Application Passwords、メディアAPI などの未使用APIを制限します。', 'ghostgate');
    echo '</p>';
}


//REST APIの試行回数制限
function ghostgate_field_rest_limit_group() {
    $enabled  = get_option('ghostgate_enable_rest_limit', 0);
    $limit    = absint(get_option('ghostgate_rest_limit_count', 20));
    $cooldown = absint(get_option('ghostgate_rest_limit_cooldown', 300));

    // ✅ REST API 制限 ON/OFF スイッチ
    echo '<br>';
    echo '<label class="ghostgate-switch">';
    echo '<input type="checkbox" name="ghostgate_enable_rest_limit" value="1"' . checked($enabled, 1, false) . '>';
    echo '<span class="ghostgate-slider"></span>';
    echo '</label>';
    echo '<span>' . esc_html__('REST APIアクセス制限を有効化', 'ghostgate') . '</span><br><br>';

    // ✅ 回数上限入力
    echo '<label>';
    echo esc_html__('回数上限（1分）: ', 'ghostgate');
    echo '<input type="number" name="ghostgate_rest_limit_count" value="' . esc_attr($limit) . '" class="small-text" min="1">';
    echo '</label>';

    // ✅ ブロック時間入力
    echo '<label>';
    echo ' ' . esc_html__('ブロック時間（秒）: ', 'ghostgate');
    echo '<input type="number" name="ghostgate_rest_limit_cooldown" value="' . esc_attr($cooldown) . '" class="small-text" min="1">';
    echo '</label><br><br>';

    // ✅ 説明文
    echo '<p class="description">';
    echo esc_html__('1分間に指定回数を超えるRESTリクエストを送信すると、そのIPは指定秒数ブロックされます。', 'ghostgate');
    echo '</p>';
}

// ✅ wp-json出力の秘匿化（RESTエンドポイントのステルス化 + ホワイトリストUI）
function ghostgate_field_hide_json_endpoints() {
    global $ghostgate_bypass_json_filter;
    $ghostgate_bypass_json_filter = true; // ✅ フィルター一時解除

    $enabled           = get_option('ghostgate_hide_json_endpoints', '0');
    $allowed_prefixes  = get_option('ghostgate_json_allowed_prefixes', '');
    $allowed_routes    = get_option('ghostgate_json_allowed_routes', []);

    $registered_routes = array_keys(rest_get_server()->get_routes());

    $ghostgate_bypass_json_filter = false; // ✅ フィルター復帰

    // ✅ トグルスイッチ
    echo '<br>';
    echo '<label class="ghostgate-switch">';

    echo '<input type="hidden" name="ghostgate_hide_json_endpoints" value="0">';

    echo '<input type="checkbox" id="ghostgate_hide_json_endpoints" name="ghostgate_hide_json_endpoints" value="1"' . checked($enabled, '1', false) . '>';
    echo '<span class="ghostgate-slider"></span>';
    echo '</label>';
    echo '<span>' . esc_html__('wp-json構造から隠す', 'ghostgate') . '</span>';
    echo '<p class="description">' . esc_html__('有効にすると、下記で選択されなかったREST APIは wp-json に表示されなくなります。', 'ghostgate') . '</p>';

    // ✅ アコーディオンのトグルボタン
    echo '<button type="button" class="button" id="ghostgate-toggle-route-list">';
    echo esc_html__('▼ 表示を許可するRESTエンドポイントを選択', 'ghostgate');
    echo '</button>';

    // ✅ ホワイトリスト選択フィールド（アコーディオン）
    echo '<fieldset id="ghostgate-route-list" class="hidden" style="margin-top: 15px; padding: 10px; border: 1px solid #ccc;">';
    echo '<legend style="font-weight: bold;">' . esc_html__('表示を許可するRESTエンドポイント', 'ghostgate') . '</legend>';
    foreach ($registered_routes as $route) {
        $checked = in_array($route, (array) $allowed_routes, true) ? 'checked' : '';
echo '<label class="ghostgate-route-label">';
echo '<input type="checkbox" name="ghostgate_json_allowed_routes[]" value="' . esc_attr($route) . '" ' . checked(in_array($route, (array) $allowed_routes, true), true, false) . '>';
echo '<span class="ghostgate-route-name">' . esc_html($route) . '</span>';
echo '</label>';
    }
    echo '</fieldset>';

    // ✅ 自作接頭辞入力
    echo '<p style="margin-top: 20px;"><strong>' . esc_html__('自作REST APIの接頭辞（カンマ区切り）', 'ghostgate') . '</strong></p>';
    echo '<input type="text" name="ghostgate_json_allowed_prefixes" value="' . esc_attr($allowed_prefixes) . '" style="width:100%">';
    echo '<p class="description">' . esc_html__('例: sbp2, ghostgate など。該当するルートは常に表示されます。', 'ghostgate') . '</p>';
}


//サニタイズ関数
function ghostgate_sanitize_array($value) {
    return is_array($value) ? array_map('sanitize_text_field', $value) : [];
}

//ログインセッション管理
function ghostgate_field_session_control() {
    $enabled = get_option('ghostgate_enable_session_control', 0);
    $timeout = (int) get_option('ghostgate_session_timeout', 1800);

    echo '<br>';
    echo '<label class="ghostgate-switch">';
    echo '<input type="checkbox" name="ghostgate_enable_session_control" value="1"' . checked($enabled, 1, false) . '>';
    echo '<span class="ghostgate-slider"></span>';
    echo '</label>';
    echo ' <span>' . esc_html__('ログイン中のセッションを監視する（アイドル時間で強制ログアウト）', 'ghostgate') . '</span>';

    echo '<p class="description">';
    echo esc_html__('セッション有効中は、アイドル時間（何も操作されていない時間）が設定時間を超えると自動ログアウトされます。', 'ghostgate');
    echo '</p>';

    echo '<label for="ghostgate_session_timeout">' . esc_html__('アイドル時間の上限（秒）:', 'ghostgate') . '</label><br>';
    echo '<input type="number" id="ghostgate_session_timeout" name="ghostgate_session_timeout" value="' . esc_attr($timeout) . '" min="30" step="1" style="width: 100px;"> ';
    echo esc_html__('秒', 'ghostgate');

    // === 現在のステータス表示 ===
    // セッションが未開始でも安全に読む
    if ( ! session_id() ) { @session_start(); }

    if ( isset($_SESSION['ghostgate_last_active'], $_SESSION['ghostgate_login_time']) ) {
        $now   = ghostgate_now(); // ← サイトTZで統一
        $login = (int) $_SESSION['ghostgate_login_time'];
        $last  = (int) $_SESSION['ghostgate_last_active'];

        // まれな時計ずれ（last > now）を吸収
        if ( $last > $now ) { $last = $now; }

        $elapsed   = max($now - $last, 0);
        $remaining = max($timeout - $elapsed, 0);

        echo '<hr><strong>' . esc_html__('現在のセッション情報', 'ghostgate') . '</strong><br>';
        echo esc_html__('ログイン時刻: ', 'ghostgate') . esc_html( ghostgate_wp_date('Y-m-d H:i:s', $login) ) . '<br>';
        echo esc_html__('最終操作: ', 'ghostgate') . esc_html( ghostgate_wp_date('Y-m-d H:i:s', $last) ) . '<br>';
        echo '<span>' . esc_html__('⏱ アイドル経過時間: ', 'ghostgate') . '</span>';
        echo '<span id="ghostgate-elapsed" data-elapsed="' . esc_attr($elapsed) . '">' . esc_html($elapsed) . esc_html__(' 秒', 'ghostgate') . '</span><br>';
        echo '<span>' . esc_html__('⏳ タイムアウトまで残り: ', 'ghostgate') . '</span>';
        echo '<span id="ghostgate-remaining" data-remaining="' . esc_attr($remaining) . '">' . esc_html($remaining) . esc_html__(' 秒', 'ghostgate') . '</span><br>';
    } else {
        echo '<hr><em>' . esc_html__('セッションはまだ開始されていません。', 'ghostgate') . '</em>';
    }
}


// プレビュー表示のブロック（URL直叩きを403で遮断）
function ghostgate_field_block_preview() {
    $enabled = get_option('ghostgate_block_preview', 0);

    echo '<br>';
    echo '<label class="ghostgate-switch">';
    echo '<input type="checkbox" id="ghostgate_block_preview" name="ghostgate_block_preview" value="1" ' . checked($enabled, 1, false) . '>';
    echo '<span class="ghostgate-slider"></span>';
    echo '</label> ';

    echo '<span>' . esc_html__('有効にする', 'ghostgate') . '</span>';
    echo '<p class="description" style="margin-top:.5em;">'
        . esc_html__('フロントエンドの ?preview=…（下書き・変更のプレビュー）をURL直叩きで開けないようにします。', 'ghostgate')
        . '<br>'
        . esc_html__('有効時は、ブラウザのアドレスバー等でプレビューURLを開くと 403 Forbidden を返します。', 'ghostgate')
        . '</p>';
}



////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//ステータス管理画面
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

//ワードプレス内のセキュリティ系ツールの一覧調査
function ghostgate_render_security_status_rows() {
    if (!function_exists('get_plugins')) {
        require_once ABSPATH . 'wp-admin/includes/plugin.php';
    }

    $plugins = get_plugins();
    $active_plugins = get_option('active_plugins', []);
    $security_plugins = [];

    // ✅ 1. セキュリティ系プラグイン一覧（GhostGate含む）
    foreach ($plugins as $plugin_file => $plugin_data) {
        $text = strtolower($plugin_data['Name'] . ' ' . $plugin_data['Description'] . ' ' . $plugin_data['TextDomain']);
        if (preg_match('/(security|protect|firewall|auth|guard|ghostgate)/', $text)) {
            $security_plugins[] = [
                'name'   => $plugin_data['Name'],
                'file'   => $plugin_file,
                'slug'   => strtolower(basename(dirname($plugin_file))),
                'active' => in_array($plugin_file, $active_plugins, true),
            ];
        }
    }

    echo '<tr><th scope="row">🔌 ' . esc_html__('セキュリティ系プラグインの状態', 'ghostgate') . '</th><td>';
    if (empty($security_plugins)) {
        echo '<span class="ghostgate-badge warning">' . esc_html__('見つかりませんでした', 'ghostgate') . '</span>';
    } else {
        echo '<ul>';
        foreach ($security_plugins as $plugin) {
            $status = $plugin['active']
                ? '<span class="ghostgate-badge success">' . esc_html__('有効', 'ghostgate') . '</span>'
                : '<span class="ghostgate-badge danger">' . esc_html__('無効', 'ghostgate') . '</span>';
            $status = $plugin['active']
                ? '<span class="ghostgate-badge success">' . esc_html__('有効', 'ghostgate') . '</span>'
                : '<span class="ghostgate-badge danger">' . esc_html__('無効', 'ghostgate') . '</span>';

            echo '<li>' . esc_html($plugin['name']) . ' ' . wp_kses_post($status) . '</li>';

        }
        echo '</ul>';
    }
    echo '</td></tr>';

    // ✅ 2. .htaccess の保護解析
    $htaccess = ABSPATH . '.htaccess';
    $htaccess_features = [];

    if (file_exists($htaccess)) {

        require_once ABSPATH . 'wp-admin/includes/file.php';
        WP_Filesystem(); global $wp_filesystem;
        $ht_path = wp_normalize_path( $htaccess );
        $content = ( $wp_filesystem && $wp_filesystem->exists( $ht_path ) && $wp_filesystem->is_readable( $ht_path ) )
            ? (string) $wp_filesystem->get_contents( $ht_path )
            : '';


        if (preg_match('/RewriteCond\s+%{QUERY_STRING}.*author=/i', $content)) {
            $htaccess_features[] = [__('ユーザー列挙の遮断', 'ghostgate'), 'success'];
        }
        if (preg_match('/xmlrpc/i', $content)) {
            $htaccess_features[] = [__('XML-RPCの無効化', 'ghostgate'), 'success'];
        }
        if (preg_match('/wp-config\.php/i', $content)) {
            $htaccess_features[] = [__('wp-config.phpへのアクセス防止', 'ghostgate'), 'success'];
        }
        if (preg_match('/<IfModule\s+mod_security/i', $content)) {
            $htaccess_features[] = [__('mod_securityによる保護', 'ghostgate'), 'success'];
        }

        if (preg_match('/#.*(cloudsecure|wp-cerber|wordfence|ithemes)/i', $content, $match)) {
            $slug = strtolower($match[1]);
            $matched_plugin = null;

            foreach ($security_plugins as $plugin) {
                if (strpos($plugin['slug'], $slug) !== false) {
                    $matched_plugin = $plugin;
                    break;
                }
            }

            if ($matched_plugin && !$matched_plugin['active']) {
                // translators: %s is the plugin slug (plugin name) that left a signature in .htaccess while being inactive.
                $htaccess_features[] = [sprintf(__('%s の署名が残っています（プラグイン無効）', 'ghostgate'), $slug), 'warning'];
            } else {
                 // translators: %s is the plugin slug (plugin name) that added a signature to .htaccess.
                $htaccess_features[] = [sprintf(__('%s による .htaccess署名', 'ghostgate'), $slug), 'success'];
            }
        }
    }

    echo '<tr><th scope="row">🧱 ' . esc_html__('.htaccess による保護', 'ghostgate') . '</th><td>';
    if (!empty($htaccess_features)) {
        echo '<ul>';
        foreach ($htaccess_features as [$feat, $risk]) {
            echo '<li><span class="ghostgate-badge ' . esc_attr($risk) . '">' . esc_html($feat) . '</span></li>';
        }
        echo '</ul>';
    } else {
        echo '<span class="ghostgate-badge danger">' . esc_html__('保護設定が見つかりませんでした', 'ghostgate') . '</span>';
    }
    echo '</td></tr>';

    // ✅ 3. wp-config.php の保護強度
    $wpconfig = ABSPATH . 'wp-config.php';
    $config_features = [];

    if (file_exists($wpconfig)) {

        require_once ABSPATH . 'wp-admin/includes/file.php';
        WP_Filesystem(); global $wp_filesystem;
        $wpconfig_path = wp_normalize_path( $wpconfig );
        $content = ( $wp_filesystem && $wp_filesystem->exists( $wpconfig_path ) && $wp_filesystem->is_readable( $wpconfig_path ) )
            ? (string) $wp_filesystem->get_contents( $wpconfig_path )
            : '';


        if (preg_match('/DISALLOW_FILE_EDIT/i', $content)) {
            $config_features[] = __('テーマ・プラグイン編集禁止（DISALLOW_FILE_EDIT）', 'ghostgate');
        }
        if (preg_match('/DISALLOW_FILE_MODS/i', $content)) {
            $config_features[] = __('管理画面でのインストール禁止（DISALLOW_FILE_MODS）', 'ghostgate');
        }
        if (preg_match('/FORCE_SSL_ADMIN/i', $content)) {
            $config_features[] = __('管理画面SSL強制（FORCE_SSL_ADMIN）', 'ghostgate');
        }
        if (preg_match('/WP_DEBUG\s*,\s*false/i', $content)) {
            $config_features[] = __('デバッグモードOFF', 'ghostgate');
        }
    }

    echo '<tr><th scope="row">🔐 ' . esc_html__('wp-config.php のセキュリティ設定', 'ghostgate') . '</th><td>';
    if (!empty($config_features)) {
        echo '<ul>';
        foreach ($config_features as $feat) {
            echo '<li><span class="ghostgate-badge success">' . esc_html($feat) . '</span></li>';
        }
        echo '</ul>';
    } else {
        echo '<span class="ghostgate-badge warning">' . esc_html__('保護設定が検出されていません', 'ghostgate') . '</span>';
    }
    echo '</td></tr>';

    // ✅ 追加のファイル解析
    ghostgate_render_file_security_details();
}



//詳細を調べて表示する
function ghostgate_render_file_security_details() {
    $loginurl_features     = ghostgate_analyze_login_url_change();
    $twofa_features        = ghostgate_analyze_2fa_status();
    $adminajax_features    = ghostgate_analyze_admin_ajax_limit();
    $loginlimit_features   = ghostgate_analyze_login_attempt_limit();
    $authorenum_features   = ghostgate_analyze_author_enumeration_protection();        // ?author=1
    $restenum_features     = ghostgate_analyze_rest_user_enumeration_protection();     // /wp-json/wp/v2/users
    $xmlrpc_features = ghostgate_analyze_xmlrpc_blocking(); // ✅ XML-RPC遮断チェック


    $htaccess_features     = ghostgate_analyze_htaccess_security();
    $wpconfig_features     = ghostgate_analyze_wpconfig_security();
    $userini_features      = ghostgate_analyze_userini_security();
    $phpini_features       = ghostgate_analyze_php_ini_security();
    $access_features       = ghostgate_analyze_dot_access_security();
    $env_features          = ghostgate_analyze_env_security();
    $htpasswd_features     = ghostgate_analyze_htpasswd_security();
    $robots_features       = ghostgate_analyze_robots_txt_security();
    $ftpaccess_features    = ghostgate_analyze_ftpaccess_security();

    $restapi_detailed = ghostgate_analyze_rest_api_detailed();
    $restlimit_features = ghostgate_analyze_rest_rate_limit_status(); 
    $json_hide_features = ghostgate_analyze_json_structure_obfuscation();

    $all_features = [
        'loginurl'   => $loginurl_features,
        '2fa'        => $twofa_features,
        'adminajax'  => $adminajax_features,
        'loginlimit' => $loginlimit_features,
        'authorenum' => $authorenum_features,
        'restenum'   => $restenum_features,
        'restlimit'  => $restlimit_features,
        'xmlrpc'     => $xmlrpc_features,
        'htaccess'   => $htaccess_features,
        'wpconfig'   => $wpconfig_features,
        'userini'    => $userini_features,
        'phpini'     => $phpini_features,
        'access'     => $access_features,
        'env'        => $env_features,
        'htpasswd'   => $htpasswd_features,
        'robots'     => $robots_features,
        'ftpaccess'  => $ftpaccess_features,
    ];

    $evaluation = ghostgate_calculate_security_level($all_features);

    echo '<tr><th scope="row">🛡 ' . esc_html__('セキュリティレベル', 'ghostgate') . '</th><td>';
    printf(
        '<strong>%s</strong> %s (%s / %s)',
        esc_html($evaluation['grade']),
        esc_html__('評価', 'ghostgate'),
        esc_html($evaluation['score']),
        esc_html($evaluation['total'])
    );
    echo '</td></tr>';


    $sections = [
        ['title' => '🔐 ' . __('ログインURLの秘匿状況', 'ghostgate'), 'features' => $loginurl_features],
        ['title' => '📧 ' . __('2FA認証の設定状況', 'ghostgate'), 'features' => $twofa_features],
        ['title' => '🚦 ' . __('admin-ajax.php のアクセス制限', 'ghostgate'), 'features' => $adminajax_features],
        ['title' => '🔁 ' . __('ログイン試行回数の制限', 'ghostgate'), 'features' => $loginlimit_features],
        ['title' => '🙅‍♂️ ' . __('ユーザー列挙の遮断状況 (?author=)', 'ghostgate'), 'features' => $authorenum_features],
        ['title' => '🙅‍♀️ ' . __('REST APIのユーザー列挙遮断', 'ghostgate'), 'features' => $restenum_features],
        ['title' => '📉 ' . __('REST APIリクエスト制限の状況', 'ghostgate'), 'features' => $restlimit_features],
        ['title' => '📴 ' . __('XML-RPC の遮断状況', 'ghostgate'), 'features' => $xmlrpc_features],
        ['title' => '👻 ' . __('REST API 構造の秘匿化状況', 'ghostgate'), 'features' => $json_hide_features,],
        ['title' => '🧾 ' . __('.htaccess の詳細', 'ghostgate'), 'features' => $htaccess_features],
        ['title' => '⚙️ ' . __('wp-config.php の詳細', 'ghostgate'), 'features' => $wpconfig_features],
        ['title' => '📄 ' . __('.user.ini の詳細', 'ghostgate'), 'features' => $userini_features],
        ['title' => '🧬 ' . __('php.ini の詳細', 'ghostgate'), 'features' => $phpini_features],
        ['title' => '📄 ' . __('.access の詳細', 'ghostgate'), 'features' => $access_features],
        ['title' => '🧬 ' . __('.env の詳細', 'ghostgate'), 'features' => $env_features],
        ['title' => '🔒 ' . __('.htpasswd の詳細', 'ghostgate'), 'features' => $htpasswd_features],
        ['title' => '🤖 ' . __('robots.txt の詳細', 'ghostgate'), 'features' => $robots_features],
        ['title' => '📁 ' . __('.ftpaccess の詳細', 'ghostgate'), 'features' => $ftpaccess_features],
        ['title' => '🔌 ' . __('標準REST APIの利用状況', 'ghostgate'), 'features' => $restapi_detailed],
    ];

    foreach ($sections as $section) {

        echo '<tr><th scope="row">' . esc_html( $section['title'] ) . '</th><td>';

echo wp_kses_post(
    '<details' . (empty($section['features']) ? '' : ' open') . ' style="margin-bottom:10px;">'
);

        echo '<summary><strong>' . esc_html( $section['title'] ) . '</strong></summary>';

        if (empty($section['features'])) {
            echo '<p><span class="ghostgate-badge warning">' . esc_html__('特徴的な保護設定は見つかりません', 'ghostgate') . '</span></p>';
        } else {
            echo '<ul>';
            foreach ($section['features'] as $f) {
                $badge = 'success';
                if (!empty($f['risk']) && $f['risk'] === 'danger') {
                    $badge = 'danger';
                } elseif (!empty($f['risk']) && $f['risk'] === 'warning') {
                    $badge = 'warning';
                }

                echo '<li><span class="ghostgate-badge ' . esc_attr($badge) . '">' . esc_html($f['feature']) . '</span>';
                if (!empty($f['tool'])) {
                    // translators: %1$s is the name of the detected tool.
                    printf(' <small>(' . esc_html__('推定: %s', 'ghostgate') . ')</small>', esc_html($f['tool']));
                }
                echo '</li>';
            }
            echo '</ul>';
        }

        echo '</details>';
        echo '</td></tr>';
    }

}




function ghostgate_calculate_security_level($all_features) {
    $score = 0;
    $total = 0;
    $warnings = [];

    foreach ($all_features as $section => $features) {
        foreach ($features as $f) {
            $total++;
            $suggestion = isset($f['suggestion']) ? $f['suggestion'] : __('詳細情報なし', 'ghostgate');

            if (isset($f['risk']) && $f['risk'] === 'danger') {
                // translators: %1$s is the feature name, %2$s is the suggestion message
                $warnings[] = sprintf(__('%1$s（%2$s）', 'ghostgate'), $f['feature'], $suggestion);
            } elseif (isset($f['risk']) && $f['risk'] === 'warning') {
                $score += 0.6;
                // translators: %1$s is the feature name, %2$s is the suggestion message
                $warnings[] = sprintf(__('%1$s（%2$s）', 'ghostgate'), $f['feature'], $suggestion);
            } else {
                $score += 1.0;
            }

        }
    }

    $ratio = $total ? $score / $total : 0;
    $grade = 'E';

    if ($ratio >= 0.85) {
        $grade = 'S';
    } elseif ($ratio >= 0.70) {
        $grade = 'A';
    } elseif ($ratio >= 0.55) {
        $grade = 'B';
    } elseif ($ratio >= 0.40) {
        $grade = 'C';
    } elseif ($ratio >= 0.25) {
        $grade = 'D';
    }

    return [
        'grade'    => $grade,
        'warnings' => $warnings,
        'score'    => round($score, 2),
        'total'    => $total
    ];
}




//ログインURLの変更の詳細確認
function ghostgate_analyze_login_url_change() {
    $features = [];
    $plugins = get_option('active_plugins', []);
    $plugin_detected = false;
    $ghostgate_detected = false;
    $other_plugins = [];

    $known_plugins = [
        'wps-hide-login'     => 'WPS Hide Login',
        'rename-wp-login'    => 'Rename WP Login',
        'hide-my-wp'         => 'Hide My WP',
        'loginpress'         => 'LoginPress',
        'ithemes-security'   => 'iThemes Security',
        'wp-cerber'          => 'WP Cerber',
        'wordfence'          => 'Wordfence Login Security',
    ];

    foreach ($plugins as $plugin_file) {
        $slug = plugin_basename($plugin_file);

        foreach ($known_plugins as $key => $name) {
            if (stripos($slug, $key) !== false) {
                $plugin_detected = true;
                $other_plugins[] = $name;
            }
        }

        if (stripos($slug, 'ghostgate') !== false) {
            $ghostgate_detected = true;
        }
    }

    $ghostgate_enabled = get_option('ghostgate_enable_login_slug') && get_option('ghostgate_login_slug');

    $htaccess_path = wp_normalize_path( ABSPATH . '.htaccess' );
    $htaccess_blocks_login = false;

    require_once ABSPATH . 'wp-admin/includes/file.php';
    WP_Filesystem(); global $wp_filesystem;

    if ( $wp_filesystem && $wp_filesystem->exists( $htaccess_path ) && $wp_filesystem->is_readable( $htaccess_path ) ) {
        $ht = (string) $wp_filesystem->get_contents( $htaccess_path );
        if ( $ht !== '' && preg_match( '/wp-login\.php.*(deny|forbid|redirect|rewrite|RewriteRule)/i', $ht ) ) {
            $htaccess_blocks_login = true;
        }
    }


    $wp_login_path   = wp_normalize_path( ABSPATH . 'wp-login.php' );
    $wp_login_exists = ( $wp_filesystem && $wp_filesystem->exists( $wp_login_path ) );


    global $wp_rewrite;
    $rewrite_rules_modified = false;
    if (isset($wp_rewrite) && method_exists($wp_rewrite, 'wp_rewrite_rules')) {
        $rules = $wp_rewrite->wp_rewrite_rules();
        foreach ($rules as $regex => $query) {
            if (stripos($query, 'wp-login.php') !== false && stripos($regex, 'login') === false) {
                $rewrite_rules_modified = true;
                break;
            }
        }
    }

    if ($ghostgate_enabled) {
        $features[] = [
            'feature'    => __('GhostGate によるログインURL秘匿が有効です', 'ghostgate'),
            // translators: %s is the login slug used in the GhostGate settings
            'tool'       => sprintf(__('GhostGate（slug: %s）', 'ghostgate'), get_option('ghostgate_login_slug')),
            'risk'       => 'safe',
            'suggestion' => __('GhostGateが動作中です。認証コード＋URL秘匿により、高いセキュリティが確保されています。', 'ghostgate'),
        ];
    } elseif ($plugin_detected || $htaccess_blocks_login || $rewrite_rules_modified) {
        $tools = implode(', ', array_unique($other_plugins));
        $features[] = [
            'feature'    => __('ログインURLの変更・秘匿化が検出されました', 'ghostgate'),
            'tool'       => $tools !== '' ? $tools : __('.htaccess または rewrite', 'ghostgate'),
            'risk'       => 'safe',
            'suggestion' => __('ログインURLの秘匿は攻撃リスクの低減に有効です。2FAやブロック制御と併用すると強固です。', 'ghostgate'),
        ];
    } elseif (!$wp_login_exists) {
        $features[] = [
            'feature'    => __('wp-login.php が存在しません（削除 or 隠蔽の可能性）', 'ghostgate'),
            'tool'       => __('カスタム構成・セキュリティ系プラグイン', 'ghostgate'),
            'risk'       => 'safe',
            'suggestion' => __('意図的な秘匿であれば問題ありませんが、アクセス不能になっていないか確認を推奨します', 'ghostgate'),
        ];
    } else {
        $features[] = [
            'feature'    => __('ログインURLは既定のまま（wp-login.php）', 'ghostgate'),
            'tool'       => __('WordPress標準のログイン構成', 'ghostgate'),
            'risk'       => 'warning',
            'suggestion' => __('ログインURLの変更または2FA導入を推奨します（GhostGateや他セキュリティプラグイン等）', 'ghostgate'),
        ];
    }

    return $features;
}




//2faの設定状況確認
function ghostgate_analyze_2fa_status() {
    $features = [];
    $users = get_users(['fields' => ['ID', 'user_login']]);
    if (empty($users)) {
        return [[
            'feature'    => __('ユーザー情報が取得できませんでした', 'ghostgate'),
            'tool'       => __('2FA構成チェック（コード: 2FA-000）', 'ghostgate'),
            'risk'       => 'warning',
            'suggestion' => __('WordPressのユーザー情報を確認できませんでした。2FAのステータス確認が行えません。', 'ghostgate'),
        ]];
    }

    $enabled_count = 0;
    $total_count = count($users);
    $disabled_users = [];

    $active_plugins = get_option('active_plugins', []);
    $ghostgate_2fa_enabled = get_option('ghostgate_enable_2fa', '1') === '1';

    $plugin_map = [
        'wp-2fa/wp-2fa.php' => [
            'name'       => 'WP 2FA',
            'active'     => false,
            'check_meta' => fn($uid) => get_user_meta($uid, 'wp_2fa_enabled', true) === 'enabled',
            'ui_field'   => 'wp_2fa_enabled',
        ],
        'miniorange-2-factor-authentication/miniorange-2fa.php' => [
            'name'       => 'miniOrange 2FA',
            'active'     => false,
            'check_meta' => fn($uid) => get_user_meta($uid, 'mo_2fa_status', true) === 'enabled',
            'ui_field'   => 'mo_2fa_status',
        ],
        'wordfence/wordfence.php' => [
            'name'       => 'Wordfence 2FA',
            'active'     => false,
            'check_meta' => fn($uid) => get_user_meta($uid, 'wfls_twoFactorStatus', true) === 'enabled',
            'ui_field'   => 'wfls_twoFactorStatus',
        ],
    ];

    foreach ($plugin_map as $plugin_file => &$data) {
        foreach ($active_plugins as $active) {
            if (stripos($active, $plugin_file) !== false) {
                $data['active'] = true;
                break;
            }
        }
    }

    foreach ($users as $user) {
        $uid = $user->ID;
        $is_enabled = false;

        if ($ghostgate_2fa_enabled && get_user_meta($uid, 'ghostgate_2fa_enabled', true) === '1') {
            $is_enabled = true;
        }

        foreach ($plugin_map as $data) {
            if (
                $data['active'] &&
                $data['check_meta']($uid) &&
                metadata_exists('user', $uid, $data['ui_field'])
            ) {
                $is_enabled = true;
                break;
            }
        }

        if ($is_enabled) {
            $enabled_count++;
        } else {
            $disabled_users[] = $user->user_login;
        }
    }

    if ($enabled_count === $total_count) {
        $features[] = [
            'feature'    => __('すべてのユーザーに 2FA（メール認証）が有効です', 'ghostgate'),
            'tool'       => __('GhostGate / 他2FA（コード: 2FA-200）', 'ghostgate'),
            'risk'       => 'safe',
            'suggestion' => __('2FAが全ユーザーに対して有効です。セキュリティは強化されています。', 'ghostgate'),
        ];
    } elseif ($enabled_count === 0) {
        $features[] = [
            'feature'    => __('2FA（メール認証）が有効なユーザーが存在しません', 'ghostgate'),
            'tool'       => __('GhostGate 未保護状態（コード: 2FA-500）', 'ghostgate'),
            'risk'       => 'danger',
            'suggestion' => __('すべてのユーザーに2FAの有効化を推奨します。特に管理者権限のあるアカウントは優先的に保護してください。', 'ghostgate'),
        ];
    } else {
        $features[] = [
            'feature'    => sprintf(
                // translators: %1$d is the number of users with 2FA enabled, %2$d is the total number of users
                __('%1$d / %2$d のユーザーが 2FA を有効にしています', 'ghostgate'),
                $enabled_count,
                $total_count
            ),
            'tool'       => __('GhostGate / 他2FA 部分保護（コード: 2FA-206）', 'ghostgate'),
            'risk'       => 'warning',
            'suggestion' => __('全ユーザーへの 2FA 適用が推奨されます。', 'ghostgate'),
        ];

        if (!empty($disabled_users)) {
            $list = implode(', ', array_map('esc_html', array_slice($disabled_users, 0, 5)));

            
            $more = count($disabled_users) > 5
            // translators: %d is the number of additional users not shown
                ? sprintf(__(' 他%d名', 'ghostgate'), count($disabled_users) - 5)
                : '';

            
            $features[] = [
                // translators: %1$s is the comma-separated user list, %2$s is the additional user count string (e.g. " 他3名")
                'feature'    => sprintf(__('2FA未設定のユーザー: %1$s%2$s', 'ghostgate'), $list, $more),
                'tool'       => __('未設定ユーザー警告（コード: 2FA-207）', 'ghostgate'),
                'risk'       => 'warning',
                'suggestion' => __('上記のユーザーに対し、GhostGate または他2FA機能を有効にしてください。', 'ghostgate'),
            ];

        }

    }

    $detected = array_filter($plugin_map, fn($d) => $d['active']);
    if (!empty($detected)) {
        $features[] = [
            'feature'    => __('他の2FAプラグインが有効です', 'ghostgate'),
            'tool'       => implode(', ', array_column($detected, 'name')),
            'risk'       => 'info',
            'suggestion' => __('GhostGate以外の2FA機能も正しく動作しているか確認を推奨します。', 'ghostgate'),
        ];
    }

    return $features;
}


//admin-ajaxへのアクセス制限詳細チェック
function ghostgate_analyze_admin_ajax_limit() {
    $features = [];

    // GhostGate設定確認
    $ghostgate_enabled = get_option('ghostgate_enable_ajax_limit');
    $limit = intval(get_option('ghostgate_ajax_limit', 3));
    $cooldown = intval(get_option('ghostgate_ajax_cooldown', 60));

    if ($ghostgate_enabled) {
        $features[] = [
            // translators: %1$d is the number of allowed requests, %2$d is the cooldown period in seconds
            'feature'    => sprintf(__('GhostGateによるadmin-ajax.php制限が有効（%1$d回/%2$d秒）', 'ghostgate'), $limit, $cooldown),
            'tool'       => __('GhostGate 設定機能', 'ghostgate'),
            'risk'       => 'success',
            'suggestion' => __('GhostGateの制限は正常に動作しています。', 'ghostgate'),
        ];
    }

    // .htaccessチェック
    $htaccess_path = wp_normalize_path( ABSPATH . '.htaccess' );

    require_once ABSPATH . 'wp-admin/includes/file.php';
    WP_Filesystem(); global $wp_filesystem;

    if ( $wp_filesystem && $wp_filesystem->exists( $htaccess_path ) ) {
        $content = (string) $wp_filesystem->get_contents( $htaccess_path );

        if ( $content !== '' && preg_match( '/RewriteCond\s+%{REQUEST_URI}.*admin-ajax\.php/i', $content ) ) {
            $features[] = [
                'feature'    => __( 'admin-ajax.php に mod_rewrite によるアクセス制限あり', 'ghostgate' ),
                'tool'       => __( '.htaccess (Apache)', 'ghostgate' ),
                'risk'       => 'info',
                'suggestion' => __( 'GhostGateと併用する場合、意図せぬ競合が発生しないか確認してください。', 'ghostgate' ),
            ];
        }

        if ( $content !== '' && preg_match( '/<Files\s+"admin-ajax\.php">/i', $content ) && preg_match( '/(Deny from|Require ip)/i', $content ) ) {
            $features[] = [
                'feature'    => __( 'admin-ajax.php にIP制限あり', 'ghostgate' ),
                'tool'       => __( '.htaccess', 'ghostgate' ),
                'risk'       => 'info',
                'suggestion' => __( 'サーバーレベルでの制限が有効です。GhostGateと重複の可能性があります。', 'ghostgate' ),
            ];
        }
    }


    // Use trailingslashit(WP_PLUGIN_DIR) to comply with WordPress.org directory path standards.
    // This is used strictly for scanning active plugin folders for compatibility diagnostics.
    $plugin_dirs = glob(trailingslashit(WP_PLUGIN_DIR) . '*', GLOB_ONLYDIR);

    $check_patterns = [
        '/add_action\s*\(\s*[\'"](init|admin_init)[\'"]\s*,\s*[\'"]?.*ajax.*(limit|rate|throttle).*?/i',
        '/wp_die\s*\(\s*[\'"](Too many requests|429)/i',
        '/429 Too Many Requests/i',
        '/(rate_limit|throttle|ajax_limit)_handler/i',
    ];

    foreach ($plugin_dirs as $dir) {
        $php_files = glob($dir . '/*.php');
        foreach ($php_files as $php_file) {

            // すでに上で WP_Filesystem を初期化しているなら次の2行は省略可
            require_once ABSPATH . 'wp-admin/includes/file.php';
            WP_Filesystem(); global $wp_filesystem;

            $php_path = wp_normalize_path( $php_file );
            $code = ( $wp_filesystem && $wp_filesystem->exists( $php_path ) && $wp_filesystem->is_readable( $php_path ) )
                ? (string) $wp_filesystem->get_contents( $php_path )
                : '';


            foreach ($check_patterns as $pattern) {
                if (preg_match($pattern, $code)) {
                    $features[] = [
                        'feature'    => sprintf(
                            // translators: %s is the filename (e.g. plugin or theme PHP file) where the restriction pattern was detected
                            __('他プラグインによる admin-ajax.php 制限処理が検出されました（%s）', 'ghostgate'),
                            basename($php_file)
                        ),
                        'tool'       => __('functions.php または外部プラグイン', 'ghostgate'),
                        'risk'       => 'warning',
                        'suggestion' => __('GhostGateと機能が重複する可能性があります。二重制御に注意してください。', 'ghostgate'),
                    ];
                    break 2;
                }
            }
        }
    }

    // 何もなければ警告
    if (count($features) === 0) {
        $features[] = [
            'feature'    => __('admin-ajax.php へのアクセス制限が見つかりません', 'ghostgate'),
            'tool'       => __('なし', 'ghostgate'),
            'risk'       => 'danger',
            'suggestion' => __('Bot対策のため GhostGate などの制限設定を推奨します。', 'ghostgate'),
        ];
    }

    return $features;
}



//ログインの試行回数制限
function ghostgate_analyze_login_attempt_limit() {
    $features = [];

    // GhostGate自身の設定
    $enabled       = get_option('ghostgate_enable_login_limit', 0);
    $attempt_limit = intval(get_option('ghostgate_login_attempt_limit', 3));

    if ($enabled && $attempt_limit > 0) {
        $features[] = [
            // translators: %d is the maximum number of login attempts allowed in 30 minutes
            'feature'    => sprintf(__('GhostGateによるログイン試行回数制限が有効（30分で最大%d回）', 'ghostgate'), $attempt_limit),
            'tool'       => __('GhostGate（コード: LOGIN-200）', 'ghostgate'),
            'risk'       => 'success',
            'suggestion' => __('制限は正しく動作しています。セキュリティ強化済みです。', 'ghostgate'),
        ];
    }

    // wp-config.php でカスタム定数が使われているか調査
    $wpconfig = ABSPATH . 'wp-config.php';
    if (file_exists($wpconfig)) {

        require_once ABSPATH . 'wp-admin/includes/file.php';
        WP_Filesystem(); global $wp_filesystem;
        $wpconfig_path = wp_normalize_path( $wpconfig );
        $content = ( $wp_filesystem && $wp_filesystem->exists( $wpconfig_path ) && $wp_filesystem->is_readable( $wpconfig_path ) )
            ? (string) $wp_filesystem->get_contents( $wpconfig_path )
            : '';


        if (preg_match('/define\s*\(\s*[\'"]LOGIN_ATTEMPT/i', $content)) {
            $features[] = [
                'feature'    => __('wp-config.php にカスタムの試行回数制御が存在', 'ghostgate'),
                'tool'       => __('カスタム定数定義（コード: LOGIN-300）', 'ghostgate'),
                'risk'       => 'info',
                'suggestion' => __('独自の定数で制限されています。GhostGateと競合しないか確認してください。', 'ghostgate'),
            ];
        }
    }

    // ログイン制限系プラグインの検出
    $active_plugins = get_option('active_plugins', []);
    $known_plugins = [
        'limit-login-attempts-reloaded/limit-login-attempts-reloaded.php' => 'Limit Login Attempts Reloaded',
        'loginizer/loginizer.php' => 'Loginizer',
        'wp-limit-login-attempts/wp-limit-login-attempts.php' => 'WP Limit Login Attempts',
        'wordfence/wordfence.php' => __('Wordfence（ログイン制限機能あり）', 'ghostgate'),
    ];

    $detected_plugins = [];

    foreach ($known_plugins as $file => $name) {
        foreach ($active_plugins as $active) {
            if (stripos($active, $file) !== false) {
                $detected_plugins[] = $name;
                break;
            }
        }
    }

    if (!empty($detected_plugins)) {
        $features[] = [
            'feature'    => __('他プラグインによるログイン試行制限が有効', 'ghostgate'),
            'tool'       => implode(', ', $detected_plugins),
            'risk'       => 'info',
            'suggestion' => __('GhostGateとの機能重複に注意し、優先順位を検討してください。', 'ghostgate'),
        ];
    }

    // 制限が一切無い場合の警告
    if (empty($features)) {
        $features[] = [
            'feature'    => __('ログイン試行回数の制限が検出されませんでした', 'ghostgate'),
            'tool'       => __('未設定（コード: LOGIN-500）', 'ghostgate'),
            'risk'       => 'danger',
            'suggestion' => __('ブルートフォース攻撃防止のため、GhostGate または他の試行回数制限を導入してください。', 'ghostgate'),
        ];
    }

    return $features;
}



//ユーザー列挙のチェック
function ghostgate_analyze_author_enumeration_protection() {
    $features = [];
    $detected = false;

    $active_plugins = get_option('active_plugins', []);
    $known_plugins = [
        'cloudsecurewp/cloudsecurewp.php'                         => 'CloudSecure',
        'wordfence/wordfence.php'                                 => 'Wordfence',
        'ithemes-security-pro/ithemes-security-pro.php'           => 'iThemes Security',
        'all-in-one-wp-security-and-firewall/wp-security.php'    => 'All In One WP Security',
    ];

    // 1. GhostGate による遮断
    if (get_option('ghostgate_block_author_enum') === '1') {
        $features[] = [
            'feature' => __('GhostGate により ?author=ID リクエストが遮断されています', 'ghostgate'),
            'tool'    => __('GhostGate プラグイン', 'ghostgate'),
            'risk'    => 'success',
        ];
        $detected = true;
    }

    // 2. .htaccess による遮断と署名検出
    $htaccess_path = wp_normalize_path( ABSPATH . '.htaccess' );

    require_once ABSPATH . 'wp-admin/includes/file.php';
    WP_Filesystem(); global $wp_filesystem;

    if ( $wp_filesystem && $wp_filesystem->exists( $htaccess_path ) && $wp_filesystem->is_readable( $htaccess_path ) ) {
        $content = (string) $wp_filesystem->get_contents( $htaccess_path );

        if (
            $content !== '' &&
            preg_match( '/RewriteCond\s+%{QUERY_STRING}.*author=\d+/i', $content ) &&
            preg_match( '/RewriteRule\s+.*\s+\[F\]/i', $content )
        ) {
            $features[] = [
                'feature' => __( '?author=ID による列挙を .htaccess で遮断中', 'ghostgate' ),
                'tool'    => '.htaccess',
                'risk'    => 'success',
            ];
            $detected = true;
        }

        if ( $content !== '' && preg_match( '/#.*(cloudsecure|wordfence|sucuri|ithemes|aio|mod[_\s]?security)/i', $content, $matches ) ) {
            $tool_id   = strtolower( $matches[1] );
            $tool_name = ucfirst( $tool_id );

            $found_plugin = null;
            foreach ( $known_plugins as $slug => $label ) {
                if ( stripos( $slug, $tool_id ) !== false ) {
                    $found_plugin = $slug;
                    $tool_name    = $label;
                    break;
                }
            }

if ( $found_plugin && ! in_array( $found_plugin, $active_plugins, true ) ) {
    $msg = sprintf(
        /* translators: %s: Plugin name whose signature remains in .htaccess despite being deactivated */
        __( '%s の署名が .htaccess に残存しています（プラグイン無効）', 'ghostgate' ),
        $tool_name
    );

    $features[] = [
        'feature' => $msg,
        'tool'    => $tool_name,
        'risk'    => 'warning',
    ];
} else {
    $msg = sprintf(
        /* translators: %s: Plugin name whose .htaccess comment signature was detected */
        __( '%s による .htaccess コメント署名を検出', 'ghostgate' ),
        $tool_name
    );

    $features[] = [
        'feature' => $msg,
        'tool'    => $tool_name,
        'risk'    => 'info',
    ];
}


            $detected = true;
        }
    }


    // 3. 有効中のセキュリティ系プラグイン確認
    foreach ($known_plugins as $slug => $name) {
        if (in_array($slug, $active_plugins, true)) {
            $features[] = [
                // translators: %s is the plugin name that is currently active
                'feature' => sprintf(__('%s が有効化されています', 'ghostgate'), $name),
                'tool'    => $name,
                'risk'    => 'info',
            ];
            $detected = true;
        }
    }

    // 4. テーマやプラグインの template_redirect フックによる検出
    global $wp_filter;
    if (
        isset($wp_filter['template_redirect']) &&
        (strpos(json_encode($wp_filter['template_redirect']), 'author') !== false)
    ) {
        $features[] = [
            'feature' => __('template_redirect による author= 遮断コードが存在', 'ghostgate'),
            'tool'    => __('テーマまたはプラグイン', 'ghostgate'),
            'risk'    => 'success',
        ];
        $detected = true;
    }

    // 5. 保護なし警告
    if (!$detected) {
        $features[] = [
            'feature' => __('?author=ID による列挙が保護されていない可能性があります', 'ghostgate'),
            'tool'    => __('WordPress デフォルト', 'ghostgate'),
            'risk'    => 'danger',
        ];
    }

    return $features;
}




//REST APIによるユーザー列挙の遮断設定
function ghostgate_analyze_rest_user_enumeration_protection() {
    $features = [];
    $success_found = false;
    $success_features = [];

    // 1. GhostGate オプションチェック
    if (get_option('ghostgate_block_author_enum') === '1') {
        $success_features[] = [
            'feature'    => __('GhostGate により REST API の不要エンドポイントが制御対象', 'ghostgate'),
            'tool'       => __('GhostGate プラグイン', 'ghostgate'),
            'risk'       => 'success',
            'suggestion' => __('GhostGateが /wp/v2/users を遮断中です', 'ghostgate'),
        ];
        $success_found = true;
    }

    // 2. フィルターによる制御確認
    if (has_filter('rest_endpoints')) {
        $filters = $GLOBALS['wp_filter']['rest_endpoints'] ?? null;
        if (strpos(json_encode($filters), '/wp/v2/users') !== false) {
            $success_features[] = [
                'feature'    => __('rest_endpoints フィルターで /users エンドポイントを制御中の痕跡あり', 'ghostgate'),
                'tool'       => __('functions.php など', 'ghostgate'),
                'risk'       => 'success',
                'suggestion' => __('コードで制御されています', 'ghostgate'),
            ];
            $success_found = true;
        }
    }

    // 3. .htaccess による REST API 制限検出
    $htaccess = ABSPATH . '.htaccess';
    $htaccess_signatures = [];
    $htaccess_has_block = false;
    $content = '';

    if (file_exists($htaccess)) {

        require_once ABSPATH . 'wp-admin/includes/file.php';
        WP_Filesystem(); global $wp_filesystem;
        $ht_path = wp_normalize_path( $htaccess );
        $content = ( $wp_filesystem && $wp_filesystem->exists( $ht_path ) && $wp_filesystem->is_readable( $ht_path ) )
            ? (string) $wp_filesystem->get_contents( $ht_path )
            : '';


        if (preg_match('/wp-json\/wp\/v2\/users/i', $content)) {
            $success_features[] = [
                'feature'    => __('.htaccess に /wp/v2/users を制限する記述あり', 'ghostgate'),
                'tool'       => '.htaccess',
                'risk'       => 'success',
                'suggestion' => __('.htaccess で REST API を制限しています', 'ghostgate'),
            ];
            $success_found = true;
            $htaccess_has_block = true;
        }

        if (preg_match_all('/#.*(cloudsecure|wordfence|wp-cerber|ithemes)/i', $content, $matches)) {
            $htaccess_signatures = array_unique(array_map('strtolower', $matches[1]));
        }
    }

    // 4. 実リクエストで確認
    $response = wp_remote_get(home_url('/wp-json/wp/v2/users'), ['timeout' => 5]);
    $code = wp_remote_retrieve_response_code($response);

    if (in_array($code, [401, 403], true)) {
        $success_features[] = [
            'feature'    => __('REST API の /wp/v2/users は実際にブロックされています', 'ghostgate'),
            'tool'       => __('実リクエスト検証', 'ghostgate'),
            'risk'       => 'success',
            // translators: %d is the HTTP status code returned by the REST API request (e.g., 401 or 403)
            'suggestion' => sprintf(__('HTTPコード %d によりユーザー列挙が防止されています', 'ghostgate'), $code),
        ];
        $success_found = true;
    }

    if ($success_found) {
        return [reset($success_features)];
    }

    // === 以下は success 判定がなかった場合 ===
    $plugin_map = [
        'wp-cerber/wp-cerber.php'                              => 'WP Cerber Security',
        'wordfence/wordfence.php'                              => 'Wordfence',
        'ithemes-security-pro/ithemes-security-pro.php'        => 'iThemes Security',
        'all-in-one-wp-security-and-firewall/wp-security.php' => 'All In One WP Security',
        'cloudsecurewp/cloudsecurewp.php'                      => 'Cloud Secure',
    ];
    $active_plugins = get_option('active_plugins', []);

    foreach ($plugin_map as $slug => $label) {
        $is_active = in_array($slug, $active_plugins, true);
        $plugin_id = explode('/', $slug)[0];

    if ($is_active) {
        $features[] = [
            // translators: %s is the plugin label/name
            'feature'    => sprintf(__('%s が有効化されています', 'ghostgate'), $label),
            'tool'       => $label,
            'risk'       => 'info',
            // translators: %s is the plugin label/name for REST API control setting
            'suggestion' => sprintf(__('%s の REST 制御設定をご確認ください', 'ghostgate'), $label),
        ];
    } elseif (in_array(strtolower($plugin_id), $htaccess_signatures, true)) {
        $features[] = [
            // translators: %s is the plugin label/name whose .htaccess signature remains
            'feature'    => sprintf(__('%s の署名が .htaccess に残存しています（プラグイン無効）', 'ghostgate'), $label),
            'tool'       => $label,
            'risk'       => 'warning',
            'suggestion' => __('.htaccess の残留署名を削除するか、再度有効化をご検討ください', 'ghostgate'),
        ];
    }

    }

    // 6. 実リクエスト結果
    if ($code === 200) {
        $features[] = [
            'feature'    => __('REST API 経由のユーザー一覧エンドポイントが有効化されています', 'ghostgate'),
            'tool'       => '/wp/v2/users',
            'risk'       => 'danger',
            'suggestion' => __('GhostGate や .htaccess による保護を推奨します', 'ghostgate'),
        ];
    } elseif (!in_array($code, [200, 401, 403], true)) {
        $features[] = [
            // translators: %d is the HTTP status code returned from the REST API response (e.g., 500 or other unexpected code)
            'feature'    => sprintf(__('REST API から不明な応答（HTTP %d）を受信', 'ghostgate'), $code),
            'tool'       => __('サーバー応答', 'ghostgate'),
            'risk'       => 'warning',
            'suggestion' => __('REST 応答の状態を確認してください', 'ghostgate'),
        ];
    }

    // 7. 最後の保険：何もない場合
    if (empty($features)) {
        $features[] = [
            'feature'    => __('REST API によるユーザー列挙が保護されていない可能性', 'ghostgate'),
            'tool'       => __('WordPress デフォルト', 'ghostgate'),
            'risk'       => 'danger',
            'suggestion' => __('GhostGate または .htaccess による制御を推奨します', 'ghostgate'),
        ];
    }

    return $features;
}


//REST APIへのアクセス制限
function ghostgate_analyze_rest_rate_limit_status() {
    $features = [];
    $success_found = false;

    // 1. GhostGate の設定を確認
    if (get_option('ghostgate_enable_rest_limit') === '1') {
        $limit    = absint(get_option('ghostgate_rest_limit_count', 20));
        $cooldown = absint(get_option('ghostgate_rest_limit_cooldown', 300));

        $features[] = [
            'feature'    => __('GhostGate による REST API アクセス制限が有効化されています', 'ghostgate'),
            'tool'       => __('GhostGate プラグイン', 'ghostgate'),
            'risk'       => 'success',
            // translators: %1$d is the number of allowed requests per minute, %2$d is the block duration in seconds
            'suggestion' => sprintf(__('1分間に %1$d 回を超えると %2$d 秒間ブロックされます', 'ghostgate'), $limit, $cooldown),
        ];

        $success_found = true;
    }

    // 2. .htaccess に制限記述があるか
    $htaccess = ABSPATH . '.htaccess';
    $htaccess_signatures = [];
    if (file_exists($htaccess)) {

            require_once ABSPATH . 'wp-admin/includes/file.php';
            WP_Filesystem(); global $wp_filesystem;
            $ht_path = wp_normalize_path( $htaccess );
            $content = ( $wp_filesystem && $wp_filesystem->exists( $ht_path ) && $wp_filesystem->is_readable( $ht_path ) )
                ? (string) $wp_filesystem->get_contents( $ht_path )
                : '';


        if (preg_match('/wp-json/i', $content) && preg_match('/deny|403/i', $content)) {
            $features[] = [
                'feature'    => __('.htaccess による REST API 制限の記述を検出', 'ghostgate'),
                'tool'       => '.htaccess',
                'risk'       => 'success',
                'suggestion' => __('リクエスト量や条件に応じて制限されている可能性があります', 'ghostgate'),
            ];
            $success_found = true;
        }

        if (preg_match_all('/#.*(cloudsecure|wordfence|wp-cerber|ithemes)/i', $content, $matches)) {
            $htaccess_signatures = array_unique(array_map('strtolower', $matches[1]));
        }
    }

    // 3. 他のセキュリティプラグインの検出
    $plugin_map = [
        'wp-cerber/wp-cerber.php'                              => 'WP Cerber Security',
        'wordfence/wordfence.php'                              => 'Wordfence',
        'ithemes-security-pro/ithemes-security-pro.php'        => 'iThemes Security',
        'all-in-one-wp-security-and-firewall/wp-security.php' => 'All In One WP Security',
        'cloudsecurewp/cloudsecurewp.php'                      => 'Cloud Secure',
    ];
    $active_plugins = get_option('active_plugins', []);

    foreach ($plugin_map as $slug => $label) {
        $is_active = in_array($slug, $active_plugins, true);
        $plugin_id = explode('/', $slug)[0];

        if ($is_active) {
            $features[] = [
                // translators: %s is the plugin name or label
                'feature'    => sprintf(__('%s が有効化されています（REST制限機能を含む可能性あり）', 'ghostgate'), $label),
                'tool'       => $label,
                'risk'       => 'info',
                // translators: %s is the plugin name or label
                'suggestion' => sprintf(__('%s の設定で REST API の制御状況を確認してください', 'ghostgate'), $label),
            ];
        } elseif (in_array(strtolower($plugin_id), $htaccess_signatures, true)) {
            $features[] = [
                // translators: %s is the plugin name or label
                'feature'    => sprintf(__('%s の署名が .htaccess に残っています（プラグイン無効）', 'ghostgate'), $label),
                'tool'       => $label,
                'risk'       => 'warning',
                'suggestion' => __('.htaccess の記述を確認し、不要であれば削除してください', 'ghostgate'),
            ];
        }

    }

    // 4. 何も対策が検出されなかった場合の警告
    if (!$success_found) {
        $features[] = [
            'feature'    => __('REST API へのアクセス制限が検出されませんでした', 'ghostgate'),
            'tool'       => __('WordPress デフォルト', 'ghostgate'),
            'risk'       => 'danger',
            'suggestion' => __('GhostGate で REST API 制限を有効化することを推奨します', 'ghostgate'),
        ];
    }

    return $features;
}



//xmlrpcの遮断チェック（完全版）
function ghostgate_analyze_xmlrpc_blocking() {
    $features = [];
    $success_found = false;
    $already = [];

    // 1. WordPress フィルター確認
// REVIEW: WordPress core filter name (read-only). Not plugin-defined.
/* phpcs:disable WordPress.NamingConventions.PrefixAllGlobals */
$filter_result = apply_filters( 'xmlrpc_enabled', true );
/* phpcs:enable */



    if ($filter_result === false) {
        $features[] = [
            'feature'    => __('✅ WordPress フィルターで XML-RPC が無効化されています', 'ghostgate'),
            'tool'       => __("add_filter('xmlrpc_enabled', '__return_false')", 'ghostgate'),
            'risk'       => 'success',
            'suggestion' => __('一部のプラグインやテーマ、コードによって XML-RPC が無効化されています（セキュリティ対策の一環として一般的です）', 'ghostgate'),
        ];
        $success_found = true;
        $already['xmlrpc'] = true;
    }

    // 2. GhostGate オプション確認
    $option = (int) get_option('ghostgate_disable_xmlrpc');

    if ($option === 1) {
        $features[] = [
            'feature'    => __('🛠 GhostGate 設定で XML-RPC 無効化が有効です', 'ghostgate'),
            'tool'       => __('GhostGate オプション', 'ghostgate'),
            'risk'       => 'info',
            'suggestion' => __('保存時に .htaccess に自動反映されます', 'ghostgate'),
        ];
    }

    // 3. .htaccess 解析（WP_Filesystem で読み取り）
    $htaccess_path = wp_normalize_path( ABSPATH . '.htaccess' );
    $htaccess_signatures = [];

    require_once ABSPATH . 'wp-admin/includes/file.php';
    WP_Filesystem(); global $wp_filesystem;

    if ( $wp_filesystem && $wp_filesystem->exists( $htaccess_path ) && $wp_filesystem->is_readable( $htaccess_path ) ) {
        $content = (string) $wp_filesystem->get_contents( $htaccess_path );

        $pattern_matched = (
            $content !== '' && (
                preg_match( '/<Files\s+"?xmlrpc\.php"?\s*>/i', $content ) ||
                preg_match( '/xmlrpc\.php.*(Deny from all|Require all denied)/i', $content ) ||
                preg_match( '/RewriteCond.*xmlrpc\.php/i', $content )
            )
        );

        if ( $pattern_matched ) {
            $features[] = [
                'feature'    => __( '✅ .htaccess により XML-RPC アクセスが遮断されています', 'ghostgate' ),
                'tool'       => '.htaccess',
                'risk'       => 'success',
                'suggestion' => __( 'Webサーバーレベルでブロックされています', 'ghostgate' ),
            ];
            $success_found     = true;
            $already['xmlrpc'] = true;
        }

        if ( $content !== '' && preg_match_all( '/#.*?(cloudsecure|wordfence|wp-cerber|ithemes|sucuri)/i', $content, $matches ) ) {
            foreach ( $matches[1] as $sig ) {
                $htaccess_signatures[] = strtolower( trim( $sig ) );
            }
        }

        if ( $content !== '' && preg_match( '/# BEGIN GhostGate XML-RPC Block/i', $content ) ) {
            $features[] = [
                'feature'    => __( '✅ GhostGate により XML-RPC が .htaccess で遮断されています', 'ghostgate' ),
                'tool'       => __( 'GhostGate', 'ghostgate' ),
                'risk'       => 'success',
            ];
            $success_found     = true;
            $already['xmlrpc'] = true;
        }
    }


    // 4. プラグイン署名・状態確認
    $plugin_map = [
        'cloudsecurewp/cloudsecurewp.php'                        => 'Cloud Secure',
        'wordfence/wordfence.php'                                => 'Wordfence',
        'ithemes-security-pro/ithemes-security-pro.php'          => 'iThemes Security',
        'wp-cerber/wp-cerber.php'                                => 'WP Cerber Security',
        'sucuri-scanner/sucuri.php'                              => 'Sucuri Security',
        'all-in-one-wp-security-and-firewall/wp-security.php'   => 'All In One WP Security',
    ];

    $active_plugins = get_option('active_plugins', []);

    foreach ($plugin_map as $plugin_path => $label) {
        $slug = explode('/', $plugin_path)[0];
        $is_active = in_array($plugin_path, $active_plugins, true);
        $has_signature = in_array(strtolower($slug), $htaccess_signatures, true);

        if ($is_active) {
            $features[] = [
                // translators: %s is the plugin name or label
                'feature'    => sprintf(__('🔎 %s プラグインが有効です', 'ghostgate'), $label),
                'tool'       => $label,
                'risk'       => 'info',
                // translators: %s is the plugin name or label
                'suggestion' => sprintf(__('%s 側の設定で XML-RPC 無効化が行われている可能性があります', 'ghostgate'), $label),
            ];
        } elseif ($has_signature) {
            $features[] = [
                // translators: %s is the plugin name or label whose signature remains in .htaccess
                'feature'    => sprintf(__('⚠️ %s の署名が .htaccess に残っています（プラグインは無効）', 'ghostgate'), $label),
                'tool'       => '.htaccess',
                'risk'       => 'warning',
                // translators: %s is the plugin name or label that might still be affecting XML-RPC
                'suggestion' => sprintf(__('%s の影響が残っている可能性があります。署名の削除をご検討ください', 'ghostgate'), $label),
            ];
        }

    }

    // 5. 保護が見つからなかった場合
    if (!$success_found) {
        $features[] = [
            'feature'    => __('⚠️ XML-RPC は現在有効です（無防備）', 'ghostgate'),
            'tool'       => __('WordPress デフォルト', 'ghostgate'),
            'risk'       => 'danger',
            'suggestion' => __('GhostGate または .htaccess による無効化を推奨します', 'ghostgate'),
        ];
    }

    return $features;
}

//jsonの秘匿化
function ghostgate_analyze_json_structure_obfuscation() {
    $features = [];
    $detected = false;

    $enabled = get_option('ghostgate_hide_json_endpoints');
    if ($enabled) {
        $features[] = [
            'feature' => __('wp-json 構造が GhostGate により秘匿化されています', 'ghostgate'),
            'tool'    => __('GhostGate プラグイン', 'ghostgate'),
            'risk'    => 'success',
        ];
        $detected = true;
    }

    if (!$detected) {
        $features[] = [
            'feature' => __('wp-json に全てのエンドポイントが表示されています', 'ghostgate'),
            'tool'    => __('WordPress デフォルト', 'ghostgate'),
            'risk'    => 'warning',
        ];
    }

    return $features;
}


//.htaccessの詳細調査
function ghostgate_analyze_htaccess_security() {

    $htaccess_path = wp_normalize_path( ABSPATH . '.htaccess' );

    require_once ABSPATH . 'wp-admin/includes/file.php';
    WP_Filesystem(); global $wp_filesystem;

    if ( ! ( $wp_filesystem && $wp_filesystem->exists( $htaccess_path ) && $wp_filesystem->is_readable( $htaccess_path ) ) ) {
        return [];
    }

    $content = (string) $wp_filesystem->get_contents( $htaccess_path );
    if ( $content === '' ) { return []; }

    $features = [];
    $already = [];

    $plugin_signatures = [
        'cloudsecure'     => 'cloudsecurewp/cloudsecurewp.php',
        'ithemes'         => 'ithemes-security-pro/ithemes-security-pro.php',
        'wordfence'       => 'wordfence/wordfence.php',
        'sucuri'          => 'sucuri-scanner/sucuri.php',
        'mod_security'    => null,
        'ghostgate'       => 'ghostgate/ghostgate.php'
    ];

    $active_plugins = get_option('active_plugins', []);

    // 🔹 GhostGate 署名チェック
    if (preg_match('/# BEGIN GhostGate XML-RPC Block/i', $content)) {
        $features[] = [
            'feature' => __('GhostGate により XML-RPC がブロックされています', 'ghostgate'),
            'tool'    => __('GhostGate プラグイン', 'ghostgate'),
            'risk'    => 'safe',
        ];
        $already['xmlrpc'] = true;
        $already['sig_ghostgate'] = true;
    }

    // 🔹 他プラグイン署名検出
    if (preg_match_all('/#.*(cloudsecure|ithemes|wordfence|sucuri|mod[_\s]?security)/i', $content, $matches)) {
        foreach ($matches[1] as $match) {
            $slug = strtolower(trim($match));
            $slug = str_replace(['_', ' '], '', $slug);
            if (isset($already["sig_{$slug}"])) continue;
            $already["sig_{$slug}"] = true;

            $label = ucfirst($slug);
            $plugin_path = $plugin_signatures[$slug] ?? null;
            $active = $plugin_path ? in_array($plugin_path, $active_plugins, true) : null;

            $features[] = [
                // translators: %s is the name or label of the plugin/tool whose signature was found in .htaccess
                'feature'    => sprintf(__('.htaccess に %s の署名を検出', 'ghostgate'), $label),
                'tool'       => $label,
                'risk'       => $active === false ? 'warning' : 'safe',
                // translators: %s is the name or label of the plugin/tool whose signature remains
                'suggestion' => $active === false ? sprintf(__('%s の署名が残っています。不要であれば削除を検討してください。', 'ghostgate'), $label) : null,
            ];

        }
    }


    // 🔹 XML-RPC 一般検出（GhostGateが処理済みでなければ）
    if (!isset($already['xmlrpc'])) {
        $already['xmlrpc'] = true;
        if (
            preg_match('/<Files\s+"?xmlrpc\.php"?\s*>/i', $content) ||
            preg_match('/xmlrpc\.php.*(Deny from all|Require all denied)/i', $content) ||
            preg_match('/RewriteCond.*xmlrpc\.php/i', $content)
        ) {
            $features[] = [
                'feature' => __('XML-RPC へのアクセスブロック', 'ghostgate'),
                'tool'    => __('.htaccess', 'ghostgate'),
                'risk'    => 'safe',
            ];
        } else {
            $features[] = [
                'feature'    => __('XML-RPC へのブロックが未検出', 'ghostgate'),
                'tool'       => __('未対策', 'ghostgate'),
                'risk'       => 'danger',
                'suggestion' => __('xmlrpc.php へのアクセスを制限してください', 'ghostgate'),
            ];
        }
    }

    // 🔹 wp-config.php 保護
    if (!isset($already['wpconfig'])) {
        $already['wpconfig'] = true;
        if (
            preg_match('/<Files\s+"?wp-config\.php"?\s*>/i', $content) ||
            preg_match('/wp-config\.php.*(Deny from all|Require all denied)/i', $content)
        ) {
            $features[] = [
                'feature' => __('wp-config.php の外部アクセス遮断', 'ghostgate'),
                'tool'    => __('.htaccess', 'ghostgate'),
                'risk'    => 'safe',
            ];
        } else {
            $features[] = [
                'feature'    => __('wp-config.php への保護が未検出', 'ghostgate'),
                'tool'       => __('未対策', 'ghostgate'),
                'risk'       => 'danger',
                'suggestion' => __('.htaccess で wp-config.php を保護してください', 'ghostgate'),
            ];
        }
    }


// 🔹 wp-admin Rewrite制限
    if (!isset($already['wpadmin']) &&
        (preg_match('/RewriteCond.*REQUEST_URI.*wp-admin/i', $content) ||
        preg_match('/RewriteRule.*wp-admin/i', $content))) {
        $already['wpadmin'] = true;
        $features[] = [
            'feature' => __('wp-admin へのアクセス条件制限', 'ghostgate'),
            'tool'    => __('カスタム制御', 'ghostgate'),
            'risk'    => 'safe',
        ];
    }

    // 🔹 ディレクトリリスト制限
    if (!isset($already['indexes'])) {
        $already['indexes'] = true;
        if (preg_match('/Options\s+-Indexes/i', $content)) {
            $features[] = [
                'feature' => __('ディレクトリリストの無効化', 'ghostgate'),
                'tool'    => __('.htaccess', 'ghostgate'),
                'risk'    => 'safe',
            ];
        } else {
            $features[] = [
                'feature'    => __('ディレクトリリストの制限が未検出', 'ghostgate'),
                'tool'       => __('未対策', 'ghostgate'),
                'risk'       => 'warning',
                'suggestion' => __('Options -Indexes を設定し、リスト表示を防止してください', 'ghostgate'),
            ];
        }
    }

    // 🔹 IP制限
    if (!isset($already['ip']) &&
        preg_match('/(Allow from\s+[0-9\.]+|Deny from\s+[0-9\.]+|Require ip\s+[0-9\.]+)/i', $content)) {
        $already['ip'] = true;
        $features[] = [
            'feature' => __('IPアドレスによるアクセス制限', 'ghostgate'),
            'tool'    => __('.htaccess', 'ghostgate'),
            'risk'    => 'safe',
        ];
    }


    // 🔹 wp-content, uploads, includes（ディレクトリ制限）
    foreach (['wp-content' => 'wpcontent', 'wp-content/uploads' => 'uploads', 'wp-includes' => 'includes'] as $dir => $key) {
        if (!isset($already[$key])) {
            $pattern = "#{$dir}.*(deny from|<Files|Require all denied)#i";
            if (preg_match($pattern, $content)) {
                $already[$key] = true;
                $features[] = [
                    // translators: %s is the directory name or path being restricted via .htaccess
                    'feature' => sprintf(__('%s のアクセス制限', 'ghostgate'), $dir),
                    'tool'    => __('.htaccess', 'ghostgate'),
                    'risk'    => 'safe',
                ];

            }
        }
    }

    // 🔹 基本セキュリティヘッダー
    if (!isset($already['headers_basic'])) {
        if (preg_match('/Header\s+set\s+(X-Frame-Options|X-Content-Type-Options|X-XSS-Protection)/i', $content)) {
            $features[] = [
                'feature' => __('基本セキュリティヘッダー設定済み', 'ghostgate'),
                'tool'    => __('.htaccess', 'ghostgate'),
                'risk'    => 'safe',
            ];
        } else {
            $features[] = [
                'feature'    => __('基本的なセキュリティヘッダーが未設定', 'ghostgate'),
                'tool'       => __('未対策', 'ghostgate'),
                'risk'       => 'warning',
                'suggestion' => __('X-Frame-Options などのセキュリティヘッダーを設定してください', 'ghostgate'),
            ];
        }
        $already['headers_basic'] = true;
    }

    // 🔹 拡張ヘッダー
    if (!isset($already['headers_ext']) &&
        preg_match('/Header\s+set\s+(Strict-Transport-Security|Referrer-Policy|Permissions-Policy|Content-Security-Policy)/i', $content)) {
        $already['headers_ext'] = true;
        $features[] = [
            'feature' => __('追加のセキュリティヘッダーが設定されています', 'ghostgate'),
            'tool'    => __('.htaccess', 'ghostgate'),
            'risk'    => 'safe',
        ];
    }


    // 🔹 mod_security
    if (!isset($already['modsec']) &&
        preg_match('/<IfModule\s+mod_security/i', $content)) {
        $already['modsec'] = true;
        $features[] = [
            'feature' => __('mod_security によるWAF制御', 'ghostgate'),
            'tool'    => __('ModSecurity', 'ghostgate'),
            'risk'    => 'safe',
        ];
    }

    // 🔹 User-Agent 制限
    if (!isset($already['useragent']) &&
        preg_match('/(SetEnvIf|RewriteCond).*User-Agent/i', $content)) {
        $already['useragent'] = true;
        $features[] = [
            'feature' => __('User-Agent によるアクセス制御', 'ghostgate'),
            'tool'    => __('.htaccess', 'ghostgate'),
            'risk'    => 'safe',
        ];
    }

    // 🔹 PHP実行ブロック
    if (!isset($already['phpdeny']) &&
        preg_match('/<FilesMatch\s+"?\.php/i', $content)) {
        $already['phpdeny'] = true;
        $features[] = [
            'feature' => __('.phpファイルの実行ブロック', 'ghostgate'),
            'tool'    => __('.htaccess', 'ghostgate'),
            'risk'    => 'safe',
        ];
    }

    // 🔹 ErrorDocument
    if (!isset($already['errorpage']) &&
        preg_match('/ErrorDocument\s+(403|404)/i', $content)) {
        $already['errorpage'] = true;
        $features[] = [
            'feature' => __('ErrorDocument によるエラーハンドリング制御', 'ghostgate'),
            'tool'    => __('.htaccess', 'ghostgate'),
            'risk'    => 'safe',
        ];
    }


    return $features;
}



//wp-config.phpの詳細調査
function ghostgate_analyze_wpconfig_security() {
    $config_path = ABSPATH . 'wp-config.php';
    if (!file_exists($config_path)) {
        return [];
    }

    require_once ABSPATH . 'wp-admin/includes/file.php';
    WP_Filesystem(); global $wp_filesystem;
    $cfg_path = wp_normalize_path( $config_path );
    $content = ( $wp_filesystem && $wp_filesystem->exists( $cfg_path ) && $wp_filesystem->is_readable( $cfg_path ) )
        ? (string) $wp_filesystem->get_contents( $cfg_path )
        : '';


    $features = [];

    // 🔹 テーマ・プラグイン編集無効化
    if (preg_match('/define\s*\(\s*[\'"]DISALLOW_FILE_EDIT[\'"]\s*,\s*true\s*\)/i', $content)) {
        $features[] = [
            'feature' => __('テーマ/プラグイン編集の無効化', 'ghostgate'),
            'tool'    => __('iThemes Security, Wordfence, All In One WP Security', 'ghostgate'),
            'risk'    => 'safe',
        ];
    } else {
        $features[] = [
            'feature'    => __('テーマ/プラグイン編集が許可されています', 'ghostgate'),
            'tool'       => __('未対策', 'ghostgate'),
            'risk'       => 'warning',
            'suggestion' => __('DISALLOW_FILE_EDIT を true にして編集機能を無効化してください', 'ghostgate'),
        ];
    }

    // 🔹 インストール・更新の無効化
    if (preg_match('/define\s*\(\s*[\'"]DISALLOW_FILE_MODS[\'"]\s*,\s*true\s*\)/i', $content)) {
        $features[] = [
            'feature' => __('インストール/更新操作の全面禁止', 'ghostgate'),
            'tool'    => __('手動設定 or セキュリティ系プラグイン', 'ghostgate'),
            'risk'    => 'safe',
        ];
    }

    // 🔹 SSL 管理画面強制
    if (preg_match('/define\s*\(\s*[\'"]FORCE_SSL_ADMIN[\'"]\s*,\s*true\s*\)/i', $content)) {
        $features[] = [
            'feature' => __('管理画面へのSSL接続を強制', 'ghostgate'),
            'tool'    => __('WordPress標準機能 or セキュリティプラグイン', 'ghostgate'),
            'risk'    => 'safe',
        ];
    } else {
        $features[] = [
            'feature'    => __('管理画面へのSSL強制が未設定', 'ghostgate'),
            'tool'       => __('未対策', 'ghostgate'),
            'risk'       => 'danger',
            'suggestion' => __('FORCE_SSL_ADMIN を true に設定してください', 'ghostgate'),
        ];
    }

    // 🔹 WP_DEBUG
    if (preg_match('/define\s*\(\s*[\'"]WP_DEBUG[\'"]\s*,\s*true\s*\)/i', $content)) {
        $features[] = [
            'feature'    => __('WP_DEBUGが有効（本番環境では非推奨）', 'ghostgate'),
            'tool'       => __('開発者向け一時的設定', 'ghostgate'),
            'risk'       => 'danger',
            'suggestion' => __('本番環境では WP_DEBUG を false にしてください', 'ghostgate'),
        ];
    }

    // 🔹 WP_AUTO_UPDATE_CORE
    if (preg_match('/define\s*\(\s*[\'"]WP_AUTO_UPDATE_CORE[\'"]\s*,\s*(false|true|[\'"]major[\'"])\s*\)/i', $content, $match)) {
        $value = strtolower(trim($match[1], "'\""));
        $desc = match ($value) {
            'true'   => __('すべてのコア更新を許可', 'ghostgate'),
            'false'  => __('コア自動更新を完全に無効化', 'ghostgate'),
            'major'  => __('メジャーアップデートのみ許可', 'ghostgate'),
            default  => __('自動更新のカスタム制御', 'ghostgate'),
        };
        $risk = ($value === 'false') ? 'warning' : 'safe';
        $suggestion = ($value === 'false') ? __('重要な更新が適用されない可能性があります', 'ghostgate') : null;

        $features[] = [
            // translators: %s is a description of the core auto-update configuration status
            'feature'    => sprintf(__('コアの自動更新設定: %s', 'ghostgate'), $desc),
            'tool'       => __('wp-config手動 or セキュリティ系プラグイン', 'ghostgate'),
            'risk'       => $risk,
            'suggestion' => $suggestion,
        ];

    }

    // 🔹 セキュリティソルト
    if (preg_match_all('/define\s*\(\s*\'(AUTH_KEY|SECURE_AUTH_KEY|LOGGED_IN_KEY|NONCE_KEY|AUTH_SALT|SECURE_AUTH_SALT|LOGGED_IN_SALT|NONCE_SALT)\'\s*,\s*\'[^\']+\'\s*\)/i', $content, $matches)) {
        if (count($matches[0]) >= 8) {
            $features[] = [
                'feature' => __('セキュリティソルト（キー類）が定義済み', 'ghostgate'),
                'tool'    => __('WordPress標準（強く推奨）', 'ghostgate'),
                'risk'    => 'safe',
            ];
        } else {
            $features[] = [
                'feature'    => __('セキュリティソルトが一部未定義', 'ghostgate'),
                'tool'       => __('再生成を推奨', 'ghostgate'),
                'risk'       => 'warning',
                'suggestion' => __('全てのソルトキーを wp-config.php に追加してください', 'ghostgate'),
            ];
        }
    } else {
        $features[] = [
            'feature'    => __('セキュリティソルトが見つかりません', 'ghostgate'),
            'tool'       => __('WordPress公式の生成ツールを利用推奨', 'ghostgate'),
            'risk'       => 'danger',
            'suggestion' => __('https://api.wordpress.org/secret-key/1.1/salt/ を使用してキーを定義してください', 'ghostgate'),
        ];
    }

    // 🔹 DBクエリログ
    if (preg_match('/define\s*\(\s*[\'"]SAVEQUERIES[\'"]\s*,\s*true\s*\)/i', $content)) {
        $features[] = [
            'feature'    => __('DBクエリログ（SAVEQUERIES）が有効', 'ghostgate'),
            'tool'       => __('開発用の一時設定', 'ghostgate'),
            'risk'       => 'warning',
            'suggestion' => __('本番環境では SAVEQUERIES を false にしてください', 'ghostgate'),
        ];
    }

    // 🔹 WP_ENV
    if (preg_match('/define\s*\(\s*[\'"]WP_ENV[\'"]\s*,\s*[\'"](development|production|staging)[\'"]\s*\)/i', $content, $m)) {
        $features[] = [
            // translators: %s is the current environment value (e.g. development, production, or staging)
            'feature' => sprintf(__('環境識別フラグが定義済み: %s', 'ghostgate'), $m[1]),
            'tool'    => __('開発・本番環境の切り替え用途', 'ghostgate'),
            'risk'    => 'safe',
        ];
    }

    return $features;
}



//.user.iniの調査関数
function ghostgate_analyze_userini_security() {
    $userini_path = ABSPATH . '.user.ini';
    if (!file_exists($userini_path)) {
        return [];
    }

    require_once ABSPATH . 'wp-admin/includes/file.php';
    WP_Filesystem(); global $wp_filesystem;
    $userini_path_n = wp_normalize_path( $userini_path );
    $content = ( $wp_filesystem && $wp_filesystem->exists( $userini_path_n ) && $wp_filesystem->is_readable( $userini_path_n ) )
        ? (string) $wp_filesystem->get_contents( $userini_path_n )
        : '';

    $features = [];

    // 🔹 disable_functions の確認
    if (preg_match('/^\s*disable_functions\s*=\s*(.+)$/mi', $content, $match)) {
        $disabled = trim($match[1]);
        $features[] = [
            // translators: %s is the list of disabled PHP functions (e.g. exec, shell_exec)
            'feature' => sprintf(__('特定のPHP関数が無効化されています（disable_functions）: %s', 'ghostgate'), $disabled),
            'tool'    => __('セキュリティ強化（exec, shell_exec など）', 'ghostgate'),
            'risk'    => 'safe',
        ];
    } else {
        $features[] = [
            'feature'    => __('disable_functions が設定されていません', 'ghostgate'),
            'tool'       => __('セキュリティ対策不足', 'ghostgate'),
            'risk'       => 'warning',
            'suggestion' => __('exec, shell_exec などを disable_functions で無効化してください', 'ghostgate'),
        ];
    }

    // 🔹 expose_php の無効化
    if (preg_match('/^\s*expose_php\s*=\s*off\s*$/mi', $content)) {
        $features[] = [
            'feature' => __('PHPバージョン情報の非公開（expose_php = Off）', 'ghostgate'),
            'tool'    => __('セキュリティ強化設定', 'ghostgate'),
            'risk'    => 'safe',
        ];
    } else {
        $features[] = [
            'feature'    => __('PHPバージョン情報が公開されています（expose_php）', 'ghostgate'),
            'tool'       => __('セキュリティ設定不備', 'ghostgate'),
            'risk'       => 'warning',
            'suggestion' => __('expose_php = Off に設定してください', 'ghostgate'),
        ];
    }

    // 🔹 allow_url_fopen の無効化
    if (preg_match('/^\s*allow_url_fopen\s*=\s*off\s*$/mi', $content)) {
        $features[] = [
            'feature' => __('URLファイルアクセスの無効化（allow_url_fopen = Off）', 'ghostgate'),
            'tool'    => __('外部リソース読み込みの制限', 'ghostgate'),
            'risk'    => 'safe',
        ];
    } else {
        $features[] = [
            'feature'    => __('allow_url_fopen が有効になっています', 'ghostgate'),
            'tool'       => __('外部リソース読み込みリスクあり', 'ghostgate'),
            'risk'       => 'warning',
            'suggestion' => __('allow_url_fopen = Off にして外部ファイル実行リスクを抑えてください', 'ghostgate'),
        ];
    }

    // 🔹 open_basedir の制限
    if (preg_match('/^\s*open_basedir\s*=\s*(.+)$/mi', $content, $match)) {
        $features[] = [
            // translators: %s is the open_basedir path or list of directories that file access is restricted to
            'feature' => sprintf(__('ファイルアクセス制限（open_basedir 設定あり）: %s', 'ghostgate'), trim($match[1])),
            'tool'    => __('ファイルアクセス制限強化', 'ghostgate'),
            'risk'    => 'safe',
        ];
    } else {
        $features[] = [
            'feature'    => __('open_basedir が未設定', 'ghostgate'),
            'tool'       => __('任意ファイルアクセスの危険性あり', 'ghostgate'),
            'risk'       => 'warning',
            'suggestion' => __('open_basedir に許可ディレクトリを指定してください', 'ghostgate'),
        ];
    }

    // 🔹 display_errors の無効化
    if (preg_match('/^\s*display_errors\s*=\s*off\s*$/mi', $content)) {
        $features[] = [
            'feature' => __('エラーメッセージの非表示（display_errors = Off）', 'ghostgate'),
            'tool'    => __('本番環境向け設定', 'ghostgate'),
            'risk'    => 'safe',
        ];
    } else {
        $features[] = [
            'feature'    => __('display_errors が有効です（エラー内容が漏洩する可能性あり）', 'ghostgate'),
            'tool'       => __('開発環境用設定が残存', 'ghostgate'),
            'risk'       => 'danger',
            'suggestion' => __('display_errors = Off に変更し、エラー表示を抑えてください', 'ghostgate'),
        ];
    }

    return $features;
}



//php.iniの詳細調査
function ghostgate_analyze_php_ini_security() {
    $features = [];
    $php_ini_path = php_ini_loaded_file();

    if (!$php_ini_path || !is_readable($php_ini_path)) {
        return $features;
    }

    $ini = parse_ini_file($php_ini_path, true, INI_SCANNER_RAW);
    if (!$ini) return $features;

    // 🔹 expose_php
    if (isset($ini['expose_php']) && strtolower($ini['expose_php']) === 'off') {
        $features[] = [
            'feature' => __('PHPバージョン情報の非公開（expose_php = Off）', 'ghostgate'),
            'tool'    => __('php.ini セキュリティ強化', 'ghostgate'),
            'risk'    => 'safe',
        ];
    } else {
        $features[] = [
            'feature'    => __('PHPバージョン情報が公開状態（expose_php = On）', 'ghostgate'),
            'tool'       => __('セキュリティ設定不足', 'ghostgate'),
            'risk'       => 'warning',
            'suggestion' => __('expose_php = Off に設定してください', 'ghostgate'),
        ];
    }

    // 🔹 display_errors
    if (isset($ini['display_errors']) && strtolower($ini['display_errors']) === 'off') {
        $features[] = [
            'feature' => __('エラーメッセージの非表示（display_errors = Off）', 'ghostgate'),
            'tool'    => __('開発 → 本番移行済み設定', 'ghostgate'),
            'risk'    => 'safe',
        ];
    } else {
        $features[] = [
            'feature'    => __('display_errors が有効（エラー情報が漏洩する可能性）', 'ghostgate'),
            'tool'       => __('開発環境設定のまま運用中', 'ghostgate'),
            'risk'       => 'danger',
            'suggestion' => __('display_errors = Off に変更してください', 'ghostgate'),
        ];
    }

    // 🔹 disable_functions
    if (!empty($ini['disable_functions'])) {
        $features[] = [
            // translators: %s is a list of dangerous PHP functions disabled via disable_functions (e.g. exec, system, shell_exec)
            'feature' => sprintf(__('危険な関数が無効化（disable_functions）: %s', 'ghostgate'), $ini['disable_functions']),
            'tool'    => __('ホスティング or 管理者手動', 'ghostgate'),
            'risk'    => 'safe',
        ];
    } else {
        $features[] = [
            'feature'    => __('disable_functions が未設定', 'ghostgate'),
            'tool'       => __('セキュリティ設定不足', 'ghostgate'),
            'risk'       => 'warning',
            'suggestion' => __('exec, shell_exec などの危険関数を無効化してください', 'ghostgate'),
        ];
    }

    // 🔹 open_basedir
    if (!empty($ini['open_basedir'])) {
        $features[] = [
            'feature' => __('open_basedir によるアクセス制限が設定されている', 'ghostgate'),
            'tool'    => __('ホスティング or 高度設定', 'ghostgate'),
            'risk'    => 'safe',
        ];
    } else {
        $features[] = [
            'feature'    => __('open_basedir が未設定（ファイル制限なし）', 'ghostgate'),
            'tool'       => __('ディレクトリ横断アクセスリスクあり', 'ghostgate'),
            'risk'       => 'warning',
            'suggestion' => __('open_basedir で許可ディレクトリを制限してください', 'ghostgate'),
        ];
    }

    // 🔹 allow_url_fopen
    if (isset($ini['allow_url_fopen']) && strtolower($ini['allow_url_fopen']) === 'off') {
        $features[] = [
            'feature' => __('外部URLからのファイル取得を無効（allow_url_fopen = Off）', 'ghostgate'),
            'tool'    => __('php.ini 保護設定', 'ghostgate'),
            'risk'    => 'safe',
        ];
    } else {
        $features[] = [
            'feature'    => __('allow_url_fopen が有効（外部リソース取得可能）', 'ghostgate'),
            'tool'       => __('任意コード実行のリスクあり', 'ghostgate'),
            'risk'       => 'warning',
            'suggestion' => __('allow_url_fopen = Off に設定してください', 'ghostgate'),
        ];
    }

    // 🔹 session.cookie_httponly
    if (!empty($ini['session.cookie_httponly'])) {
        $features[] = [
            'feature' => __('セッションCookieに HttpOnly 属性を付加', 'ghostgate'),
            'tool'    => __('セッション保護', 'ghostgate'),
            'risk'    => 'safe',
        ];
    } else {
        $features[] = [
            'feature'    => __('session.cookie_httponly が無効', 'ghostgate'),
            'tool'       => __('XSS による Cookie 盗難のリスクあり', 'ghostgate'),
            'risk'       => 'warning',
            'suggestion' => __('session.cookie_httponly = 1 を追加してください', 'ghostgate'),
        ];
    }

    // 🔹 session.cookie_secure
    if (!empty($ini['session.cookie_secure'])) {
        $features[] = [
            'feature' => __('セッションCookieに Secure 属性を付加（HTTPS専用）', 'ghostgate'),
            'tool'    => __('SSL運用時の保護', 'ghostgate'),
            'risk'    => 'safe',
        ];
    } else {
        $features[] = [
            'feature'    => __('session.cookie_secure が無効', 'ghostgate'),
            'tool'       => __('HTTPS未使用時のセッション漏洩リスク', 'ghostgate'),
            'risk'       => 'warning',
            'suggestion' => __('session.cookie_secure = 1 を追加してください', 'ghostgate'),
        ];
    }

    // 🔹 memory_limit の極端な設定チェック（64MB未満）
    if (isset($ini['memory_limit'])) {
        $limit = strtolower($ini['memory_limit']);
        if ($limit !== '-1') {
            preg_match('/(\d+)/', $limit, $matches);
            $numeric = isset($matches[1]) ? (int)$matches[1] : 0;
            if ($numeric > 0 && $numeric < 64) {
                $features[] = [
                    // translators: %s is the current memory_limit value from PHP configuration (e.g. 32M, 40M)
                    'feature'    => sprintf(__('memory_limit が小さすぎます（%s）', 'ghostgate'), $limit),
                    'tool'       => __('安定動作に影響の可能性あり', 'ghostgate'),
                    'risk'       => 'warning',
                    'suggestion' => __('64M 以上の設定を推奨します', 'ghostgate'),
                ];
            }
        }
    }

    return $features;
}




//.accessの詳細調査
function ghostgate_analyze_dot_access_security() {
    $path = ABSPATH . '.access';
    if (!file_exists($path) || !is_readable($path)) {
        return [];
    }

    require_once ABSPATH . 'wp-admin/includes/file.php';
    WP_Filesystem(); global $wp_filesystem;
    $path_n = wp_normalize_path( $path );
    $content = ( $wp_filesystem && $wp_filesystem->exists( $path_n ) && $wp_filesystem->is_readable( $path_n ) )
        ? (string) $wp_filesystem->get_contents( $path_n )
        : '';


    $features = [];

    // 🔹 セキュリティ関連コメント
    if (preg_match('/#.*(security|protect|firewall|cloudflare|deny)/i', $content)) {
        $features[] = [
            'feature' => __('セキュリティ関連コメントの記述あり', 'ghostgate'),
            'tool'    => __('Cloudflare や 手動設定の可能性', 'ghostgate'),
            'risk'    => 'safe',
        ];
    }

    // 🔹 Deny from all / Require all denied
    if (
        preg_match('/Deny\s+from\s+all/i', $content) ||
        preg_match('/Require\s+all\s+denied/i', $content)
    ) {
        $features[] = [
            'feature' => __('全アクセス拒否設定（Deny from all）', 'ghostgate'),
            'tool'    => __('WAF的なディレクトリ制限', 'ghostgate'),
            'risk'    => 'safe',
        ];
    } else {
        $features[] = [
            'feature'    => __('全アクセス拒否設定が存在しない', 'ghostgate'),
            'tool'       => __('公開ディレクトリでの制限漏れの可能性', 'ghostgate'),
            'risk'       => 'warning',
            'suggestion' => __('.access に「Deny from all」や「Require all denied」を設定してください', 'ghostgate'),
        ];
    }

    // 🔹 Allow/Deny/Require IP
    if (preg_match('/(Allow from|Deny from|Require ip)/i', $content)) {
        $features[] = [
            'feature' => __('IP制限によるアクセス制御あり', 'ghostgate'),
            'tool'    => __('Apacheベースの制限', 'ghostgate'),
            'risk'    => 'safe',
        ];
    } else {
        $features[] = [
            'feature'    => __('IP制限の記述なし', 'ghostgate'),
            'tool'       => __('特定のアクセス元制限が未設定', 'ghostgate'),
            'risk'       => 'warning',
            'suggestion' => __('「Allow from」や「Require ip」などで特定IPの制限を推奨', 'ghostgate'),
        ];
    }

    // 🔹 Referer 制限
    if (preg_match('/RewriteCond\s+%{HTTP_REFERER}/i', $content)) {
        $features[] = [
            'feature' => __('Referer を利用したアクセス制御あり', 'ghostgate'),
            'tool'    => __('画像保護や直リンク防止用途', 'ghostgate'),
            'risk'    => 'safe',
        ];
    } else {
        $features[] = [
            'feature'    => __('Referer制御が未設定', 'ghostgate'),
            'tool'       => __('画像直リンクやリソース盗用リスクあり', 'ghostgate'),
            'risk'       => 'warning',
            'suggestion' => __('「RewriteCond %{HTTP_REFERER}」でReferer制限を加えることを推奨', 'ghostgate'),
        ];
    }

    return $features;
}


//.envの詳細調査
function ghostgate_analyze_env_security() {
    $env_path = ABSPATH . '.env';
    if (!file_exists($env_path) || !is_readable($env_path)) {
        return [];
    }

    require_once ABSPATH . 'wp-admin/includes/file.php';
    WP_Filesystem(); global $wp_filesystem;
    $env_path_n = wp_normalize_path( $env_path );
    $content = ( $wp_filesystem && $wp_filesystem->exists( $env_path_n ) && $wp_filesystem->is_readable( $env_path_n ) )
        ? (string) $wp_filesystem->get_contents( $env_path_n )
        : '';


    $features = [];

    // 🔹 DB接続情報
    if (preg_match('/DB_PASSWORD\s*=\s*/i', $content) || preg_match('/DB_HOST\s*=\s*/i', $content)) {
        $features[] = [
            'feature'    => __('データベース接続情報の定義あり', 'ghostgate'),
            'tool'       => __('.env による設定（Laravel, Bedrock 等）', 'ghostgate'),
            'risk'       => 'warning',
            'suggestion' => __('.env ファイルのパーミッションを 600 に設定し、Gitに含めないでください', 'ghostgate'),
        ];
    }

    // 🔹 APP_KEY / SECRET_KEY など
    if (preg_match('/(APP_KEY|SECRET_KEY|JWT_SECRET)\s*=\s*/i', $content)) {
        $features[] = [
            'feature'    => __('アプリケーションの秘密鍵が定義されています', 'ghostgate'),
            'tool'       => __('Laravel系 or JWT・認証システムの設定', 'ghostgate'),
            'risk'       => 'warning',
            'suggestion' => __('機密情報が漏洩しないよう、Git除外とファイル権限を確認', 'ghostgate'),
        ];
    }

    // 🔹 デバッグモード有効
    if (preg_match('/APP_DEBUG\s*=\s*true/i', $content)) {
        $features[] = [
            'feature'    => __('デバッグモードが有効（本番では非推奨）', 'ghostgate'),
            'tool'       => __('Laravel や Symfony等のフレームワーク設定', 'ghostgate'),
            'risk'       => 'danger',
            'suggestion' => __('本番環境では APP_DEBUG=false に設定してください', 'ghostgate'),
        ];
    }

    // 🔹 HTTPS構成
    if (preg_match('/(FORCE_SSL|APP_URL=https:\\/\\/)/i', $content)) {
        $features[] = [
            'feature' => __('HTTPS強制または安全なURL定義あり', 'ghostgate'),
            'tool'    => __('アプリケーション構成のセキュリティ強化', 'ghostgate'),
            'risk'    => 'safe',
        ];
    } else {
        $features[] = [
            'feature'    => __('HTTPS強制設定が見つかりません', 'ghostgate'),
            'tool'       => __('SSL運用推奨', 'ghostgate'),
            'risk'       => 'warning',
            'suggestion' => __('APP_URL を https:// で設定、または FORCE_SSL=true を検討', 'ghostgate'),
        ];
    }

    // 🔹 メール認証情報
    if (preg_match('/MAIL_(USERNAME|PASSWORD)/i', $content)) {
        $features[] = [
            'feature'    => __('メールサーバー認証情報が含まれています', 'ghostgate'),
            'tool'       => __('SMTPやメール通知用設定', 'ghostgate'),
            'risk'       => 'warning',
            'suggestion' => __('外部送信を伴う設定は .env の保護を徹底してください', 'ghostgate'),
        ];
    }

    // 🔹 AWSやCloud接続
    if (preg_match('/(AWS_ACCESS_KEY_ID|AWS_SECRET_ACCESS_KEY)/i', $content)) {
        $features[] = [
            'feature'    => __('クラウドストレージ（AWS等）認証情報が含まれています', 'ghostgate'),
            'tool'       => __('S3・外部API接続などの環境設定', 'ghostgate'),
            'risk'       => 'danger',
            'suggestion' => __('これらのキーが漏れると第三者による課金や操作が可能です。公開厳禁。', 'ghostgate'),
        ];
    }

    // 🔹 CAPTCHAキー
    if (preg_match('/(RECAPTCHA_SITE_KEY|RECAPTCHA_SECRET_KEY)/i', $content)) {
        $features[] = [
            'feature'    => __('reCAPTCHA関連のキーが含まれています', 'ghostgate'),
            'tool'       => __('スパム対策やフォーム保護', 'ghostgate'),
            'risk'       => 'warning',
            'suggestion' => __('SECRET_KEY は漏洩するとバイパスされる可能性あり。管理を厳格に。', 'ghostgate'),
        ];
    }

    return $features;
}



//.htpasswdの詳細調査
function ghostgate_analyze_htpasswd_security() {
    $path = ABSPATH . '.htpasswd';
    $features = [];

    if (!file_exists($path)) return [];

    WP_Filesystem();
    global $wp_filesystem;
    $content = $wp_filesystem->exists( $path )
        ? (string) $wp_filesystem->get_contents( $path )
        : '';


    $lines = array_filter(explode("\n", $content));

    if (!empty($lines)) {
        $features[] = [
            // translators: %d is the number of user entries defined for Apache Basic authentication
            'feature'    => sprintf(__('Basic認証のユーザー定義あり（%d件）', 'ghostgate'), count($lines)),
            'tool'       => __('Apache Basic認証 or ディレクトリ制限', 'ghostgate'),
            'risk'       => 'safe',
            'suggestion' => __('.htaccess 側と連携して保護ディレクトリを確認してください', 'ghostgate'),
        ];
    }

    return $features;
}



//robots.txtの詳細調査
function ghostgate_analyze_robots_txt_security() {
    $path = ABSPATH . 'robots.txt';
    $features = [];

    if (!file_exists($path)) return [];

    WP_Filesystem();
    global $wp_filesystem;
    $content = $wp_filesystem->exists( $path )
        ? (string) $wp_filesystem->get_contents( $path )
        : '';


    // 🔹 wp-admin ブロック
    if (preg_match('/Disallow:\s*\/wp-admin/i', $content)) {
        $features[] = [
            'feature'    => __('管理画面（/wp-admin）へのクローラブロック', 'ghostgate'),
            'tool'       => __('SEOとセキュリティ目的のrobots.txt設定', 'ghostgate'),
            'risk'       => 'safe',
            'suggestion' => __('このブロックはSEOやBotアクセス制限に有効です', 'ghostgate'),
        ];
    }

    // 🔹 システムディレクトリ制限
    if (preg_match('/Disallow:\s*\/(wp-includes|wp-content|config|cgi-bin)/i', $content)) {
        $features[] = [
            'feature'    => __('システムディレクトリへのクロール制限', 'ghostgate'),
            'tool'       => __('情報露出抑止・SEO設定', 'ghostgate'),
            'risk'       => 'safe',
            'suggestion' => __('ディレクトリ一覧やファイル露出を防止できます', 'ghostgate'),
        ];
    }

    // 🔹 サイト全体ブロック
    if (preg_match('/User-agent:\s*\*\s*Disallow:\s*\/\s*$/i', $content)) {
        $features[] = [
            'feature'    => __('全ページのクロール拒否', 'ghostgate'),
            'tool'       => __('開発中サイト or アクセス遮断対策', 'ghostgate'),
            'risk'       => 'warning',
            'suggestion' => __('公開サイトであれば、意図せずSEOが無効化されていないか確認してください', 'ghostgate'),
        ];
    }

    return $features;
}



//.ftpaccessの詳細調査
function ghostgate_analyze_ftpaccess_security() {
    $path = ABSPATH . '.ftpaccess';
    $features = [];

    if (!file_exists($path)) return [];

    WP_Filesystem();
    global $wp_filesystem;
    $content = $wp_filesystem->exists( $path )
        ? (string) $wp_filesystem->get_contents( $path )
        : '';


    // 🔹 IP制限
    if (preg_match('/(deny from|allow from|require ip)/i', $content)) {
        $features[] = [
            'feature'    => __('FTPアクセスに対するIP制限やアクセス制御あり', 'ghostgate'),
            'tool'       => __('旧Apache向けFTPアクセス保護設定', 'ghostgate'),
            'risk'       => 'safe',
            'suggestion' => __('許可IPが適切に限定されているかを確認してください', 'ghostgate'),
        ];
    }

    // 🔹 パスワード記述あり
    if (preg_match('/password/i', $content)) {
        $features[] = [
            'feature'    => __('FTPパスワード制御の記述あり', 'ghostgate'),
            'tool'       => __('パスワード制限またはログ管理', 'ghostgate'),
            'risk'       => 'warning',
            'suggestion' => __('パスワードが平文で記述されていないかを確認してください', 'ghostgate'),
        ];
    }

    return $features;
}



//標準のREST APIの利用状況確認
function ghostgate_analyze_rest_api_detailed() {
    $features = [];

    // ✅ GhostGate は 'rest_enabled' フィルターを変更しません。
// REVIEW: WordPress core filter name (read-only). Not plugin-defined.
/* phpcs:disable WordPress.NamingConventions.PrefixAllGlobals */
$rest_core_enabled = apply_filters( 'rest_enabled', true );
/* phpcs:enable */



    $features[] = [
        // translators: %s is the REST API status label (e.g. "✅ 有効" or "❌ 無効")
        'feature' => sprintf(__('REST APIの有効化（コア設定）: %s', 'ghostgate'),
            $rest_core_enabled ? '✅ ' . __('有効', 'ghostgate') : '❌ ' . __('無効', 'ghostgate')),
        'tool'    => 'apply_filters("rest_enabled")',
        'risk'    => $rest_core_enabled ? 'success' : 'danger',
    ];


    $rest_response = wp_remote_get(rest_url());
    $rest_accessible = (!is_wp_error($rest_response) && wp_remote_retrieve_response_code($rest_response) === 200);
    $features[] = [
        // translators: %s is the REST API response status (e.g. "✅ 正常" or "❌ エラー")
        'feature' => sprintf(__('REST API 応答: %s', 'ghostgate'), $rest_accessible ? '✅ ' . __('正常', 'ghostgate') : '❌ ' . __('エラー', 'ghostgate')),
        'tool'    => 'GET ' . rest_url(),
        'risk'    => $rest_accessible ? 'success' : 'warning',
    ];

    $app_pw_enabled = defined('WP_APPLICATION_PASSWORDS_ENABLED') ? WP_APPLICATION_PASSWORDS_ENABLED : true;
    $features[] = [
        // translators: %s is the status of Application Passwords feature (e.g. "✅ 有効" or "❌ 無効")
        'feature' => sprintf(__('Application Passwords: %s', 'ghostgate'), $app_pw_enabled ? '✅ ' . __('有効', 'ghostgate') : '❌ ' . __('無効', 'ghostgate')),
        'tool'    => 'WP_APPLICATION_PASSWORDS_ENABLED',
        'risk'    => $app_pw_enabled ? 'success' : 'danger',
    ];

// REVIEW: WordPress core filter to detect Gutenberg usage (read-only).
/* phpcs:disable WordPress.NamingConventions.PrefixAllGlobals */
$use_gutenberg = has_filter( 'use_block_editor_for_post_type' )
	? apply_filters( 'use_block_editor_for_post_type', true, 'post' )
	: false;
/* phpcs:enable */



    $features[] = [
        // translators: %s is the status of the Gutenberg block editor (e.g. "✅ 有効" or "❌ 無効")
        'feature' => sprintf(__('Gutenberg ブロックエディタ: %s', 'ghostgate'), $use_gutenberg ? '✅ ' . __('有効', 'ghostgate') : '❌ ' . __('無効', 'ghostgate')),
        'tool'    => 'use_block_editor_for_post_type',
        'risk'    => $use_gutenberg ? 'success' : 'warning',
    ];

    // WP_CLI は WordPress が CLI 実行時に定義するコア定数です。
    // WP_CLI is a core WordPress constant that is defined when running via WP-CLI.
    $is_wp_cli = defined('WP_CLI') && WP_CLI;
    $features[] = [
        // translators: %s is the WP-CLI execution status label (e.g. "✅ 使用中" or "⛔ 通常モード")
        'feature' => sprintf(__('WP-CLI 実行モード: %s', 'ghostgate'), $is_wp_cli ? '✅ ' . __('使用中', 'ghostgate') : '⛔ ' . __('通常モード', 'ghostgate')),
        'tool'    => 'WP_CLI',
        'risk'    => $is_wp_cli ? 'warning' : 'success',
    ];

    $post_types = get_post_types([], 'objects');
    $rest_post_types = array_filter($post_types, fn($pt) => !empty($pt->show_in_rest));
    $features[] = [
        // translators: %d is the number of post types that are publicly exposed via the REST API
        'feature' => sprintf(__('REST 公開投稿タイプ: %d件', 'ghostgate'), count($rest_post_types)),
        'tool'    => implode(', ', array_keys($rest_post_types)),
        'risk'    => 'success',
    ];


    $taxonomies = get_taxonomies([], 'objects');
    $rest_taxonomies = array_filter($taxonomies, fn($tx) => !empty($tx->show_in_rest));
    $features[] = [
        // translators: %d is the number of taxonomies that are publicly exposed via the REST API
        'feature' => sprintf(__('REST 公開タクソノミー: %d件', 'ghostgate'), count($rest_taxonomies)),
        'tool'    => implode(', ', array_keys($rest_taxonomies)),
        'risk'    => 'success',
    ];

    $endpoints = rest_get_server()->get_routes();

    $has_jetpack_rest = defined('JETPACK__VERSION') && has_filter('rest_api_init', 'jetpack_rest_api_init');
    $features[] = [
        // translators: %s is the Jetpack REST extension status label (e.g. "✅ 使用中" or "❌ なし")
        'feature' => sprintf(__('Jetpack REST拡張: %s', 'ghostgate'), $has_jetpack_rest ? '✅ ' . __('使用中', 'ghostgate') : '❌ ' . __('なし', 'ghostgate')),
        'tool'    => 'jetpack_rest_api_init',
        'risk'    => $has_jetpack_rest ? 'warning' : 'success',
    ];

    $has_wc_rest = class_exists('WooCommerce') && class_exists('WC_REST_Products_Controller');
    $features[] = [
        // translators: %s is the WooCommerce REST API availability status (e.g. "✅ 利用可" or "❌ 未使用")
        'feature' => sprintf(__('WooCommerce REST API: %s', 'ghostgate'), $has_wc_rest ? '✅ ' . __('利用可', 'ghostgate') : '❌ ' . __('未使用', 'ghostgate')),
        'tool'    => 'WC_REST_Products_Controller',
        'risk'    => $has_wc_rest ? 'success' : 'warning',
    ];

    $user_enum_exposed = isset($endpoints['/wp/v2/users']) && empty($endpoints['/wp/v2/users'][0]['permission_callback']);
    $features[] = [
        // translators: %s is the user enumeration exposure status (e.g. "⚠️ あり" or "✅ 対策済み")
        'feature' => sprintf(__('ユーザー列挙リスク: %s', 'ghostgate'), $user_enum_exposed ? '⚠️ ' . __('あり', 'ghostgate') : '✅ ' . __('対策済み', 'ghostgate')),
        'tool'    => '/wp/v2/users',
        'risk'    => $user_enum_exposed ? 'danger' : 'success',
    ];

    $plugin_theme_routes = array_filter(array_keys($endpoints), fn($ep) => preg_match('#/wp/v2/(plugins|themes|settings)#', $ep));
    $features[] = [
        // translators: %s is the status of plugin/theme REST API exposure (e.g. "⚠️ 有効" or "✅ 無効")
        'feature' => sprintf(__('プラグイン・テーマAPI: %s', 'ghostgate'), $plugin_theme_routes ? '⚠️ ' . __('有効', 'ghostgate') : '✅ ' . __('無効', 'ghostgate')),
        'tool'    => implode(', ', $plugin_theme_routes),
        'risk'    => $plugin_theme_routes ? 'warning' : 'success',
    ];

    $media_enabled = isset($endpoints['/wp/v2/media']);
    $features[] = [
        // translators: %s is the status of the media REST API endpoint (e.g. "✅ 有効" or "❌ 無効")
        'feature' => sprintf(__('メディア REST API: %s', 'ghostgate'), $media_enabled ? '✅ ' . __('有効', 'ghostgate') : '❌ ' . __('無効', 'ghostgate')),
        'tool'    => '/wp/v2/media',
        'risk'    => $media_enabled ? 'success' : 'warning',
    ];

    $jwt_route = isset($endpoints['/jwt-auth/v1/token']);
    $features[] = [
            // translators: %s is the status of JWT authentication API (e.g. "✅ 有効" or "❌ 無効")
        'feature' => sprintf(__('JWT認証 API: %s', 'ghostgate'), $jwt_route ? '✅ ' . __('有効', 'ghostgate') : '❌ ' . __('無効', 'ghostgate')),
        'tool'    => '/jwt-auth/v1/token',
        'risk'    => $jwt_route ? 'success' : 'warning',
    ];

    $has_restrictions = has_filter('rest_authentication_errors') || has_filter('rest_endpoints');
    $features[] = [
            // translators: %s is the status of REST API restriction filters (e.g. "✅ 有効" or "❌ 無効")
        'feature' => sprintf(__('REST API制限フィルタ: %s', 'ghostgate'), $has_restrictions ? '✅ ' . __('有効', 'ghostgate') : '❌ ' . __('無効', 'ghostgate')),
        'tool'    => 'rest_authentication_errors / rest_endpoints',
        'risk'    => $has_restrictions ? 'success' : 'danger',
    ];


    return $features;
}

//css javascript を読み込む
add_action('admin_enqueue_scripts', 'ghostgate_enqueue_adminui_assets');
function ghostgate_enqueue_adminui_assets($hook_suffix) {

    if (strpos($hook_suffix, 'ghostgate') === false) return;

    $base_url = plugin_dir_url(dirname(__FILE__)) . 'assets/';

    // ✅ JavaScript
    wp_enqueue_script(
        'ghostgate-adminui',
        $base_url . 'js/ghost-script-adminui.js',
        [],
        defined('GHOSTGATE_VERSION') ? GHOSTGATE_VERSION : '1.3.2',
        true
    );

    // ✅ CSS
    wp_enqueue_style(
        'ghostgate-adminui-style',
        $base_url . 'css/ghost-style-adminui.css',
        [],
        defined('GHOSTGATE_VERSION') ? GHOSTGATE_VERSION : '1.3.2'
    );
}
