// session-tracker.js
let lastActivity = Date.now();       // 最後のユーザー操作時刻
let lastPingAt   = 0;                // 最後にサーバへ ping した時刻

// 閾値（ms）: PHP側の ghostgate_session_interval を使う。最低1秒に丸め
const threshold = Math.max(1000, parseInt(ghostgateSession.intervalMs || '5000', 10));

function touchSession() {
  fetch(ghostgateSession.ajax_url, {
    method: 'POST',
    credentials: 'same-origin',
    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
    body: 'nonce=' + encodeURIComponent(ghostgateSession.nonce)
  })
  .then(() => { lastPingAt = Date.now(); })
  .catch(() => { /* ネットワーク一時エラーは無視（任意でログ出力可） */ });
}

// ユーザー操作を検出 → lastActivity を更新し、前回 ping から閾値経過なら ping
const activityEvents = [
  'mousemove', 'keydown', 'click', 'scroll',
  'wheel', 'touchstart', 'pointerdown', 'visibilitychange'
];

activityEvents.forEach(evt => {
  window.addEventListener(evt, () => {
    // 非表示遷移は無視（戻ってきた時＝可視化で拾う）
    if (evt === 'visibilitychange' && document.hidden) return;

    const now = Date.now();
    lastActivity = now;

    // 直近 ping から threshold 経過していれば更新
    if (now - lastPingAt >= threshold) {
      touchSession();
    }
  }, { passive: true });
});

// タブにフォーカスが戻ったときの保険（しばらく離れていた後の復帰）
window.addEventListener('focus', () => {
  const now = Date.now();
  if (now - lastPingAt >= threshold) touchSession();
}, { passive: true });

// --- 以下は管理画面の残り時間表示（元のロジック踏襲） ---
document.addEventListener('DOMContentLoaded', function () {
  const remainingEl = document.getElementById('ghostgate-remaining');
  const elapsedEl   = document.getElementById('ghostgate-elapsed');
  if (!remainingEl || !elapsedEl) return;

  let remaining = parseInt(remainingEl.dataset.remaining, 10);
  let elapsed   = parseInt(elapsedEl.dataset.elapsed, 10);

  setInterval(() => {
    if (remaining > 0) remaining--;
    elapsed++;
    remainingEl.textContent = String(remaining);
    elapsedEl.textContent   = String(elapsed);
  }, 1000);
});


// ==== アイドル超過で自動遷移（クライアント側） ====
const timeoutMs = Math.max(1000, parseInt(ghostgateSession.timeoutSec || '1800', 10) * 1000);
let ghostgateExpired = false;

function ghostgateMaybeExpire() {
  if (ghostgateExpired) return;
  const idle = Date.now() - lastActivity;
  if (idle >= timeoutMs) {
    ghostgateExpired = true;
    // ここではサーバに ping せず、そのままログイン画面へ誘導
    window.location.href = ghostgateSession.loginUrl;
  }
}

// 1秒ごとにアイドル時間を監視
setInterval(ghostgateMaybeExpire, 1000);

// ページ復帰時（非表示→表示）にも即チェック
document.addEventListener('visibilitychange', () => {
  if (!document.hidden) ghostgateMaybeExpire();
});
