<?php

namespace GeoFlex;

if (!defined("ABSPATH")) {
    exit();
}

use GeoFlex\Cache\Regions;
use GeoFlex\Admin\Metabox;
use GeoFlex\Utils\Helpers;

class Region
{
    /**
     * post
     *
     * Current region post
     *
     * @var \WP_POST
     */
    private $post;

    /**
     * home
     *
     * Default home URL
     *
     * @var string
     */
    private $home;

    /**
     * slug
     *
     * Current region slug
     *
     * @var string
     */
    private $slug;

    /**
     * replaces
     *
     * Array of replacements
     *
     * @var array
     */
    private $replaces = [];
    
    /**
     * default
     * 
     * Is default region
     *
     * @var bool
     */
    public $default = false;
    
    /**
     * regions
     *
     * All regions
     * 
     * @var array
     */
    private $regions = [];
    
    /**
     * options
     *
     * Plugin options
     * 
     * @var array
     */
    private $options;

    /**
     * __construct
     *
     * Class constructor
     *
     * @param  mixed $post_id
     * @return void
     */
    public function __construct(int $post_id)
    {
        $this->post = get_post($post_id);
        $this->home = home_url();
        $this->options = Helpers::get_settings();
        $this->regions = Regions::get_regions();

        if (!$this->post) {
            return;
        }
        
        // Region slug
        $this->slug = $this->regions[$post_id]->slug;

        // Is default
        $this->default = $this->regions[$post_id]->default;

        // Default replaces
        $this->replaces["{region_name}"] = get_the_title($post_id);
        $this->replaces["{region_link}"] = $this->replace_url($this->home);

        // Fields replaces
        foreach (Metabox::get_fields() as $field) {
            if (empty($field["id"]) || $field["type"] === "checkbox") {
                continue;
            }
            $this->replaces["{region_" . $field["id"] . "}"] = get_post_meta($post_id, Helpers::get_meta_key($field["id"]), true);
        }

        // Load styles & scripts
        add_action("wp_enqueue_scripts", [$this, "wp_enqueue_scripts"]);

        // WP footer popup code
        add_action("wp_footer", [$this, "wp_footer_popup"]);

        // Filters
        add_filter("home_url", [$this, "replace_url"], 10);
        add_filter("redirect_canonical", [$this, "redirect_canonical"], 10, 2);

        // Replacements filters
        add_filter("document_title", [$this, "replace_strip_tags"], 10);
        add_filter("the_title", [$this, "replace"], 10);
        add_filter("the_content", [$this, "replace"], 10);
        add_filter("render_block", [$this, "replace"], 10);
        add_filter("get_the_tags", [$this, "get_the_tags"], 10);

        // WP header and footer code blocks
        add_action("wp_head", [$this, "wp_head"]);
        add_action("wp_footer", [$this, "wp_footer"]);

        // Custom filters
        do_action("geoflex_filters", $this);
    }
    
    /**
     * get_home
     *
     * Return default home
     * 
     * @return string
     */
    public function get_home()
    {
        return $this->home;
    }
    
    /**
     * get_regions
     *
     * Return list of regions
     * 
     * @return array
     */
    public function get_regions()
    {
        return $this->regions;
    }
    
    /**
     * get_options
     *
     * Return plugin options
     * 
     * @return array
     */
    public function get_options()
    {
        return $this->options;
    }
    
    /**
     * get_replaces
     *
     * Return plugin replaces
     * 
     * @return array
     */
    public function get_replaces()
    {
        return $this->replaces;
    }

    /**
     * replace_url
     *
     * Replace url
     *
     * @param  string $url
     * @param  string $slug
     * @return string
     */
    public function replace_url(string $url, string $slug = null)
    {
        if (empty($slug) && $this->default) {
            return $url;
        }

        if (empty($slug)) {
            $slug = $this->slug;
        }

        // Domain of current URL
        $domain = wp_parse_url($url, PHP_URL_HOST);

        // Path of current URL
        $path = wp_parse_url($url, PHP_URL_PATH);

        switch ($this->options["type"]) {
            case "domain":
                $domain = $slug . "." . $domain;
                break;
            default:
                $path = "/" . $slug . $path;
        }

        return set_url_scheme(esc_url_raw(wp_unslash("http://" . $domain . $path)));
    }
    
    /**
     * redirect_canonical
     *
     * Prevent cannonical redirect for home page
     * 
     * @param  string $redirect
     * @param  string $requested
     * @return void
     */
    public function redirect_canonical($redirect, $requested)
    {
        $redirect_domain = wp_parse_url($redirect, PHP_URL_HOST);
        $requested_domain = wp_parse_url($requested, PHP_URL_HOST);

        if (function_exists("idn_to_ascii")) {
            $redirect_domain = idn_to_ascii($redirect_domain );
        }

        if ($redirect_domain === $requested_domain) {
            return false;
        }

        return $redirect;
    }
    
    /**
     * replace
     * 
     * Replace texts by replaces array
     *
     * @param  string|array $text
     * @param  bool   $strip_tags
     * @return string
     */
    public function replace(string|array $text, bool $strip_tags = false)
    {
        // Chech array of strings
        if (is_array($text)) {
            $not_strings = false;
            array_walk_recursive($text, function($value, $key) use (&$not_strings) {
                $not_strings = $not_strings || !is_string($value);
            });
            if ($not_strings) {
                return $text;
            }
        }
        
        // Maybe strip tags
        $replaces = $this->replaces;
        if ($strip_tags) {
            foreach ($replaces as $key => &$value) {
                $value = wp_strip_all_tags($value, true);
            }
        }

        // Replace text
        $text = str_replace(array_keys($replaces), array_values($replaces), $text);

        // Replace links (skip replaced, wp-admin, etc.)
        if (!$this->default) {
            $skip = apply_filters("geoflex_skip", [
                $this->slug,
                "wp-admin",
                basename(WP_CONTENT_DIR),
            ]);
            $text = preg_replace("/https?:\/\/" . preg_quote(wp_parse_url($this->home, PHP_URL_HOST)) . "\/(?!(" . implode("|", $skip) . "))/", $this->replace_url($this->home) . "/", $text);
        }

        return $text;
    }
    
    /**
     * replace_strip_tags
     *
     * Replacing texts with an array of replaces with deleted tags and breaks
     * 
     * @param  string|array $text
     * @return void
     */
    public function replace_strip_tags(string|array $text)
    {
        return $this->replace($text, true);
    }
    
    /**
     * get_the_tags
     *
     * Get tags filter
     * 
     * @param  mixed $terms
     * @return mixed
     */
    public function get_the_tags($terms)
    {
        if (is_array($terms)) {
            foreach ($terms as &$term) {
                $term->name = $this->replace_strip_tags($term->name);
            }
        }
        
        return $terms;
    }
    
    /**
     * wp_enqueue_scripts
     *
     * Add styles and scripts
     * 
     * @return void
     */
    public function wp_enqueue_scripts()
    {
        wp_enqueue_style("geoflex", GEOFLEX_URL . "assets/css/styles.css", array(), GEOFLEX_VER);
        wp_enqueue_script("geoflex", GEOFLEX_URL . "assets/js/scripts.js", array(), GEOFLEX_VER, true);
    }

    private function get_region_html($region)
    {
        ?>
            <li <?php if ($region->slug === $this->slug) : ?>class="current"<?php endif; ?>>
                <a href="<?php echo esc_url($region->default ? $this->home : $this->replace_url($this->home, $region->slug)); ?>">
                    <?php echo esc_html($region->name); ?>
                </a>
            </li>
        <?php
    }
    
    /**
     * wp_footer_popup
     *
     * Add popup html to the footer
     * 
     * @return void
     */
    public function wp_footer_popup()
    {
        $regions = [...$this->regions];
        $count = is_array($regions) ? count($regions) : 0;
        if ($count > 1) :
            switch (true) {
                case ($count > 10 && $count <= 20):
                    $class = "columns-2";
                    break;
                case ($count > 20 && $count <= 30):
                    $class = "columns-3";
                    break;
                case ($count > 30 && $count <= 80):
                    $class = "columns-3 chars";
                    break;
                case ($count > 80):
                    $class = "columns-4 chars";
                    break;
                default:
                    $class = "columns-1";
            }

            $default = array_shift($regions);
            
            usort($regions, function ($a, $b) { 
                return strcasecmp(mb_strtoupper($a->name), mb_strtoupper($b->name)); 
            });

            $letters = [];
            foreach ($regions as $region) {
                $letter = mb_strtoupper(mb_substr($region->name, 0, 1));
                if (!array_key_exists($letter, $letters)) {
                    $letters[$letter] = [];
                }
                $letters[$letter][] = $region;
            }
        ?>
        <div class="geoflex geoflex--blocker">
            <div class="geoflex--popup <?php echo esc_attr($class); ?>" data-count="<?php echo esc_attr($count); ?>">
                <div class="geoflex--header">
                    <h3><?php esc_html_e("Select a region", "geoflex"); ?></h3>
                    <a class="geoflex--close" href="#" rel="region"></a>
                </div>
                <ul class="geoflex--list">
                    <!-- Default region -->
                    <?php $this->get_region_html($default); ?>
                    <!-- // Default region -->

                    <?php
                        if ($count > 31) {
                            foreach ($letters as $letter => $regions) :
                    ?>
                    <ul class="geoflex--letter">
                        <span><?php echo esc_html($letter); ?></span>
                        <?php
                                foreach ($regions as $region) :
                                    $this->get_region_html($region);
                                endforeach;
                        ?>
                    </ul>
                    <?php
                            endforeach;
                        } else {
                            foreach ($regions as $region) :
                                $this->get_region_html($region);
                            endforeach;
                        }
                    ?>
                </ul>
            </div>
        </div>
        <?php
        endif;
    }
    
    /**
     * wp_head
     *
     * Add header code block
     * 
     * @return void
     */
    public function wp_head()
    {
        if ($custom_css = get_post_meta($this->post->ID, Helpers::get_meta_key("custom_css"), true)) {
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
            echo '<style id="geoflex-custom-css">'.$custom_css.'</style>', "\n";
        }

        if ($header_js = get_post_meta($this->post->ID, Helpers::get_meta_key("header_js"), true)) {
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
            echo '<script id="geoflex-header-js-code">'.$header_js.'</script>', "\n";
        }

        if ($header_html = get_post_meta($this->post->ID, Helpers::get_meta_key("header_html"), true)) {
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
            echo $header_html;
        }
    }
    
    /**
     * wp_footer
     * 
     * Add footer code block
     *
     * @return void
     */
    public function wp_footer()
    {
        if ($footer_js = get_post_meta($this->post->ID, Helpers::get_meta_key("footer_js"), true)) {
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
            echo '<script id="geoflex-footer-js-code">'.$footer_js.'</script>', "\n";
        }

        if ($footer_html = get_post_meta($this->post->ID, Helpers::get_meta_key("footer_html"), true)) {
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
            echo $footer_html;
        }
    }
}
