<?php

namespace GeoFlex;

if (!defined("ABSPATH")) {
    exit();
}

use GeoFlex\Region;
use GeoFlex\Admin;
use GeoFlex\Cache;
use GeoFlex\Utils\Helpers;

class Plugin
{
    /**
     * __construct
     *
     * Class constructor
     *
     * @return void
     */
    public function __construct()
    {
        new Admin\PostType();
        new Admin\Metabox();
        new Admin\Settings();
        new Admin\Support();
        new Admin\Notices();
        new Cache\Regions();

        if (!is_admin()) {
            add_filter("do_parse_request", [$this, "parse_request"], 10, 1);
        }

        add_filter("wp_kses_allowed_html", [$this, "kses_allowed_html"], 10, 2);

    }

    /**
     * get_requsted_url
     *
     * Get current requested URL include region
     *
     * @return mixed
     */
    public static function get_requsted_url()
    {
        if (isset($_SERVER["HTTP_HOST"], $_SERVER["REQUEST_URI"])) {
            return set_url_scheme(wp_unslash("http://" . $_SERVER["HTTP_HOST"] . $_SERVER["REQUEST_URI"]));
        }
        return home_url();
    }

    /**
     * parse_request
     *
     * Parse request
     *
     * @param bool $do_parse
     * @return bool
     */
    public function parse_request(bool $do_parse)
    {
        if (is_admin() || wp_count_posts(GEOFLEX_POST_TYPE)->publish === 0) {
            return $do_parse;
        }

        $options = Helpers::get_settings();

        switch ($options["type"]) {
            case "domain":
                $region = self::get_region_by_domain();
                break;
            default:
                $region = self::get_region_by_path();
        }

        if (!empty($region)) {
            $GLOBALS["geoflex_region"] = new Region($region);
        }

        return $do_parse;
    }

    /**
     * get_region_by_domain
     *
     * Get region by domain
     *
     * @return bool|int
     */
    private static function get_region_by_domain()
    {
        if (wp_count_posts(GEOFLEX_POST_TYPE)->publish === 0) {
            return false;
        }

        $domain = wp_parse_url(self::get_requsted_url(), PHP_URL_HOST);
        $home = wp_parse_url(home_url(), PHP_URL_HOST);
        $slug = preg_replace("/\./", "", str_replace($home, "", $domain));

        if ($slug && function_exists("idn_to_utf8")) {
            $slug = sanitize_title(idn_to_utf8($slug));
        }

        return $slug ? self::get_region_by_slug($slug) : self::get_default_region();
    }

    /**
     * get_region_by_path
     *
     * Get region by URL path
     *
     * @return bool|int
     */
    private static function get_region_by_path()
    {
        if (wp_count_posts(GEOFLEX_POST_TYPE)->publish === 0) {
            return false;
        }

        $path = trim(wp_parse_url(self::get_requsted_url(), PHP_URL_PATH), "/");

        // Empty path is default region
        if (empty($path)) {
            return self::get_default_region();
        }

        $path = explode("/", trim($path, "/"));
        $slug = !empty($path) ? reset($path) : false;
        $region = self::get_region_by_slug($slug);

        // Region not found, return default
        if (empty($region)) {
            return self::get_default_region();
        }

        $_SERVER["REQUEST_URI"] = preg_replace("/^\/$slug\/?/", "/", $_SERVER["REQUEST_URI"]);

        return $region;
    }

    /**
     * get_region_by_slug
     *
     * Get region post_id by slug
     *
     * @param string $name
     * @return int
     */
    public static function get_region_by_slug(string $name)
    {
        if (empty($name)) {
            return false;
        }

        $regions = get_posts([
            "post_type" => GEOFLEX_POST_TYPE,
            "posts_per_page" => 1,
            "meta_key" => Helpers::get_meta_key("slug"),
            "meta_value" => $name,
            "fields" => "ids",
        ]);

        if (!empty($regions)) {
            return reset($regions);
        }

        return false;
    }

    /**
     * get_default_region
     *
     * Get default region post_id by slug
     *
     * @return int
     */
    public static function get_default_region()
    {
        $regions = get_posts([
            "post_type" => GEOFLEX_POST_TYPE,
            "posts_per_page" => 1,
            "meta_key" => Helpers::get_meta_key("default"),
            "meta_value" => true,
            "fields" => "ids",
        ]);

        if (!empty($regions)) {
            return reset($regions);
        }

        return false;
    }

    public function kses_allowed_html($tags, $context) {

        $attr = [
            'id'    => [],
            'class' => [],
            'style' => [],
        ];

        switch ($context) {

            case 'geoflex-base' :
                $tags = [
                    'br' => [],
                    'a'  => [
                        'id'     => [],
                        'class'  => [],
                        'style'  => [],
                        'href'   => [],
                        'target' => [],
                        'title'  => [],
                        'rel'    => [],
                    ],
                    'ul' => [
                        'id'    => [],
                        'class' => [],
                        'style' => [],
                        'type'  => [],
                    ],
                    'ol' => [
                        'id'    => [],
                        'class' => [],
                        'style' => [],
                        'type'  => [],
                    ],
                    'div'    => $attr,
                    'p'      => $attr,
                    'h1'     => $attr,
                    'h2'     => $attr,
                    'h3'     => $attr,
                    'h4'     => $attr,
                    'h5'     => $attr,
                    'h6'     => $attr,
                    'li'     => $attr,
                    'strong' => $attr,
                    'b'      => $attr,
                    'span'   => $attr,
                    'mark'   => $attr,
                    'i'      => $attr,
                    'del'    => $attr,
                    'ins'    => $attr,
                    'em'     => $attr,
                    'sup'    => $attr,
                ];
                break;

            case 'geoflex-extended' :
                $tags = [
                    'script' => [
                        'id'             => [],
                        'src'            => [],
                        'type'           => [],
                        'blocking'       => [],
                        'language'       => [],
                        'async'          => true,
                        'crossorigin'    => true,
                        'defer'          => true,
                        'integrity'      => true,
                        'nomodule'       => true,
                        'referrerpolicy' => true,
                    ],
                    'style' => [
                        'id'    => [],
                        'src'   => [],
                        'type'  => [],
                        'media' => [],
                        'scoped',
                        'disabled'
                    ],
                    'br'  => [],
                    'img' => [
                        'id'     => [],
                        'class'  => [],
                        'style'  => [],
                        'src'    => [],
                        'width'  => [],
                        'height' => [],
                        'alt'    => [],
                        'sizes'  => [],
                        'srcset' => [],
                        'data-srcset' => [],
                        'data-id' => [],
                    ],
                    'a'  => [
                        'id'     => [],
                        'class'  => [],
                        'style'  => [],
                        'href'   => [],
                        'target' => [],
                        'title'  => [],
                        'rel'    => [],
                    ],
                    'ul' => [
                        'id'    => [],
                        'class' => [],
                        'style' => [],
                        'type'  => [],
                    ],
                    'ol' => [
                        'id'    => [],
                        'class' => [],
                        'style' => [],
                        'type'  => [],
                    ],
                    'div'    => $attr,
                    'p'      => $attr,
                    'h1'     => $attr,
                    'h2'     => $attr,
                    'h3'     => $attr,
                    'h4'     => $attr,
                    'h5'     => $attr,
                    'h6'     => $attr,
                    'li'     => $attr,
                    'strong' => $attr,
                    'b'      => $attr,
                    'span'   => $attr,
                    'mark'   => $attr,
                    'i'      => $attr,
                    'del'    => $attr,
                    'ins'    => $attr,
                    'em'     => $attr,
                    'sup'    => $attr,
                ];
        
        }

        return $tags;

    }

}
