<?php

namespace GeoFlex\Admin;

if (!defined("ABSPATH")) {
    exit();
}

use GeoFlex\Utils\Helpers;

class PostType
{
    /**
     * __construct
     *
     * Class constructor
     *
     * @return void
     */
    public function __construct()
    {
        add_action("init", [&$this, "register_post_type"]);
        add_action("init", [&$this, "admin_custom_columns"]);
        add_filter("admin_head", [&$this, "remove_publishing_actions"], PHP_INT_MAX);
    }

    /**
     * register_post_type
     *
     * Register post type
     *
     * @return void
     */
    public function register_post_type()
    {
        $labels = [
            "name" => esc_html__("Regions", "geoflex"),
            "singular_name" => esc_html__("Region", "geoflex"),
            "menu_name" => esc_html__("GeoFlex", "geoflex"),
            "name_admin_bar" => esc_html__("Region", "geoflex"),
            "archives" => esc_html__("Regions Archives", "geoflex"),
            "attributes" => esc_html__("Region Attributes", "geoflex"),
            "parent_item_colon" => esc_html__("Parent Region:", "geoflex"),
            "all_items" => esc_html__("All Regions", "geoflex"),
            "add_new_item" => esc_html__("Add New Region", "geoflex"),
            "add_new" => esc_html__("Add New", "geoflex"),
            "new_item" => esc_html__("New Region", "geoflex"),
            "edit_item" => esc_html__("Edit Region", "geoflex"),
            "update_item" => esc_html__("Update Region", "geoflex"),
            "view_item" => esc_html__("View Region", "geoflex"),
            "view_items" => esc_html__("View Regions", "geoflex"),
            "search_items" => esc_html__("Search Region", "geoflex"),
            "not_found" => esc_html__("Not found", "geoflex"),
            "not_found_in_trash" => esc_html__("Not found in Trash", "geoflex"),
            "featured_image" => esc_html__("Featured Image", "geoflex"),
            "set_featured_image" => esc_html__("Set featured image", "geoflex"),
            "remove_featured_image" => esc_html__("Remove featured image", "geoflex"),
            "use_featured_image" => esc_html__("Use as featured image", "geoflex"),
            "insert_into_item" => esc_html__("Insert into Region", "geoflex"),
            "uploaded_to_this_item" => esc_html__("Uploaded to this Region", "geoflex"),
            "items_list" => esc_html__("Regions list", "geoflex"),
            "items_list_navigation" => esc_html__("Regions list navigation", "geoflex"),
            "filter_items_list" => esc_html__("Filter Regions list", "geoflex"),
        ];
        $args = [
            "label" => esc_html__("Region", "geoflex"),
            "description" => esc_html__("Regions", "geoflex"),
            "labels" => $labels,
            "supports" => ["title"],
            "hierarchical" => false,
            "public" => true,
            "show_ui" => true,
            "show_in_menu" => true,
            "menu_position" => 100,
            "menu_icon" => $this->get_icon_svg(true),
            "show_in_admin_bar" => false,
            "show_in_nav_menus" => false,
            "can_export" => true,
            "has_archive" => false,
            "exclude_from_search" => true,
            "publicly_queryable" => false,
            "rewrite" => false,
            "capability_type" => "page",
            "show_in_rest" => false,
        ];
        register_post_type(GEOFLEX_POST_TYPE, $args);
    }
         
    /**
     * get_icon_svg
     *
     * Get plugin icon
     * 
     * @param  mixed $base64
     * @return void
     */
    public function get_icon_svg($base64 = true)
    {
        $svg = '<svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg">
        <path d="M18 8.87858C18 10.9679 17.1572 12.9717 15.6568 14.4495L14.3677 15.7027L11.585 18.3689C10.7508 19.1631 9.45375 19.2081 8.56809 18.5016L8.4161 18.3689L6.08139 16.1358L4.34321 14.4495C2.84284 12.9717 2 10.9679 2 8.87858C2 6.78926 2.84284 4.78546 4.34321 3.3077C5.08589 2.5761 5.96773 1.99574 6.93835 1.59977C7.90897 1.20381 8.94934 1 10 1C11.0507 1 12.091 1.20381 13.0616 1.59977C14.0323 1.99574 14.9141 2.5761 15.6568 3.3077C17.1572 4.78546 18 6.78926 18 8.87858Z" fill="#A5AAAD"/>
        </svg>';

        if ($base64) {
            return "data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjAiIGhlaWdodD0iMjAiIHZpZXdCb3g9IjAgMCAyMCAyMCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KPHBhdGggZD0iTTE4IDguODc4NThDMTggMTAuOTY3OSAxNy4xNTcyIDEyLjk3MTcgMTUuNjU2OCAxNC40NDk1TDE0LjM2NzcgMTUuNzAyN0wxMS41ODUgMTguMzY4OUMxMC43NTA4IDE5LjE2MzEgOS40NTM3NSAxOS4yMDgxIDguNTY4MDkgMTguNTAxNkw4LjQxNjEgMTguMzY4OUw2LjA4MTM5IDE2LjEzNThMNC4zNDMyMSAxNC40NDk1QzIuODQyODQgMTIuOTcxNyAyIDEwLjk2NzkgMiA4Ljg3ODU4QzIgNi43ODkyNiAyLjg0Mjg0IDQuNzg1NDYgNC4zNDMyMSAzLjMwNzdDNS4wODU4OSAyLjU3NjEgNS45Njc3MyAxLjk5NTc0IDYuOTM4MzUgMS41OTk3N0M3LjkwODk3IDEuMjAzODEgOC45NDkzNCAxIDEwIDFDMTEuMDUwNyAxIDEyLjA5MSAxLjIwMzgxIDEzLjA2MTYgMS41OTk3N0MxNC4wMzIzIDEuOTk1NzQgMTQuOTE0MSAyLjU3NjEgMTUuNjU2OCAzLjMwNzdDMTcuMTU3MiA0Ljc4NTQ2IDE4IDYuNzg5MjYgMTggOC44Nzg1OFoiIGZpbGw9IiNBNUFBQUQiLz4KPC9zdmc+Cg==";
        }

        return $svg;
    }

    /**
     * admin_custom_columns
     *
     * Add columns to the admin are
     *
     * @return void
     */
    public function admin_custom_columns()
    {
        add_filter("manage_" . GEOFLEX_POST_TYPE . "_posts_columns", function ($columns) {
            $new_columns = [
                "is_default" => esc_html__("Default", "geoflex"),
                "slug" => esc_html__("Slug", "geoflex"),
                "link" => esc_html__("Link", "geoflex"),
            ];
            $offset = array_search("title", array_keys($columns)) + 1;
            return array_merge(array_slice($columns, 0, $offset), $new_columns, array_slice($columns, $offset, null));
        });

        add_action(
            "manage_" . GEOFLEX_POST_TYPE . "_posts_custom_column",
            function ($column_key, $post_id) {
                $slug = urldecode(get_post_meta($post_id, Helpers::get_meta_key("slug"), true));
                $default = get_post_meta($post_id, Helpers::get_meta_key("default"), true);

                // Default column
                if ($column_key == "is_default") {
                    echo ($default) ? "&check;" : "";
                }

                // Slug column
                if ($column_key == "slug") {
                    echo esc_html($slug);
                }

                // Link column
                if ($column_key == "link") {
                    $options = Helpers::get_settings();
                    if ($default) {
                        $link = home_url();
                    } else {
                        $link = set_url_scheme(esc_url_raw(wp_unslash("http://" . ($options["type"] === "domain" ? $slug . "." : "") . wp_parse_url(home_url(), PHP_URL_HOST) . "/" . ($options["type"] === "path" ? $slug . "/" : ""))));
                    }
                    printf('<a href="%s" target="_blank">%s</a>', esc_url($link), esc_html($link));
                }
            },
            10,
            2
        );

        add_filter("manage_edit-" . GEOFLEX_POST_TYPE . "_sortable_columns", function ($sortable_columns) {
            $sortable_columns["is_default"] = ["geoflex_default", false];
            $sortable_columns["slug"] = ["geoflex_slug", false];
            return $sortable_columns;
        });

        add_filter("request", function ($vars) {
            if (isset($vars["orderby"])) {
                switch ($vars["orderby"]) {
                    case "geoflex_default":
                        $vars["meta_key"] = Helpers::get_meta_key("default");
                        $vars["orderby"] = "meta_value_num";
                        break;
                    case "geoflex_slug":
                        $vars["meta_key"] = Helpers::get_meta_key("slug");
                        $vars["orderby"] = "meta_value";
                        break;
                }
            }
	        return $vars;
        });
    }
        
    /**
     * remove_publishing_actions
     *
     * Remove visibility menu for region post type
     * 
     * @return void
     */
    public function remove_publishing_actions()
    {
        global $typenow;
        if ($typenow == GEOFLEX_POST_TYPE) { ?>
            <style>
                #visibility {
                    display: none !important;
                }
            </style>
            <script>
                document.addEventListener("DOMContentLoaded", () => {
                    const singleVisibility = document.getElementById("visibility");
                    singleVisibility && singleVisibility.remove();

                    const inlineEdit = document.getElementById("inline-edit");
                    if (inlineEdit) {
                        const passwordInput = inlineEdit.querySelector("[name=post_password]");
                        if (passwordInput) {
                            const protectionBlock = passwordInput.closest(".inline-edit-group");
                                console.log(protectionBlock);
                            if (protectionBlock) {
                                protectionBlock.style.display = "none";
                            }
                        } 
                    }
                });
            </script>   
        <?php }
    }
}
