<?php

namespace GeoFlex\Admin;

if (!defined("ABSPATH")) {
    exit();
}

use GeoFlex\Utils\Helpers;

class Metabox
{
    /**
     * fields
     *
     * Metabox fields
     *
     * @var
     */
    private $fields;

    /**
     * __construct
     *
     * Class constructor
     *
     * @return void
     */
    public function __construct()
    {
        add_action("init", [$this, "load_fields"], 20);
        add_action("edit_form_after_title", [$this, "metabox_callback"]);
        add_action("admin_enqueue_scripts", [$this, "init_code_fields"]);
        add_action("save_post", [$this, "save_post"]);
        add_filter("geoflex_field_default", [&$this, "default_field_value"], 10);
        add_filter("geoflex_field_slug", [&$this, "slug_field_value"], 10, 2);
        add_filter("wp_insert_post_data", [&$this, "default_region_post_data"], 10, 2);
        add_filter("pre_trash_post", [&$this, "pre_trash_post"], 10, 3);
        add_filter("admin_head", [&$this, "remove_status_actions"], PHP_INT_MAX);
    }

    /**
     * get_fields
     *
     * Get array of fields
     *
     * @return array
     */
    public static function get_fields()
    {
        $fields = [
            [
                "id" => "default",
                "type" => "checkbox",
                "label" => esc_html__("Default Region", "geoflex"),
            ],
            [
                "id" => "slug",
                "type" => "text",
                "label" => esc_html__("Slug", "geoflex"),
            ],
            [
                "id" => "from",
                "type" => "text",
                "label" => esc_html__("From Region", "geoflex"),
                "note" => esc_html__("Use in content", "geoflex")." {region_from}",
            ],
            [
                "id" => "by",
                "type" => "text",
                "label" => esc_html__("By Region", "geoflex"),
                "note" => esc_html__("Use in content", "geoflex")." {region_by}",
            ],
            [
                "id" => "in",
                "type" => "text",
                "label" => esc_html__("In Region", "geoflex"),
                "note" => esc_html__("Use in content", "geoflex")." {region_in}",
            ],
            [
                "id" => "email",
                "type" => "email",
                "label" => esc_html__("Email", "geoflex"),
                "note" => esc_html__("Use in content", "geoflex")." {region_email}",
            ],
            [
                "id" => "phone",
                "type" => "tel",
                "label" => esc_html__("Phone", "geoflex"),
                "note" => esc_html__("Use in content", "geoflex")." {region_phone}",
            ],
            [
                "id" => "address",
                "type" => "textarea",
                "label" => esc_html__("Address", "geoflex"),
                "note" => esc_html__("Use in content", "geoflex")." {region_address}",
            ],
            [
                "id" => "custom_css",
                "type" => "code_css",
                "label" => esc_html__("Custom css code", "geoflex"),
                "description" => esc_html__("Your custom css code here (without wrapper &lt;style&gt;), e.g.", "geoflex").'<br>body {color:red;}<br><br>',
            ],
            [
                "id" => "header_js",
                "type" => "code_js",
                "label" => esc_html__("Header js code", "geoflex"),
                "description" => esc_html__("Your custom js code here (without wrapper &lt;script&gt;), e.g.", "geoflex").'<br>"use strict";<br>console.log("Hello");<br><br>',
            ],
            [
                "id" => "header_html",
                "type" => "code_html",
                "label" => esc_html__("Header html code", "geoflex"),
            ],
            [
                "id" => "footer_js",
                "type" => "code_js",
                "label" => esc_html__("Footer js code", "geoflex"),
                "description" => esc_html__("Your custom js code here (without wrapper &lt;script&gt;), e.g.", "geoflex").'<br>"use strict";<br>console.log("Hello");<br><br>',
            ],
            [
                "id" => "footer_html",
                "type" => "code_html",
                "label" => esc_html__("Footer html code", "geoflex"),
            ],
        ];
        return apply_filters("geoflex_metabox_fields", $fields);
    }

    /**
     * load_fields
     *
     * Fields loading
     * Must be called after translations are initialized
     *
     * @return void
     */
    public function load_fields()
    {
        $this->fields = $this->get_fields();
    }

    /**
     * metabox_callback
     *
     * Prints metabox
     *
     * @param  WP_Post $post
     * @return void
     */
    public function metabox_callback($post)
    {
        if ($post->post_type !== GEOFLEX_POST_TYPE) {
            return;
        } 
        wp_nonce_field( 'geoflex_save_data', 'geoflex_meta_nonce' );
        echo "<i>".esc_html__("Use in content", "geoflex")." {region_name}"."</i>";
        ?>
        <table class="form-table" role="presentation">
            <tbody>
                <?php
                    foreach ($this->fields as $field):
                        if (empty($field["id"]) || empty($field["type"]) || empty($field["label"])) {
                            continue;
                        }
                ?>
                    <tr>
                        <th scope="row">
                            <?php $this->label($field); ?>
                        </th>
                        <td>
                            <?php $this->field($field, $post); ?>
                        </td>
                    </tr>
                <?php
                    endforeach;
                ?>
            </tbody>
        </table>
        <?php
    }

    /**
     * label
     *
     * Prints label of the field
     *
     * @param  array $field
     * @return void
     */
    private function label(array $field)
    {
        if (!is_array($field) || !isset($field["id"]) || !isset($field["label"])) {
            return;
        }

        $id = $field["id"];
        $label = $field["label"];

        printf('<label for="%s">%s</label>', esc_attr($id), esc_html($label));
    }

    /**
     * field
     *
     * Prints the field
     *
     * @param  array $field
     * @param  WP_Post $post
     * @return void
     */
    private function field(array $field, \WP_Post $post)
    {
        if (!is_array($field) || !isset($field["id"]) || !isset($field["type"])) {
            return;
        }

        $id = Helpers::get_meta_key($field["id"], false);
        $type = $field["type"];
        $default = isset($field["default"] )? $field["default"] : "";
        $description = isset($field["description"] )? $field["description"] : "";
        $note = isset($field["note"] )? $field["note"] : "";
        $value = $this->value($field, $post) ? $this->value($field, $post) : $default;

        if ($description) {
            echo '<p>'.wp_kses($description, "geoflex-base").'</p>';
        }

        switch ($field["type"]) {
            case "checkbox":
                printf('<input %s id="%s" name="%s" type="checkbox">', checked($value, true, false), esc_attr($id), esc_attr($id));
                break;

            case "textarea":
                printf('<textarea id="%s" name="%s" rows="5" cols="60">%s</textarea>', esc_attr($id), esc_attr($id), wp_kses_post($value));
                break;

            case "code_css":
                printf('<textarea class="region-code-css" id="%s" name="%s" rows="5" cols="30">%s</textarea>', esc_attr($id), esc_attr($id), wp_kses($value, "strip"));
                break;

            case "code_js":
                printf('<textarea class="region-code-js" id="%s" name="%s" rows="5" cols="30">%s</textarea>', esc_attr($id), esc_attr($id), wp_kses_post($value));
                break;

            case "code_html":
                printf('<textarea class="region-code-html" id="%s" name="%s" rows="5" cols="30">%s</textarea>', esc_attr($id), esc_attr($id), wp_kses($value, "geoflex-extended"));
                break;

            default:
                printf('<input class="regular-text" id="%s" name="%s" type="%s" value="%s" size="25">', esc_attr($id), esc_attr($id), isset($type) ? esc_attr($type) : "text", esc_attr($value));
        }

        if ($note) {
            echo '<p><i>'.wp_kses($note, "geoflex-base").'</i></p>';
        }
    }

    /**
     * value
     *
     * Return current value by field
     *
     * @param  array $field
     * @param  WP_Post $post
     * @return void
     */
    private function value(array $field, \WP_Post $post)
    {
        if (!is_array($field) || !isset($field["id"])) {
            return false;
        }

        $value = get_post_meta($post->ID, Helpers::get_meta_key($field["id"]), true) ?: "";

        if ($field["id"] === "slug") {
            $value = urldecode($value);
        }

        return $value;
    }

    /**
     * init_code_fields
     *
     * Initialization code fields in the admin panel
     *
     * @return void
     */
    public function init_code_fields()
    {
        if (GEOFLEX_POST_TYPE !== get_current_screen()->id) {
            return;
        }

        $settings_html = wp_enqueue_code_editor(["type" => "text/html"]);
        $settings_css = wp_enqueue_code_editor(["type" => "text/css"]);
        $settings_js = wp_enqueue_code_editor(["type" => "application/javascript"]);

        if (false === $settings_html || false === $settings_css || false === $settings_js) {
            return;
        }

        $script  = 'const codeEditorSettingsHtml = '.wp_json_encode($settings_html).';';
        $script .= 'const codeEditorSettingsCss = '.wp_json_encode($settings_css).';';
        $script .= 'const codeEditorSettingsJs = '.wp_json_encode($settings_js).';';
        $script .= '
            document.addEventListener("DOMContentLoaded", () => {
                jQuery(".region-code-html").each(function () {
                    wp.codeEditor.initialize(this, codeEditorSettingsHtml);
                })
                jQuery(".region-code-css").each(function () {
                    wp.codeEditor.initialize(this, codeEditorSettingsCss);
                })
                jQuery(".region-code-js").each(function () {
                    wp.codeEditor.initialize(this, codeEditorSettingsJs);
                })
            });
        ';
        
        wp_add_inline_script("code-editor", $script);
    }

    /**
     * save_post
     *
     * Process fields saving
     *
     * @param  int $post_id
     * @return void
     */
    public function save_post(int $post_id)
    {
        if (get_post_type($post_id) !== GEOFLEX_POST_TYPE || wp_doing_ajax() || (defined("WP_CLI") && WP_CLI)) {
            return;
        }

        // Check the nonce.
		if (empty($_POST['geoflex_meta_nonce']) || !wp_verify_nonce(sanitize_key($_POST['geoflex_meta_nonce']), 'geoflex_save_data') ) {
			return;
		}

        foreach ($this->fields as $field) {
            $id = $field["id"];

            if (empty($id)) {
                continue;
            }

            $key = Helpers::get_meta_key($id, false);
            $type = $field["type"];

            switch ($type) {
                case "checkbox":
                    $value = !!array_key_exists($key, $_POST);
                    break;
                case "textarea":
                    $value = array_key_exists($key, $_POST) ? wp_kses_post(wp_unslash($_POST[$key])) : false;
                    break;
                case "code_css":
                    // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
                    $value = array_key_exists($key, $_POST) ? wp_unslash($_POST[$key]) : false;
                    break;
                case "code_js":
                    // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
                    $value = array_key_exists($key, $_POST) ? wp_unslash($_POST[$key]) : false;
                    break;
                case "code_html":
                    // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
                    $value = array_key_exists($key, $_POST) ? wp_unslash($_POST[$key]) : false;
                    break;
                default:
                    $value = array_key_exists($key, $_POST) ? sanitize_text_field(wp_unslash($_POST[$key])) : false;
            }

            $value = apply_filters("geoflex_field_" . $id, $value, $post_id);

            update_post_meta($post_id, Helpers::get_meta_key($id), $value);
        }
    }

    /**
     * default_field_value
     *
     * Default field filter
     *
     * @param  mixed $value
     * @return bool
     */
    public function default_field_value($value)
    {
        global $post;

        if (empty($post)) {
            return $value;
        }

        $defaults = get_posts([
            "post_type" => GEOFLEX_POST_TYPE,
            "posts_per_page" => -1,
            "exclude" => $post->ID,
            "meta_key" => Helpers::get_meta_key("default"),
            "meta_value" => true,
            "fields" => "ids",
        ]);

        if ($value) {
            if (!empty($defaults)) {
                foreach ($defaults as $post_id) {
                    if ($post_id !== $post->ID) {
                        update_post_meta($post_id, Helpers::get_meta_key("default"), false);
                    }
                }
            }
        } else {
            if (empty($defaults)) {
                wp_publish_post($post->ID);
                return true;
            }
        }

        return $value;
    }

    /**
     * slug_field_value
     *
     * Slug field filter
     *
     * @param  mixed $value
     * @return string
     */
    public function slug_field_value($value, $post_id)
    {
        global $wpdb;
        
        // If empty value, use post title
        if (empty($value) && get_post_status($post_id) === 'publish') {
            $value = get_post_field("post_title", $post_id);
        }

        if (empty($value)) {
            return;
        }

        // Sanitize value
        $value = sanitize_title($value);

        // Check unique
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
		$check_sql = "
            SELECT
                ID 
            FROM 
                $wpdb->posts as posts
            INNER JOIN
                $wpdb->postmeta as postmeta
            ON
                posts.ID = postmeta.post_id
            WHERE
                posts.post_type = '" . GEOFLEX_POST_TYPE . "' AND postmeta.meta_key = %s AND postmeta.meta_value = %s AND posts.ID != %d
            LIMIT 1
        ";
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
		$slug_check = $wpdb->get_var($wpdb->prepare($check_sql, Helpers::get_meta_key("slug"), $value, $post_id));

        // Maybe add suffix
        if ($slug_check) {
            $suffix = 2;
            do {
				$alt_value = _truncate_post_slug($value, 200 - (strlen($suffix) + 1)) . "-$suffix";
                // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
				$slug_check = $wpdb->get_var($wpdb->prepare($check_sql, Helpers::get_meta_key("slug"), $alt_value, $post_id));
				++$suffix;
			} while ($slug_check);
            $value = $alt_value;
        }

        return strtolower($value);
    }
    
    /**
     * default_region_post_data
     *
     * Prevent changing post status for default region
     * 
     * @param  mixed $data
     * @return void
     */
    public function default_region_post_data($data, $postarr)
    {
        $key = Helpers::get_meta_key("default", false);

        if (array_key_exists("geoflex_meta_nonce", $_POST)) {
            $is_default = !!array_key_exists($key, $_POST);
        } else {
            $is_default = get_post_meta($postarr["ID"], Helpers::get_meta_key("default"), true);
        }

        if (
            $data["post_type"] === GEOFLEX_POST_TYPE &&
            $data["post_status"] != "publish" &&
            $is_default
        ) {
            $data["post_status"] = "publish";
        }
        return $data;
    }
    
    /**
     * pre_trash_post
     *
     * Add notice when user delete default region
     * 
     * @param  mixed $trash
     * @param  mixed $post
     * @param  mixed $previous_status
     * @return void
     */
    public function pre_trash_post($trash, $post, $previous_status)
    {
        $is_default = get_post_meta($post->ID, Helpers::get_meta_key("default"), true);
        if ($is_default) {
            Notices::add_notice(["status" => "error", "text" => __("Default region can't be moved to trash.", "geoflex")]);
            return $post;
        }
        return $trash;
    }
        
    /**
     * remove_status_actions
     *
     * Remove status menu for default region
     * 
     * @return void
     */
    public function remove_status_actions()
    {
        global $typenow;
        if ($typenow == GEOFLEX_POST_TYPE) { ?>
            <script>
                document.addEventListener("DOMContentLoaded", () => {
                    const checkboxId = "<?php echo esc_js(Helpers::get_meta_key("default", false)); ?>";
                    const defaultCheckbox = document.getElementById(checkboxId);
                    const statusActions = document.querySelector("#misc-publishing-actions .misc-pub-post-status");

                    const checkdefault = () => {
                        if (defaultCheckbox.checked) {
                            statusActions.style.display = "none";
                        } else {
                            statusActions.style.display = "initial";
                        }
                    }

                    if (defaultCheckbox) {
                        checkdefault();
                        defaultCheckbox.addEventListener("change", checkdefault);
                    }
                });
            </script>   
        <?php }
    }
}
