<?php
// Check if class already exists
if (!class_exists('BD_GEOBUDDY_Admin')) :
/**
 * Admin Class
 *
 * Handles admin pages, settings, and provides hooks for addons to register their own settings tabs.
 *
 * @package    Geobuddy
 * @subpackage Geobuddy/admin
 * @author     Naveen Giri
 * 
 * @since 1.0.3
 * 
 * Hook for Addons:
 * Addons can register their own settings tabs using the 'geobuddy_register_settings_tabs' action:
 * 
 * add_action('geobuddy_register_settings_tabs', function($admin_instance) {
 *     $admin_instance->register_settings_tab(array(
 *         'id'       => 'my-addon-tab',
 *         'title'    => __('My Addon', 'textdomain'),
 *         'callback' => 'my_addon_render_tab',
 *         'priority' => 20
 *     ));
 * });
 */
class BD_GEOBUDDY_Admin {

    /**
     * Registered tabs
     *
     * @var array
     */
    private $settings_tabs = array();

    /**
     * Initialize the class
     */
    public function __construct() {
        add_action('admin_menu', array($this, 'add_menu_page'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        
        // Initialize default General tab
        $this->register_default_tabs();
        
        // Allow addons to register their tabs
        add_action('geobuddy_register_settings_tabs', array($this, 'register_addon_tabs'), 10);
    }

    /**
     * Add menu page and submenu pages
     */
    public function add_menu_page() {
        // Main menu page (Welcome page)
        add_menu_page(
            __('GeoBuddy', 'geobuddy'),
            __('GeoBuddy', 'geobuddy'),
            'manage_options',
            'geobuddy',
            array($this, 'render_welcome_page'),
            'dashicons-admin-site',
            25
        );

        // Welcome submenu (same as main page)
        add_submenu_page(
            'geobuddy',
            __('Welcome', 'geobuddy'),
            __('Welcome', 'geobuddy'),
            'manage_options',
            'geobuddy',
            array($this, 'render_welcome_page')
        );

        // Settings submenu
        add_submenu_page(
            'geobuddy',
            __('Settings', 'geobuddy'),
            __('Settings', 'geobuddy'),
            'manage_options',
            'geobuddy-settings',
            array($this, 'render_settings_page')
        );
    }

    /**
     * Enqueue admin assets
     */
    public function enqueue_admin_assets($hook) {
        // Only load on GeoBuddy admin pages
        // if (strpos($hook, 'geobuddy') === false) {
        //     return;
        // }

        wp_enqueue_style(
            'geobuddy-admin',
            BD_GEOBUDDY_PLUGIN_URL . 'admin/css/geobuddy-admin.css',
            array(),
            BD_GEOBUDDY_VERSION
        );

        // Enqueue JavaScript for settings page tabs
        if (strpos($hook, 'geobuddy-settings') !== false) {
            wp_enqueue_script(
                'geobuddy-admin',
                BD_GEOBUDDY_PLUGIN_URL . 'admin/js/geobuddy-admin.js',
                array('jquery'),
                BD_GEOBUDDY_VERSION,
                true
            );
        }
    }

    /**
     * Register settings
     */
    public function register_settings() {
        register_setting(
            'bd_geobuddy_options',
            'bd_geobuddy_custom_fields',
            array(
                'type' => 'array',
                'sanitize_callback' => array($this, 'sanitize_custom_fields'),
                'default' => array(
                    'linkedin' => 1,
                    'whatsapp' => 1,
                    'tiktok' => 1,
                    'youtube' => 1,
                    'skype' => 1,
                    'virtual_tour' => 1
                ),
                'show_in_rest' => false,
            )
        );

        add_settings_section(
            'bd_geobuddy_custom_fields_section',
            __('Social Media Fields', 'geobuddy'),
            array($this, 'section_callback'),
            'geobuddy-settings'
        );

        // Add settings fields
        $custom_fields = array(
            'linkedin'     => __('LinkedIn Profile', 'geobuddy'),
            'whatsapp'     => __('WhatsApp', 'geobuddy'),
            'tiktok'       => __('TikTok Profile', 'geobuddy'),
            'youtube'      => __('YouTube Channel', 'geobuddy'),
            'skype'        => __('Skype', 'geobuddy'),
            'virtual_tour' => __('360° Virtual Tour URL', 'geobuddy')
        );

        foreach ($custom_fields as $field_id => $field_label) {
            add_settings_field(
                'bd_geobuddy_field_' . $field_id,
                $field_label,
                array($this, 'custom_field_callback'),
                'geobuddy-settings',
                'bd_geobuddy_custom_fields_section',
                array('field_id' => $field_id)
            );
        }
    }

    /**
     * Section callback
     */
    public function section_callback() {
        echo '<p>' . esc_html__('Enable or disable social media fields for your website.', 'geobuddy') . '</p>';
    }

    /**
     * Field callback
     */
    public function custom_field_callback($args) {
        $options = get_option('bd_geobuddy_custom_fields', array());
        $field_id = $args['field_id'];
        // Change default behavior - only check if explicitly set to 1
        $checked = isset($options[$field_id]) && $options[$field_id] == 1;
        ?>
        <label>
            <input type="checkbox" 
                   name="bd_geobuddy_custom_fields[<?php echo esc_attr($field_id); ?>]" 
                   value="1" 
                   <?php checked(true, $checked); ?>>
            <?php esc_html_e('Enable this field', 'geobuddy'); ?>
        </label>
        <?php
    }

    /**
     * Sanitize custom fields options
     * 
     * @param mixed $input The value being saved
     * @return array Sanitized array of settings
     */
    public function sanitize_custom_fields($input) {
        if (!is_array($input)) {
            return array();
        }

        $sanitized = array();
        $valid_fields = array(
            'linkedin',
            'whatsapp',
            'tiktok',
            'youtube',
            'skype',
            'virtual_tour'
        );

        foreach ($valid_fields as $field) {
            // Ensure value is either 1 or 0
            $sanitized[$field] = isset($input[$field]) && $input[$field] ? 1 : 0;
        }

        return $sanitized;
    }

    /**
     * Render welcome page
     */
    public function render_welcome_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        // Render header
        $this->render_header();
        
        // Load welcome view
        $this->load_view('welcome');
        
        // Render footer
        $this->render_footer();
    }

    /**
     * Register default tabs
     */
    private function register_default_tabs() {
        // Register General tab
        $this->register_settings_tab(array(
            'id'       => 'general',
            'title'    => __('General', 'geobuddy'),
            'callback' => array($this, 'render_general_tab'),
            'priority' => 10
        ));
    }

    /**
     * Register settings tab
     * 
     * This method allows addons to register their own settings tabs in the GeoBuddy Settings page.
     * 
     * @param array $args {
     *     Tab configuration arguments.
     * 
     *     @type string   $id       Unique identifier for the tab (slug). Required.
     *     @type string   $title    Display title for the tab. Required.
     *     @type callable $callback Function or method to render the tab content. Required.
     *     @type int      $priority Priority/order of the tab (lower = earlier). Default: 20.
     *                              General tab uses priority 10.
     * }
     * @return bool True on success, false on failure.
     * 
     * @example
     * $admin_instance->register_settings_tab(array(
     *     'id'       => 'my-addon',
     *     'title'    => __('My Addon', 'textdomain'),
     *     'callback' => 'my_addon_render_tab',
     *     'priority' => 20
     * ));
     */
    public function register_settings_tab($args) {
        $defaults = array(
            'id'       => '',
            'title'    => '',
            'callback' => '',
            'priority' => 20
        );

        $args = wp_parse_args($args, $defaults);

        if (empty($args['id']) || empty($args['title']) || empty($args['callback'])) {
            return false;
        }

        // Validate callback is callable
        if (!is_callable($args['callback'])) {
            return false;
        }

        $this->settings_tabs[$args['id']] = $args;
        return true;
    }

    /**
     * Get registered tabs
     * 
     * @return array
     */
    public function get_settings_tabs() {
        // Trigger action for addons to register tabs
        do_action('geobuddy_register_settings_tabs', $this);
        
        // Sort tabs by priority
        uasort($this->settings_tabs, function($a, $b) {
            return $a['priority'] - $b['priority'];
        });

        return $this->settings_tabs;
    }

    /**
     * Register addon tabs (called via action hook)
     */
    public function register_addon_tabs($admin_instance) {
        // This method allows addons to call register_settings_tab via the hook
        // Addons will use: do_action('geobuddy_register_settings_tabs', $this);
    }

    /**
     * Render settings page
     */
    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Get active tab from URL
        $active_tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'general';
        
        // Get all registered tabs
        $tabs = $this->get_settings_tabs();
        
        // Validate active tab
        if (!isset($tabs[$active_tab])) {
            $active_tab = 'general';
        }
        
        // Render header
        $this->render_header();
        
        // Load settings tabs view
        $this->load_view('settings-tabs', array(
            'tabs'       => $tabs,
            'active_tab' => $active_tab
        ));
        
        // Render footer
        $this->render_footer();
    }

    /**
     * Render General tab content
     */
    public function render_general_tab() {
        $options = get_option('bd_geobuddy_custom_fields', array());
        $custom_fields = array(
            'linkedin'     => __('LinkedIn Profile', 'geobuddy'),
            'whatsapp'     => __('WhatsApp', 'geobuddy'),
            'tiktok'       => __('TikTok Profile', 'geobuddy'),
            'youtube'      => __('YouTube Channel', 'geobuddy'),
            'skype'        => __('Skype', 'geobuddy'),
            'virtual_tour' => __('360° Virtual Tour URL', 'geobuddy')
        );
        ?>
        <div class="general-settings">
            <form method="post" action="options.php">
                <?php settings_fields('bd_geobuddy_options'); ?>
                
                <h3><?php esc_html_e('Custom Fields Settings', 'geobuddy'); ?></h3>
                <p><?php esc_html_e('Enable or disable custom fields for your GeoDirectory listings.', 'geobuddy'); ?></p>
                
                <table class="form-table" role="presentation">
                    <tbody>
                        <?php foreach ($custom_fields as $field_id => $field_label) : 
                            $checked = isset($options[$field_id]) && $options[$field_id] == 1;
                        ?>
                            <tr>
                                <th scope="row"><?php echo esc_html($field_label); ?></th>
                                <td>
                                    <label>
                                        <input type="checkbox" 
                                               name="bd_geobuddy_custom_fields[<?php echo esc_attr($field_id); ?>]" 
                                               value="1" 
                                               <?php checked(true, $checked); ?>>
                                        <?php esc_html_e('Enable this field', 'geobuddy'); ?>
                                    </label>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
                
                <p class="submit">
                    <?php submit_button(__('Save Changes', 'geobuddy'), 'primary', 'submit', false); ?>
                </p>
            </form>
        </div>
        <?php
    }

    /**
     * Render header
     */
    private function render_header() {
        $this->load_view('header');
    }

    /**
     * Render footer
     */
    private function render_footer() {
        $this->load_view('footer');
    }

    /**
     * Load admin view file
     *
     * @param string $view_name View file name without .php extension
     * @param array  $args      Variables to pass to the view
     */
    private function load_view($view_name, $args = array()) {
        extract($args);
        $view_file = BD_GEOBUDDY_PLUGIN_DIR . 'admin/views/' . $view_name . '.php';
        
        if (file_exists($view_file)) {
            include $view_file;
        } else {
            echo '<div class="error"><p>' . esc_html__('View file not found.', 'geobuddy') . '</p></div>';
        }
    }
}
endif; 