<?php
namespace BGeo;

if (!defined('ABSPATH')) exit;

class Geo {

    /**
     * Get plugin settings (token + fallback country).
     */
    public static function get_settings() {
        $opts = get_option(BGEO_OPTION_KEY, []);
        return [
            'ipinfo_token'     => isset($opts['ipinfo_token']) ? trim($opts['ipinfo_token']) : '',
            'fallback_country' => isset($opts['fallback_country']) ? strtoupper(trim($opts['fallback_country'])) : '',
        ];
    }

    /**
     * Determine client IP (unslash + sanitize each source).
     */
    public static function get_ip() {
        $keys = [
            'HTTP_CF_CONNECTING_IP',
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_REAL_IP',
            'REMOTE_ADDR',
        ];

        foreach ($keys as $k) {
            if (!empty($_SERVER[$k])) {
                // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
                $raw   = wp_unslash($_SERVER[$k]);
                $safe  = sanitize_text_field($raw);
                $first = trim(explode(',', $safe)[0]); // respect first in chain
                if (filter_var($first, FILTER_VALIDATE_IP)) {
                    return $first;
                }
            }
        }

        // Sensible default (won't pass API geo)
        return '0.0.0.0';
    }

    /**
     * Build the IPinfo Lite URL, always adding token as query param.
     * - If $ip is null/empty: use /lite/me
     * - Else: use /lite/{ip}
     */
    private static function build_url($ip = null) {
        $settings = self::get_settings();
        $token    = $settings['ipinfo_token'];

        $base  = 'https://api.ipinfo.io/lite';
        $useIp = is_string($ip) ? trim($ip) : '';
        $url   = ($useIp !== '') ? "{$base}/" . rawurlencode($useIp) : "{$base}/me";

        if ($token !== '') {
            $url = add_query_arg(['token' => $token], $url);
        }

        return $url;
    }

    /**
     * Call IPinfo Lite API and return status/data/raw/request_url (for debugging).
     */
     
     public static function call_ipinfo_lite($ip = null, $timeout = 6) {
    $url  = self::build_url($ip);
    $resp = wp_remote_get($url, ['timeout' => $timeout]); // was fixed 6
    
    $args = [
            'timeout' => 6,
            'headers' => [
                'User-Agent' => 'geo-targeting-for-bricks-builder/' . (defined('BGEO_VERSION') ? BGEO_VERSION : 'dev'),
                'Accept'     => 'application/json',
            ],
        ];

        $resp = wp_remote_get($url, $args);

        if (is_wp_error($resp)) {
            return [
                'error'       => $resp->get_error_message(),
                'source'      => 'error',
                'request_url' => $url,
            ];
        }

        $code = (int) wp_remote_retrieve_response_code($resp);
        $body = wp_remote_retrieve_body($resp);

        // Defensive: JSON may contain keys without quotes in some examples; API returns valid JSON.
        $data = json_decode($body, true);
        if (!is_array($data)) {
            $data = [];
        }

        return [
            'status'      => $code,
            'data'        => $data,
            'raw'         => $body,
            'request_url' => $url,
        ];
    }

    /**
     * Get visitor info directly from API; fall back to default country if set.
     */
    public static function get_visitor_info() {
        $ip   = self::get_ip(); // only used as fallback for 'ip' in payload
        $resp = self::call_ipinfo_lite(null); // /lite/me

        if (!empty($resp['error'])) {
            return self::fallback_payload('error');
        }

        $code = (int) ($resp['status'] ?? 0);
        $data = (array) ($resp['data'] ?? []);

        if ($code >= 200 && $code < 300 && !empty($data['country_code'])) {
            $payload = self::sanitize_payload($data, $ip);
            $payload['source'] = 'api';
            return $payload;
        }

        return self::fallback_payload('http', $resp['raw'] ?? '');
    }

    /**
     * Normalize/sanitize the Lite payload.
     */
    private static function sanitize_payload($data, $fallbackIp) {
        return [
            'ip'             => !empty($data['ip']) ? sanitize_text_field($data['ip']) : $fallbackIp,
            'network'        => !empty($data['network']) ? sanitize_text_field($data['network']) : '',
            'asn'            => !empty($data['asn']) ? sanitize_text_field($data['asn']) : '',
            'as_name'        => !empty($data['as_name']) ? sanitize_text_field($data['as_name']) : '',
            'as_domain'      => !empty($data['as_domain']) ? sanitize_text_field($data['as_domain']) : '',
            'country_code'   => strtoupper(sanitize_text_field($data['country_code'] ?? '')),
            'country'        => !empty($data['country']) ? sanitize_text_field($data['country']) : '',
            'continent_code' => !empty($data['continent_code']) ? sanitize_text_field($data['continent_code']) : '',
            'continent'      => !empty($data['continent']) ? sanitize_text_field($data['continent']) : '',
        ];
    }

    /**
     * Fallback payload when API fails; uses admin default country if set.
     */
    private static function fallback_payload($source = 'fallback', $raw = '') {
        $settings = self::get_settings();
        $fallback = $settings['fallback_country'];

        if ($fallback) {
            return [
                'ip'             => self::get_ip(),
                'network'        => '',
                'asn'            => '',
                'as_name'        => '',
                'as_domain'      => '',
                'country_code'   => $fallback,
                'country'        => '',
                'continent_code' => '',
                'continent'      => '',
                'source'         => 'fallback',
                'raw'            => $raw,
            ];
        }

        // No fallback configured; surface that the country is unknown
        return [
            'ip'           => self::get_ip(),
            'country_code' => '',
            'source'       => $source,
            'raw'          => $raw,
        ];
    }

    /**
     * Convenience: just the country code (e.g., "US").
     */
    public static function get_country_code() {
        $info = self::get_visitor_info();
        return !empty($info['country_code']) ? strtoupper($info['country_code']) : '';
    }

    /**
     * Countries list [code => name].
     */
    public static function countries() {
        return [
            'AF'=>'Afghanistan','AL'=>'Albania','DZ'=>'Algeria','AD'=>'Andorra','AO'=>'Angola','AG'=>'Antigua and Barbuda','AR'=>'Argentina','AM'=>'Armenia','AU'=>'Australia','AT'=>'Austria','AZ'=>'Azerbaijan','BS'=>'Bahamas','BH'=>'Bahrain','BD'=>'Bangladesh','BB'=>'Barbados','BY'=>'Belarus','BE'=>'Belgium','BZ'=>'Belize','BJ'=>'Benin','BT'=>'Bhutan','BO'=>'Bolivia','BA'=>'Bosnia and Herzegovina','BW'=>'Botswana','BR'=>'Brazil','BN'=>'Brunei','BG'=>'Bulgaria','BF'=>'Burkina Faso','BI'=>'Burundi','KH'=>'Cambodia','CM'=>'Cameroon','CA'=>'Canada','CV'=>'Cape Verde','CF'=>'Central African Republic','TD'=>'Chad','CL'=>'Chile','CN'=>'China','CO'=>'Colombia','KM'=>'Comoros','CG'=>'Congo','CD'=>'Congo (DRC)','CR'=>'Costa Rica','CI'=>'Côte d’Ivoire','HR'=>'Croatia','CU'=>'Cuba','CY'=>'Cyprus','CZ'=>'Czechia','DK'=>'Denmark','DJ'=>'Djibouti','DM'=>'Dominica','DO'=>'Dominican Republic','EC'=>'Ecuador','EG'=>'Egypt','SV'=>'El Salvador','GQ'=>'Equatorial Guinea','ER'=>'Eritrea','EE'=>'Estonia','SZ'=>'Eswatini','ET'=>'Ethiopia','FJ'=>'Fiji','FI'=>'Finland','FR'=>'France','GA'=>'Gabon','GM'=>'Gambia','GE'=>'Georgia','DE'=>'Germany','GH'=>'Ghana','GR'=>'Greece','GD'=>'Grenada','GT'=>'Guatemala','GN'=>'Guinea','GW'=>'Guinea-Bissau','GY'=>'Guyana','HT'=>'Haiti','HN'=>'Honduras','HK'=>'Hong Kong','HU'=>'Hungary','IS'=>'Iceland','IN'=>'India','ID'=>'Indonesia','IR'=>'Iran','IQ'=>'Iraq','IE'=>'Ireland','IL'=>'Israel','IT'=>'Italy','JM'=>'Jamaica','JP'=>'Japan','JO'=>'Jordan','KZ'=>'Kazakhstan','KE'=>'Kenya','KI'=>'Kiribati','KW'=>'Kuwait','KG'=>'Kyrgyzstan','LA'=>'Laos','LV'=>'Latvia','LB'=>'Lebanon','LS'=>'Lesotho','LR'=>'Liberia','LY'=>'Libya','LI'=>'Liechtenstein','LT'=>'Lithuania','LU'=>'Luxembourg','MO'=>'Macao','MG'=>'Madagascar','MW'=>'Malawi','MY'=>'Malaysia','MV'=>'Maldives','ML'=>'Mali','MT'=>'Malta','MH'=>'Marshall Islands','MR'=>'Mauritania','MU'=>'Mauritius','MX'=>'Mexico','MD'=>'Moldova','MC'=>'Monaco','MN'=>'Mongolia','ME'=>'Montenegro','MA'=>'Morocco','MZ'=>'Mozambique','MM'=>'Myanmar','NA'=>'Namibia','NR'=>'Nauru','NP'=>'Nepal','NL'=>'Netherlands','NZ'=>'New Zealand','NI'=>'Nicaragua','NE'=>'Niger','NG'=>'Nigeria','KP'=>'North Korea','MK'=>'North Macedonia','NO'=>'Norway','OM'=>'Oman','PK'=>'Pakistan','PW'=>'Palau','PS'=>'Palestine','PA'=>'Panama','PG'=>'Papua New Guinea','PY'=>'Paraguay','PE'=>'Peru','PH'=>'Philippines','PL'=>'Poland','PT'=>'Portugal','QA'=>'Qatar','RO'=>'Romania','RU'=>'Russia','RW'=>'Rwanda','KN'=>'Saint Kitts and Nevis','LC'=>'Saint Lucia','VC'=>'Saint Vincent and the Grenadines','WS'=>'Samoa','SM'=>'San Marino','ST'=>'Sao Tome and Principe','SA'=>'Saudi Arabia','SN'=>'Senegal','RS'=>'Serbia','SC'=>'Seychelles','SL'=>'Sierra Leone','SG'=>'Singapore','SK'=>'Slovakia','SI'=>'Slovenia','SB'=>'Solomon Islands','SO'=>'Somalia','ZA'=>'South Africa','KR'=>'South Korea','SS'=>'South Sudan','ES'=>'Spain','LK'=>'Sri Lanka','SD'=>'Sudan','SR'=>'Suriname','SE'=>'Sweden','CH'=>'Switzerland','SY'=>'Syria','TW'=>'Taiwan','TJ'=>'Tajikistan','TZ'=>'Tanzania','TH'=>'Thailand','TL'=>'Timor-Leste','TG'=>'Togo','TO'=>'Tonga','TT'=>'Trinidad and Tobago','TN'=>'Tunisia','TR'=>'Türkiye','TM'=>'Turkmenistan','TV'=>'Tuvalu','UG'=>'Uganda','UA'=>'Ukraine','AE'=>'United Arab Emirates','GB'=>'United Kingdom','US'=>'United States','UY'=>'Uruguay','UZ'=>'Uzbekistan','VU'=>'Vanuatu','VE'=>'Venezuela','VN'=>'Vietnam','YE'=>'Yemen','ZM'=>'Zambia','ZW'=>'Zimbabwe'
        ];
    }
}