<?php
namespace BGeo;

if (!defined('ABSPATH')) exit;

class Conditions {

  

    public static function init() {
        // Register the group & option in Bricks builder
        add_filter('bricks/conditions/groups',  [__CLASS__, 'add_group']);
        add_filter('bricks/conditions/options', [__CLASS__, 'add_option']);

        // Server-side evaluation (frontend only)
        add_filter('bricks/conditions/result',  [__CLASS__, 'evaluate'], 10, 3);

    }

    /**
     * Add our condition group to the list in Bricks
     */
    public static function add_group($groups) {
        $groups[] = [
            'name'  => 'bgeo_group',
            'label' => esc_html__('Geo (Bricks Geo Targeting)', 'geo-targeting-for-bricks-builder'),
        ];
        return $groups;
    }

    /**
     * Register the "Visitor Country" condition with compare + multi-select value
     */
    public static function add_option($options) {
        $countries = Geo::countries(); // [code => name]

        $options[] = [
            'key'   => 'bgeo_country',
            'label' => esc_html__('Visitor Country', 'geo-targeting-for-bricks-builder'),
            'group' => 'bgeo_group',

            // Compare operator selection
            'compare' => [
                'type'        => 'select',
                'options'     => [
                    'in'     => esc_html__('in', 'geo-targeting-for-bricks-builder'),
                    'not_in' => esc_html__('not in', 'geo-targeting-for-bricks-builder'),
                ],
                'placeholder' => esc_html__('in', 'geo-targeting-for-bricks-builder'),
            ],

            // Multi-select of ISO country codes
            'value' => [
                'type'        => 'select',
                'options'     => $countries,
                'multiple'    => true,
                'placeholder' => esc_html__('Select countries…', 'geo-targeting-for-bricks-builder'),
            ],
        ];

        return $options;
    }

    /**
     * Evaluate the condition on the frontend
     *
     * @param bool   $result         Previous result
     * @param string $condition_key  Our condition key (bgeo_country)
     * @param array  $condition      ['compare'=>'in|not_in','value'=>array|string]
     * @return bool
     */
    public static function evaluate($result, $condition_key, $condition) {
        if ($condition_key !== 'bgeo_country') {
            return $result;
        }

        // Compare operator defaults to "in"
        $compare = !empty($condition['compare']) ? $condition['compare'] : 'in';

        // Normalize selected values to array of uppercase codes
        $values = $condition['value'] ?? [];
        if (!is_array($values)) {
            $values = array_filter(array_map('trim', explode(',', (string) $values)));
        }
        $values = array_map('strtoupper', $values);

        // Resolve visitor country via API; if empty, use fallback from settings
        $visitor = Geo::get_country_code();
        if (!$visitor) {
            $opts    = Geo::get_settings();
            $fallback = isset($opts['fallback_country']) ? strtoupper(trim($opts['fallback_country'])) : '';
            $visitor  = $fallback ?: '';
        }

        // If we still don't have a visitor code, fail closed (no render)
        if (!$visitor) {
            return false;
        }

        $visitor = strtoupper($visitor);
        $in_list = in_array($visitor, $values, true);

        switch ($compare) {
            case 'not_in':
                return !$in_list;
            case 'in':
            default:
                return $in_list;
        }
    }
}