<?php
namespace BGeo;

if (!defined('ABSPATH')) exit;

class Admin {

    public static function init() {
        // Run after Bricks adds its own menu.
        add_action('admin_menu', [__CLASS__, 'register_menu'], 100);
        add_action('admin_init', [__CLASS__, 'register_settings']);
        add_action('admin_enqueue_scripts', [__CLASS__, 'enqueue_admin']);
        add_action('wp_ajax_bgeo_test_ipinfo', [__CLASS__, 'ajax_test_ipinfo']);
    }

    /**
     * Add submenu under Bricks → after License
     */
public static function register_menu() {

    // Bricks is NOT installed or active
    if ( ! defined( 'BRICKS_VERSION' ) ) {

        // Create a hidden page so the URL does not show "Sorry, you are not allowed..."
        add_submenu_page(
            null,
            __( 'Geo Targeting for Bricks', 'geo-targeting-for-bricks-builder' ),
            __( 'Geo Targeting', 'geo-targeting-for-bricks-builder' ),
            'manage_options',
            'bgeo-settings',
            [ __CLASS__, 'render_bricks_missing_page' ]
        );

        return;
    }

    // Bricks is active → normal behavior
    $cap = apply_filters('bgeo_capability', 'edit_theme_options');

    add_submenu_page(
        'bricks',
        esc_html__('Bricks Geo Targeting', 'geo-targeting-for-bricks-builder'),
        esc_html__('Geo Targeting', 'geo-targeting-for-bricks-builder'),
        $cap,
        'bgeo-settings',
        [ __CLASS__, 'render_settings_page' ],
        12
    );
}

    /**
     * Register settings and fields
     */
    public static function register_settings() {
        register_setting(BGEO_OPTION_KEY, BGEO_OPTION_KEY, [
            'type'              => 'array',
            'sanitize_callback' => [__CLASS__, 'sanitize'],
        ]);

        add_settings_section(
            'bgeo_main',
            esc_html__('IPinfo Lite Settings', 'geo-targeting-for-bricks-builder'),
            function () {
                echo '<p>' . esc_html__(
                    'Enter your IPinfo Lite API Token. The plugin calls the /lite endpoint (with /me) on each request. If the API fails, the Default Country below is used.',
                    'geo-targeting-for-bricks-builder'
                ) . '</p>';
            },
            'bgeo-settings'
        );

        add_settings_field(
            'ipinfo_token',
            esc_html__('IPinfo Lite Token', 'geo-targeting-for-bricks-builder'),
            [__CLASS__, 'field_token'],
            'bgeo-settings',
            'bgeo_main'
        );

        add_settings_field(
            'fallback_country',
            esc_html__('Default Country (fallback)', 'geo-targeting-for-bricks-builder'),
            [__CLASS__, 'field_fallback_country'],
            'bgeo-settings',
            'bgeo_main'
        );
    }

    /**
     * Sanitize options
     */
    public static function sanitize($opts) {
        $countries = array_keys(Geo::countries());
        $fallback  = isset($opts['fallback_country']) ? strtoupper(sanitize_text_field($opts['fallback_country'])) : '';
        if ($fallback && !in_array($fallback, $countries, true)) {
            $fallback = '';
        }
        return [
            'ipinfo_token'     => isset($opts['ipinfo_token']) ? sanitize_text_field($opts['ipinfo_token']) : '',
            'fallback_country' => $fallback,
        ];
    }

    /**
     * Settings fields
     */
    public static function field_token() {
        $opts = Geo::get_settings();
        ?>
        <input type="text"
               name="<?php echo esc_attr(BGEO_OPTION_KEY); ?>[ipinfo_token]"
               value="<?php echo esc_attr($opts['ipinfo_token']); ?>"
               class="regular-text"
               placeholder="<?php echo esc_attr__('Enter Your IPinfo Lite Token Here...', 'geo-targeting-for-bricks-builder'); ?>" />
        <p class="description">
            <a href="https://ipinfo.io/dashboard" target="_blank" rel="noopener"><?php esc_html_e('Get your free Lite token from ipinfo.io', 'geo-targeting-for-bricks-builder'); ?></a>.
        </p>
        <?php
    }

    public static function field_fallback_country() {
        $opts      = Geo::get_settings();
        $countries = Geo::countries();
        ?>
        <select name="<?php echo esc_attr(BGEO_OPTION_KEY); ?>[fallback_country]" class="regular-text" style="max-width:320px;">
            <option value=""><?php esc_html_e('— None (no fallback) —', 'geo-targeting-for-bricks-builder'); ?></option>
            <?php foreach ($countries as $code => $label): ?>
                <option value="<?php echo esc_attr($code); ?>" <?php selected($opts['fallback_country'], $code); ?>>
                    <?php echo esc_html($label . ' (' . $code . ')'); ?>
                </option>
            <?php endforeach; ?>
        </select>
        <p class="description"><?php esc_html_e('If IPinfo Lite fails, this ISO country code will be used for conditions.', 'geo-targeting-for-bricks-builder'); ?></p>
        <?php
    }

    /**
     * Compute a real status for the badge
     * - Returns array: ['slug','label','class','detail']
     */
    private static function compute_status() {
        $s = Geo::get_settings();
        $token    = trim($s['ipinfo_token']);
        $fallback = trim($s['fallback_country']);

        // No token at all
        if ($token === '') {
            if ($fallback !== '') {
                return [
                    'slug'  => 'fallback-only',
                    'label' => esc_html__('Fallback only', 'geo-targeting-for-bricks-builder'),
                    'class' => 'bgeo-badge--warn',
                    'detail'=> esc_html__('Token not set; using default country when needed.', 'geo-targeting-for-bricks-builder'),
                ];
            }
            return [
                'slug'  => 'not-configured',
                'label' => esc_html__('Not configured', 'geo-targeting-for-bricks-builder'),
                'class' => 'bgeo-badge--error',
                'detail'=> esc_html__('Add an IPinfo Lite token to enable live geo detection.', 'geo-targeting-for-bricks-builder'),
            ];
        }

        // Token exists: optionally ping to validate
        $do_ping = apply_filters('bgeo_status_ping', true);
        if ($do_ping) {
            $resp = Geo::call_ipinfo_lite(null, 3); // quick 3s timeout
            $ok   = is_array($resp) && !empty($resp['status']) && (int)$resp['status'] >= 200 && (int)$resp['status'] < 300;

            if ($ok && !empty($resp['data']['country_code'])) {
                return [
                    'slug'  => 'active',
                    'label' => esc_html__('Active', 'geo-targeting-for-bricks-builder'),
                    'class' => 'bgeo-badge--ok',
                    'detail'=> esc_html__('Token valid. Live geo detection is working.', 'geo-targeting-for-bricks-builder'),
                ];
            }

            if ($fallback !== '') {
                return [
                    'slug'  => 'degraded',
                    'label' => esc_html__('Degraded (fallback)', 'geo-targeting-for-bricks-builder'),
                    'class' => 'bgeo-badge--warn',
                    'detail'=> esc_html__('Token set but API not reachable. Using default country.', 'geo-targeting-for-bricks-builder'),
                ];
            }

            return [
                'slug'  => 'token-issue',
                'label' => esc_html__('Token issue', 'geo-targeting-for-bricks-builder'),
                'class' => 'bgeo-badge--error',
                'detail'=> esc_html__('Token set but API not reachable and no fallback configured.', 'geo-targeting-for-bricks-builder'),
            ];
        }

        // If you disable ping, show "Configured" when token exists
        return [
            'slug'  => 'configured',
            'label' => esc_html__('Configured', 'geo-targeting-for-bricks-builder'),
            'class' => 'bgeo-badge--ok',
            'detail'=> esc_html__('Token present. Status ping disabled.', 'geo-targeting-for-bricks-builder'),
        ];
    }
    
    
    
    public static function render_bricks_missing_page() {

    echo '<div class="wrap">';
    echo '<h1>' . esc_html__( 'Bricks Builder Required', 'geo-targeting-for-bricks-builder' ) . '</h1>';

    echo '<div class="notice notice-error" style="padding:15px; margin-top:20px;">';
    echo '<p><strong>' . esc_html__( 'Bricks Builder is not active on this site.', 'geo-targeting-for-bricks-builder' ) . '</strong></p>';
    echo '<p>' . esc_html__( 'The Geo Targeting for Bricks plugin requires Bricks Builder. Since Bricks is not available, this plugin cannot function.', 'geo-targeting-for-bricks-builder' ) . '</p>';
    echo '</div>';

    echo '<p style="margin-top:20px;">' . esc_html__( 'The plugin has been automatically deactivated.', 'geo-targeting-for-bricks-builder' ) . '</p>';
    echo '</div>';

    // Auto-deactivate plugin safely
    deactivate_plugins( plugin_basename( BGEO_PLUGIN_FILE ) );
}




    /**
     * Render settings page
     */
    public static function render_settings_page() {
        $ajax_url = admin_url('admin-ajax.php');
        $nonce    = wp_create_nonce('bgeo_admin');

        $status = self::compute_status();
        ?>
        <div class="wrap bgeo-wrap">
            <h1><?php esc_html_e('Bricks Geo Targeting - Inspired Monks', 'geo-targeting-for-bricks-builder'); ?></h1>


            <div class="bgeo-grid">
                <!-- Left card: Settings -->
                <div class="bgeo-card">
                    <div class="bgeo-title-row">
                        <h2><?php esc_html_e('Settings', 'geo-targeting-for-bricks-builder'); ?></h2>
                        <span class="bgeo-badge <?php echo esc_attr($status['class']); ?>">
                            <?php echo esc_html($status['label']); ?>
                        </span>
                    </div>
                    <?php if (!empty($status['detail'])): ?>
                        <p class="bgeo-status-detail"><?php echo esc_html($status['detail']); ?></p>
                    <?php endif; ?>

                    <form class="bgeo-settings" method="post" action="options.php">
                        <?php
                        settings_fields(BGEO_OPTION_KEY);
                        do_settings_sections('bgeo-settings');
                        submit_button(esc_html__('Save Settings', 'geo-targeting-for-bricks-builder'));
                        ?>
                    </form>
                </div>

                <!-- Right card: Tester -->
<!-- Right card: Tester -->
<div class="bgeo-card">
  <div class="bgeo-title-row">
    <h2><?php esc_html_e('Test Lookup', 'geo-targeting-for-bricks-builder'); ?></h2>
    <span class="bgeo-badge bgeo-badge--muted"><?php esc_html_e('IPinfo Lite', 'geo-targeting-for-bricks-builder'); ?></span>
  </div>

  <p class="bgeo-inline-help">
    <?php esc_html_e('Use “Test IP” to query a specific IP (uses /lite/{ip}). Or click “Test My IP” (uses /lite/me).', 'geo-targeting-for-bricks-builder'); ?>
  </p>

  <div class="bgeo-actions" style="margin:10px 0;">
    <input type="text" id="bgeo-test-ip" placeholder="e.g. 8.8.8.8" class="regular-text" style="max-width:240px;">
    <button type="button" class="button button-primary" id="bgeo-test-btn"><?php esc_html_e('Test IP', 'geo-targeting-for-bricks-builder'); ?></button>
    <button type="button" class="button" id="bgeo-test-my-btn"><?php esc_html_e('Test My IP', 'geo-targeting-for-bricks-builder'); ?></button>
  </div>

  <!-- Result UI (no JSON) -->
  <div id="bgeo-test-result" class="bgeo-result" style="display:none;">
    <div class="bgeo-result-head">
      <span id="bgeo-status-pill" class="bgeo-pill bgeo-pill--muted">—</span>
      <span id="bgeo-status-text" class="bgeo-status-text">—</span>
    </div>

    <div class="bgeo-kv">
      <div class="bgeo-kv__row">
        <div class="bgeo-kv__key"><?php esc_html_e('IP', 'geo-targeting-for-bricks-builder'); ?></div>
        <div class="bgeo-kv__val" id="kv-ip">—</div>
      </div>
      <div class="bgeo-kv__row">
        <div class="bgeo-kv__key"><?php esc_html_e('ASN', 'geo-targeting-for-bricks-builder'); ?></div>
        <div class="bgeo-kv__val" id="kv-asn">—</div>
      </div>
      <div class="bgeo-kv__row">
        <div class="bgeo-kv__key"><?php esc_html_e('AS Name', 'geo-targeting-for-bricks-builder'); ?></div>
        <div class="bgeo-kv__val" id="kv-asname">—</div>
      </div>
      <div class="bgeo-kv__row">
        <div class="bgeo-kv__key"><?php esc_html_e('AS Domain', 'geo-targeting-for-bricks-builder'); ?></div>
        <div class="bgeo-kv__val" id="kv-asdomain">—</div>
      </div>
      <div class="bgeo-kv__row">
        <div class="bgeo-kv__key"><?php esc_html_e('Country Code', 'geo-targeting-for-bricks-builder'); ?></div>
        <div class="bgeo-kv__val" id="kv-cc">—</div>
      </div>
      <div class="bgeo-kv__row">
        <div class="bgeo-kv__key"><?php esc_html_e('Country', 'geo-targeting-for-bricks-builder'); ?></div>
        <div class="bgeo-kv__val" id="kv-country">—</div>
      </div>
      <div class="bgeo-kv__row">
        <div class="bgeo-kv__key"><?php esc_html_e('Continent Code', 'geo-targeting-for-bricks-builder'); ?></div>
        <div class="bgeo-kv__val" id="kv-continent-code">—</div>
      </div>
      <div class="bgeo-kv__row">
        <div class="bgeo-kv__key"><?php esc_html_e('Continent', 'geo-targeting-for-bricks-builder'); ?></div>
        <div class="bgeo-kv__val" id="kv-continent">—</div>
      </div>
    </div>
  </div>
</div>


            </div>

            <div class="bgeo-sep"></div>

<!-- How To Use (Optimized UI) -->
<div class="bgeo-card bgeo-howto is-compact" style="margin-top:10px;">
  <div class="bgeo-howto-head">
    <h2 class="bgeo-howto-title"><?php esc_html_e('How to Use', 'geo-targeting-for-bricks-builder'); ?></h2>
    <span class="bgeo-pill"><?php esc_html_e('Quick Start', 'geo-targeting-for-bricks-builder'); ?></span>
  </div>

  <div class="bgeo-steps">
    <div class="bgeo-step">
      <span class="dashicons dashicons-admin-site-alt3"></span>
      <h4><?php esc_html_e('Install & Set Up', 'geo-targeting-for-bricks-builder'); ?></h4>
      <ol>
        <li><?php esc_html_e('Go to Bricks → Geo Targeting.', 'geo-targeting-for-bricks-builder'); ?></li>
        <li><?php esc_html_e('Paste your IPinfo Lite token and Save.', 'geo-targeting-for-bricks-builder'); ?></li>
        <li><?php esc_html_e('Optionally set a Default Country (fallback).', 'geo-targeting-for-bricks-builder'); ?></li>
        <li><?php esc_html_e('Use Test Lookup to verify detection.', 'geo-targeting-for-bricks-builder'); ?></li>
      </ol>
    </div>

    <div class="bgeo-step">
      <span class="dashicons dashicons-visibility"></span>
      <h4><?php esc_html_e('Add a Country Rule', 'geo-targeting-for-bricks-builder'); ?></h4>
      <ol>
        <li><?php esc_html_e('Open Bricks Builder and select an element.', 'geo-targeting-for-bricks-builder'); ?></li>
        <li><?php esc_html_e('Conditions → Add Condition → Visitor Country.', 'geo-targeting-for-bricks-builder'); ?></li>
        <li><?php esc_html_e('Choose operator: "in" or "not in".', 'geo-targeting-for-bricks-builder'); ?></li>
        <li><?php esc_html_e('Pick one or more countries, then save.', 'geo-targeting-for-bricks-builder'); ?></li>
      </ol>
      <p class="bgeo-note">
        <?php esc_html_e('Server-side: If the condition fails, the element HTML is not rendered.', 'geo-targeting-for-bricks-builder'); ?>
      </p>
    </div>

    <div class="bgeo-step">
      <span class="dashicons dashicons-admin-page"></span>
      <h4><?php esc_html_e('Copy / Paste & Badges', 'geo-targeting-for-bricks-builder'); ?></h4>
      <ul class="bgeo-bullets">
        <li><?php esc_html_e('Copy/Paste: Use the mini Geo toolbar in Bricks to clone rules between elements.', 'geo-targeting-for-bricks-builder'); ?></li>
        <li><?php esc_html_e('Inline badges: Structure panel shows "Country: US, IN" for quick scanning.', 'geo-targeting-for-bricks-builder'); ?></li>
      </ul>
    </div>
  </div>

  <div class="bgeo-guides">
    <div class="bgeo-guide">
      <span class="bgeo-guide-title"><?php esc_html_e('Fallback & Behavior', 'geo-targeting-for-bricks-builder'); ?></span>
      <ul>
        <li><?php esc_html_e('No cookies, no transients — fresh API lookup per request.', 'geo-targeting-for-bricks-builder'); ?></li>
        <li><?php esc_html_e('If IPinfo Lite fails, the Default Country is used (if set).', 'geo-targeting-for-bricks-builder'); ?></li>
        <li><?php esc_html_e('If no fallback is set and API fails, the element will not render.', 'geo-targeting-for-bricks-builder'); ?></li>
      </ul>
    </div>

    <div class="bgeo-guide">
      <span class="bgeo-guide-title"><?php esc_html_e('Tips', 'geo-targeting-for-bricks-builder'); ?></span>
      <ul>
        <li><?php esc_html_e('Compose with other Bricks conditions (role, date/time) for AND/OR logic.', 'geo-targeting-for-bricks-builder'); ?></li>
        <li><?php esc_html_e('Duplicate localized sections with complementary rules (e.g., US vs non-US).', 'geo-targeting-for-bricks-builder'); ?></li>
        <li><?php esc_html_e('Use Test Lookup with a known IP to validate country logic.', 'geo-targeting-for-bricks-builder'); ?></li>
      </ul>
    </div>
  </div>

  <div class="bgeo-faq">
    <details>
      <summary><?php esc_html_e('The builder doesn’t hide/show my element', 'geo-targeting-for-bricks-builder'); ?></summary>
      <p><?php esc_html_e('Expected: Bricks does not apply conditions in the builder canvas. Check the front end or use Test Lookup.', 'geo-targeting-for-bricks-builder'); ?></p>
    </details>
    <details>
      <summary><?php esc_html_e('Wrong country detected', 'geo-targeting-for-bricks-builder'); ?></summary>
      <p><?php esc_html_e('VPN, CDN, or proxies can affect IP. Test with /lite/{ip} using a known IP.', 'geo-targeting-for-bricks-builder'); ?></p>
    </details>
    <details>
      <summary><?php esc_html_e('Geo toolbar not visible', 'geo-targeting-for-bricks-builder'); ?></summary>
      <p><?php esc_html_e('Refresh the builder and confirm the plugin is active and its assets aren’t blocked by caching/CDN.', 'geo-targeting-for-bricks-builder'); ?></p>
    </details>
  </div>
</div>



<div class="bgeo-footer">
    <?php
    $made_by_string = 
    /* translators: %s: URL of Inspired Monks website. */
    __(
        'Made with ❤️ by <a href="%s" target="_blank" rel="noopener">Inspired Monks</a>',
        'geo-targeting-for-bricks-builder'
    );

    echo wp_kses(
        sprintf( $made_by_string, esc_url( 'https://inspiredmonks.com' ) ),
        [ 'a' => [ 'href' => [], 'target' => [], 'rel' => [] ] ]
    );
    ?>

    <div class="bgeo-donate-wrap">
        <?php
        echo wp_kses(
            sprintf(
                /* translators: %s: Donation page URL. */
                __(
                    '<a href="%s" target="_blank" rel="noopener" class="bgeo-donate-btn">💖 Support Development</a>',
                    'geo-targeting-for-bricks-builder'
                ),
                esc_url( 'https://pages.razorpay.com/inspiredmonks' )
            ),
            [
                'a' => [
                    'href'   => [],
                    'target' => [],
                    'rel'    => [],
                    'class'  => []
                ]
            ]
        );
        ?>
    </div>

    <div class="bgeo-footer-note">
        <?php esc_html_e( 'Thank you for helping us build open-source tools for the WordPress community 💙', 'geo-targeting-for-bricks-builder' ); ?>
    </div>

    <div class="bgeo-footer-logo">
        <a href="https://inspiredmonks.com" target="_blank" rel="noopener">
            <!-- phpcs:ignore PluginCheck.CodeAnalysis.ImageFunctions.NonEnqueuedImage -->
            <img 
                src="<?php echo esc_url( BGEO_PLUGIN_URL . 'assets/images/Inspired-Monks-Logo.webp' ); ?>" 
                alt="<?php esc_attr_e( 'Inspired Monks Logo', 'geo-targeting-for-bricks-builder' ); ?>"
                width="200" height="50"
                class="bgeo-logo-img"
            >
        </a>
    </div>
</div>



        </div>

        <?php
    }

    /**
     * Enqueue admin assets for our settings screen
     */
public static function enqueue_admin( $hook ) {
    if ( $hook !== 'bricks_page_bgeo-settings' ) {
        return;
    }

    wp_enqueue_style(
        'bgeo-admin',
        BGEO_PLUGIN_URL . 'assets/admin.css',
        [],
        BGEO_VERSION
    );

    wp_register_script(
        'bgeo-admin',
        BGEO_PLUGIN_URL . 'assets/admin.js',
        ['jquery'],
        BGEO_VERSION,
        true
    );

    wp_localize_script(
        'bgeo-admin',
        'BGEO_ADMIN',
        [
            'ajax_url' => esc_url( admin_url( 'admin-ajax.php' ) ),
            'nonce'    => wp_create_nonce( 'bgeo_admin' ),
        ]
    );

    wp_enqueue_script('bgeo-admin');
}

    /**
     * AJAX: Test IPinfo Lite (/lite/{ip} or /lite/me)
     */
    public static function ajax_test_ipinfo() {
        check_ajax_referer('bgeo_admin', 'nonce');

        $ip = filter_input(INPUT_POST, 'ip', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
        $ip = $ip ? trim($ip) : '';

        $resp = Geo::call_ipinfo_lite($ip !== '' ? $ip : null);
        $resp['_debug'] = [
            'posted_ip'   => $ip,
            'request_url' => isset($resp['request_url']) ? $resp['request_url'] : '',
        ];

        wp_send_json_success($resp);
    }
}