<?php
/**
 * Plugin Name: Geo Targeting for Bricks Builder by Inspired Monks
 * Description: Add country-based visibility conditions to Bricks Builder elements — show or hide content based on visitor location, automatically and without coding.
 * Version: 1.5.1
 * Author: Inspired Monks
 * Author URI: https://inspiredmonks.com
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: geo-targeting-for-bricks-builder
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

define( 'BGEO_PLUGIN_FILE', __FILE__ );
define( 'BGEO_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'BGEO_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'BGEO_OPTION_KEY', 'bgeo_settings' );
define( 'BGEO_VERSION', '1.5.0' );

require_once BGEO_PLUGIN_DIR . 'inc/Geo.php';
require_once BGEO_PLUGIN_DIR . 'inc/Admin.php';
require_once BGEO_PLUGIN_DIR . 'inc/Conditions.php';

add_action( 'plugins_loaded', function() {
	\BGeo\Admin::init();

	add_action( 'init', function() {
		// Only register Bricks condition if Bricks exists
		if ( class_exists( 'Bricks\Frontend' ) || function_exists( 'bricks_is_builder' ) ) {
			\BGeo\Conditions::init();
		}
	} );
} );

/**
 * Activation redirect -> Settings page
 */
register_activation_hook( __FILE__, function() {
	add_option( 'bgeo_do_activation_redirect', 1 );
} );

add_action( 'admin_init', function() {
	if ( get_option( 'bgeo_do_activation_redirect' ) ) {
		delete_option( 'bgeo_do_activation_redirect' );

		// Sanitize GET param (no action taken other than check)
		$activate_multi = filter_input( INPUT_GET, 'activate-multi', FILTER_SANITIZE_FULL_SPECIAL_CHARS );

		if ( ! $activate_multi ) {
			wp_safe_redirect( admin_url( 'admin.php?page=bgeo-settings' ) );
			exit;
		}
	}
} );

/**
 * Add "Settings" link on the Plugins page row
 */
add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), function( $links ) {
	$settings_link = '<a href="' . esc_url( admin_url( 'admin.php?page=bgeo-settings' ) ) . '">' . esc_html__( 'Settings', 'geo-targeting-for-bricks-builder' ) . '</a>';
	array_unshift( $links, $settings_link );
	return $links;
} );

/**
 * Enqueue builder-only UI (badges + copy/paste)
 */
add_action( 'bricks_enqueue_builder_scripts', function() {
	wp_enqueue_script(
		'bgeo-builder',
		BGEO_PLUGIN_URL . 'assets/builder.js',
		[],
		BGEO_VERSION,
		true
	);

	wp_enqueue_style(
		'bgeo-builder',
		BGEO_PLUGIN_URL . 'assets/builder.css',
		[],
		BGEO_VERSION
	);

	// Pass countries to JS (may be used by builder UI)
	if ( class_exists( '\\BGeo\\Geo' ) ) {
		wp_add_inline_script(
			'bgeo-builder',
			'window.BGEO_COUNTRIES=' . wp_json_encode( \BGeo\Geo::countries() ) . ';',
			'before'
		);
	}
} );