// assets/builder.js
(function () {
  'use strict';

  // Abort if not in Bricks builder (avoid running on frontend/admin screens)
  var isBuilder =
    document.body.classList.contains('bricks-builder') ||
    document.getElementById('bricks-builder') ||
    (window.bricksBuilder || window.bricks?.builder || window.brxBuilder);

  if (!isBuilder) return;

  // -------------- helpers -----------------
  var log  = function(){ try{ console.log.apply(console, ['[BGeo]'].concat([].slice.call(arguments))); }catch(e){} };
  var dbg  = function(){ try{ console.debug.apply(console, ['[BGeo]'].concat([].slice.call(arguments))); }catch(e){} };
  var warn = function(){ try{ console.warn.apply(console, ['[BGeo]'].concat([].slice.call(arguments))); }catch(e){} };

  var $    = function(sel, root){ return (root || document).querySelector(sel); };
  var $all = function(sel, root){ return Array.from((root || document).querySelectorAll(sel)); };

  function deepClone(obj) {
    try { return JSON.parse(JSON.stringify(obj)); } catch(e) { return null; }
  }

  function ensureArray(val) {
    if (Array.isArray(val)) return val;
    if (val == null || val === '') return [];
    return String(val).split(',').map(function(s){ return s.trim(); }).filter(Boolean);
  }

  // i18n (optionally injected from PHP)
  var I18N = window.BGEO_I18N || {
    copy: 'Copy Geo',
    paste: 'Paste Geo',
    toastSelectFirst: 'Select an element first',
    toastNoGeo: 'No Geo condition on this element',
    toastCopied: 'Geo condition copied',
    toastNothing: 'Nothing to paste',
    toastPasted: 'Geo condition pasted',
    badgePrefix: 'Country:',
    opIn: 'IN',
    opNotIn: 'NOT IN'
  };

  // -------------- Bricks helpers -----------------
  function getBuilder() {
    return window.bricksBuilder || window.bricks?.builder || window.brxBuilder || null;
  }

  function getSelectedElement() {
    var b = getBuilder();
    if (!b) return null;

    // Common properties across versions
    if (b.selectedElement) return b.selectedElement;
    if (b.getSelectedElement && typeof b.getSelectedElement === 'function') return b.getSelectedElement();
    if (b.store && b.store.selectedElement) return b.store.selectedElement;

    return null;
  }

  function getElementById(id) {
    var b = getBuilder();
    if (!b || !id) return null;

    // Try internal maps / stores
    if (b.elements && b.elements[id]) return b.elements[id];
    if (b.getElement && typeof b.getElement === 'function') {
      try { return b.getElement(id); } catch(e) {}
    }
    if (b.store && b.store.elements && b.store.elements[id]) return b.store.elements[id];

    // Last resort: try to find a node with dataset and read a backing object
    return null;
  }

  function getElementConditions(el) {
    var s = el && el.settings ? el.settings : null;
    return s && Array.isArray(s.conditions) ? s.conditions : [];
  }

  function setElementConditions(el, newSets) {
    if (!el || !el.settings) return;
    el.settings.conditions = newSets;
  }

  function renderElement(el) {
    var b = getBuilder();
    if (b && typeof b.renderElement === 'function') {
      try { b.renderElement(el); } catch (e) {}
    }
  }

  function toast(msg) {
    var t = document.createElement('div');
    t.className = 'bgeo-toast';
    t.setAttribute('role', 'status');
    t.setAttribute('aria-live', 'polite');
    t.textContent = msg;
    document.body.appendChild(t);
    requestAnimationFrame(function(){
      t.classList.add('in');
      setTimeout(function(){
        t.classList.remove('in');
        setTimeout(function(){ if (t && t.parentNode) t.parentNode.removeChild(t); }, 200);
      }, 1800);
    });
  }

  // -------------- Geo condition utils -----------------
  var GEO_KEY = 'bgeo_country';
  var COPY_STORAGE_KEY = 'bgeo_copied_geo_condition';

  function findGeoConditionsInSets(sets) {
    var matches = [];
    sets.forEach(function(set, idx){
      if (!Array.isArray(set)) return;
      set.forEach(function(cond, cIdx){
        if (cond && cond.key === GEO_KEY) {
          matches.push({ setIndex: idx, condIndex: cIdx, cond: cond });
        }
      });
    });
    return matches;
  }

  function readGeoConditionFromElement(el) {
    var sets = getElementConditions(el);
    var found = findGeoConditionsInSets(sets);
    if (!found.length) return null;

    // If multiple, copy the first occurrence
    var first   = found[0].cond || {};
    var compare = first.compare || 'in';
    var value   = ensureArray(first.value).map(function(v){ return String(v).toUpperCase(); });

    return { key: GEO_KEY, compare: compare, value: value };
  }

  function normalizePayload(payload) {
    if (!payload || typeof payload !== 'object') return null;

    var compare = payload.compare === 'not_in' ? 'not_in' : 'in';
    var value   = ensureArray(payload.value).map(function(v){ return String(v).toUpperCase(); })
                                            .filter(Boolean);

    return { key: GEO_KEY, compare: compare, value: value };
  }

  function writeGeoConditionToElement(el, payload, opts) {
    opts = opts || {};
    var replaceExisting = opts.replaceExisting !== false;

    if (!el || !el.settings) return false;

    var norm = normalizePayload(payload);
    if (!norm) return false;

    var sets = deepClone(getElementConditions(el));
    if (!Array.isArray(sets) || !sets.length) {
      sets = [[]]; // Ensure at least one condition set
    }

    if (replaceExisting) {
      var matches = findGeoConditionsInSets(sets);
      if (matches.length) {
        matches.forEach(function(m){
          sets[m.setIndex][m.condIndex] = deepClone(norm);
        });
      } else {
        sets[0].push(deepClone(norm));
      }
    } else {
      sets[0].push(deepClone(norm));
    }

    setElementConditions(el, sets);
    renderElement(el);
    return true;
  }

  // -------------- Floating buttons (copy/paste) -----------------
  function ensureToolbar() {
    if ($('.bgeo-toolbar')) return;

    var bar = document.createElement('div');
    bar.className = 'bgeo-toolbar';
    bar.setAttribute('aria-label', 'Geo condition tools');

    var copyBtn = document.createElement('button');
    copyBtn.type = 'button';
    copyBtn.className = 'bgeo-btn';
    copyBtn.textContent = I18N.copy;

    var pasteBtn = document.createElement('button');
    pasteBtn.type = 'button';
    pasteBtn.className = 'bgeo-btn';
    pasteBtn.textContent = I18N.paste;

    copyBtn.addEventListener('click', function(){
      var el = getSelectedElement();
      if (!el) { toast(I18N.toastSelectFirst); return; }
      var cond = readGeoConditionFromElement(el);
      if (!cond) { toast(I18N.toastNoGeo); return; }
      try {
        localStorage.setItem(COPY_STORAGE_KEY, JSON.stringify(cond));
        toast(I18N.toastCopied);
      } catch(e) { /* storage might be disabled */ }
    });

    pasteBtn.addEventListener('click', function(){
      var el = getSelectedElement();
      if (!el) { toast(I18N.toastSelectFirst); return; }
      var cond = null;
      try { cond = JSON.parse(localStorage.getItem(COPY_STORAGE_KEY) || 'null'); } catch(e) {}
      if (!cond || cond.key !== GEO_KEY) { toast(I18N.toastNothing); return; }
      var ok = writeGeoConditionToElement(el, cond, { replaceExisting: true });
      if (ok) toast(I18N.toastPasted);
    });

    bar.appendChild(copyBtn);
    bar.appendChild(pasteBtn);
    document.body.appendChild(bar);
  }

  // -------------- Inline badges in structure panel -----------------
  function hasGeoCondition(el) {
    return !!readGeoConditionFromElement(el);
  }

  function formatBadge(el) {
    var c = readGeoConditionFromElement(el);
    if (!c) return '';

    var op   = c.compare === 'not_in' ? I18N.opNotIn : I18N.opIn;
    var list = (c.value || []).slice(0, 4).join(', ');
    var more = (c.value && c.value.length > 4) ? (' +' + (c.value.length - 4)) : '';

    return I18N.badgePrefix + ' ' + op + ' ' + list + more;
  }

  function findStructureRows() {
    // Try common selectors for Bricks structure tree across versions
    var roots = $all(
      '.bricks-structure, .structure, #structure, .builder-structure, .builder-sidebar, .brx-structure'
    );

    var rows = [];
    roots.forEach(function(root){
      $all('[data-id], [data-element-id], [data-uid], .structure-item, .brx-structure-item', root)
        .forEach(function(node){ rows.push(node); });
    });

    return rows;
  }

  function getRowElementId(row) {
    return row.getAttribute('data-id') ||
           row.getAttribute('data-element-id') ||
           row.getAttribute('data-uid') ||
           null;
  }

  function appendBadgeToRow(row, text) {
    var badge = row.querySelector('.bgeo-badge');
    if (!text) {
      if (badge) badge.style.display = 'none';
      return;
    }

    if (!badge) {
      badge = document.createElement('span');
      badge.className = 'bgeo-badge';
      // Try common label containers in the tree row
      var label =
        row.querySelector('.label, .bricks-label, .name, .title, .structure-item__title, .brx-structure-item__title') ||
        row;
      label.appendChild(badge);
    }

    badge.textContent = text;
    badge.style.display = '';
  }

  function updateBadges() {
    var rows = findStructureRows();
    rows.forEach(function(row){
      var id = getRowElementId(row);
      if (!id) return;

      var el = getElementById(id);
      if (!el) return;

      var txt = hasGeoCondition(el) ? formatBadge(el) : '';
      appendBadgeToRow(row, txt);
    });
  }

  function observeBuilder() {
    var root = document.body;
    var mo = new MutationObserver(function(){
      updateBadges();
      ensureToolbar();
    });
    mo.observe(root, { childList: true, subtree: true });
  }

  function init() {
    ensureToolbar();
    updateBadges();
    observeBuilder();
    log('Builder UI (badges + copy/paste) loaded.');
  }

  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }
})();