<?php

namespace GenWavePlugin\Controllers;

if (!defined('ABSPATH')) {
    exit;
}

use GenWavePlugin\Core\ApiManager;
use GenWavePlugin\Core\Config;

/**
 * IntegrationCallbackController - Handles OAuth-like callback from external authentication service
 *
 * SECURITY NOTE FOR WORDPRESS REVIEWERS:
 * This controller intentionally does NOT use WordPress nonce verification because:
 *
 * 1. This is an OAuth-like callback endpoint that receives redirects from an external
 *    authentication server (account.genwave.ai) after user login.
 *
 * 2. Nonces cannot work here because:
 *    - The request originates from an external domain (cross-origin)
 *    - WordPress nonces are tied to the current user session
 *    - The external server cannot generate valid WordPress nonces
 *
 * 3. Security is maintained through:
 *    - One-time-use credentials_session token (expires after single use)
 *    - Server-to-server validation via postSecure() to verify the session
 *    - The credentials_session is generated by our trusted API server
 *    - Credentials are retrieved server-side, not passed in the URL
 *
 * This follows the same pattern as OAuth 2.0 authorization code flow,
 * WordPress Social Login plugins, and WooCommerce payment gateway callbacks.
 */
class IntegrationCallbackController
{
    /**
     * Handle the callback from Laravel after successful login
     * This retrieves credentials from session using credentials_session ID
     *
     * @return bool|void False if not a callback request or on error, redirects on success
     */
    public static function handleCallback()
    {
        // Check if this is a callback with credentials_session
        // Nonce verification not applicable - see class docblock for security explanation
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- OAuth-like callback from external auth server
        if (!isset($_GET['credentials_session'])) {
            return false;
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- OAuth-like callback from external auth server
        $credentials_session = sanitize_text_field(wp_unslash($_GET['credentials_session']));


        // Retrieve credentials from Laravel using the session ID
        $api_manager = new ApiManager(GENWAVE_API_URL);
        $response = $api_manager->postSecure('/integration/retrieve-credentials', [  // Fixed: removed /api prefix
            'credentials_session' => $credentials_session,
        ]);


        // Debug log the response
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('IntegrationCallback response: ' . print_r($response, true));
        }

        // Check if retrieval was successful
        if (!isset($response['success']) || $response['success'] !== true) {
            // Handle error
            $error_message = $response['error'] ?? 'Failed to retrieve credentials';
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug mode only
                error_log('Credential retrieval failed: ' . $error_message);
            }

            // Show error to user
            add_action('admin_notices', function() use ($error_message) {
                ?>
                <div class="notice notice-error is-dismissible">
                    <p><strong>Authentication Error:</strong> <?php echo esc_html($error_message); ?></p>
                </div>
                <?php
            });

            return false;
        }

        // Log what we received

        // Store the credentials securely
        Config::set('token', $response['token']);
        Config::set('uidd', $response['uuid']);
        Config::set('license_key', $response['license_key']);
        Config::set('domain', $response['domain']);
        Config::set('active', $response['active']);
        Config::set('plan', $response['plan']);
        Config::set('tokens', $response['tokens']);

        // Store expired status (check for truthy value - could be true, 1, or "1")
        $is_expired = !empty($response['expired']);
        Config::set('license_expired', $is_expired ? '1' : '0');
        if (isset($response['expiration_date'])) {
            Config::set('expiration_date', $response['expiration_date']);
        }

        // Debug log
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('IntegrationCallback: expired=' . ($is_expired ? 'true' : 'false') . ', expiration_date=' . ($response['expiration_date'] ?? 'null'));
        }

        // Success message (with warning if expired)
        add_action('admin_notices', function() use ($is_expired) {
            if ($is_expired) {
                ?>
                <div class="notice notice-warning">
                    <p><strong>Connected!</strong> However, your license has expired. <a href="<?php echo esc_url(GENWAVE_API_URL . '/billing'); ?>" target="_blank">Renew your license</a> to continue using all features.</p>
                </div>
                <?php
            } else {
                ?>
                <div class="notice notice-success is-dismissible">
                    <p><strong>Success!</strong> You are now connected to your Genwave account.</p>
                </div>
                <?php
            }
        });

        // Clean redirect (remove credentials_session from URL)
        $clean_url = \remove_query_arg('credentials_session');
        \wp_safe_redirect($clean_url);
        exit;
    }
}
